// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI64/16AI64_Linux_3.x.x.x_DN/utils/reg.c $
// $Rev: 55142 $
// $Date: 2024-08-30 10:26:38 -0500 (Fri, 30 Aug 2024) $

// 16AI64: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	_GSC_REG(a)				"GSC " #a, AI64_GSC_##a, 0, 0
#define	SHOW_FIELD(b,e,eol,l,d)	gsc_reg_field_show(width+10,21, value,(b),(e),(eol),(l),(d))



//*****************************************************************************
static int _bcr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	aim[]	=
	{
		"Differential",
		"Single Ended",
		"Zero Test",
		"+Vref Test",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved"
	};

	static const char*	range[]	=
	{
		"+-2.5 Volts",
		"+-5 Volts",
		"+-10 Volts",
		"+-10 Volts"
	};

	static const char*	disable[]	= { "Disable",			"Enable"		};
	static const char*	fail[]		= { "Fail",				"Pass"			};
	static const char*	idle[]		= { "Idle",				"Active"		};
	static const char*	twos[]		= { "Twos Compliment",	"Offset Binary"	};

	SHOW_FIELD( 2,  0, 1, aim,		"Input Mode"		);
	SHOW_FIELD( 3,  3, 1, NULL,		"Reserved"			);
	SHOW_FIELD( 5,  4, 1, range,	"Range"				);
	SHOW_FIELD( 6,  6, 1, twos,		"Data Format"		);
	SHOW_FIELD( 7,  7, 1, disable,	"External Sync"		);
	SHOW_FIELD(11,  8, 1, NULL,		"Reserved"			);
	SHOW_FIELD(12, 12, 1, idle,		"Input Sync"		);
	SHOW_FIELD(13, 13, 1, idle,		"Autocal"			);
	SHOW_FIELD(14, 14, 1, fail,		"Autocal Status"	);
	SHOW_FIELD(15, 15, 1, idle,		"Initialize"		);
	SHOW_FIELD(31, 16, 1, NULL,		"Reserved"			);
	return(0);
}



//*****************************************************************************
static int _icr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	irq0[]	=
	{
		"Initialization Complete",
		"Autocal Completed",
		"Input Sample Initiated (Sync)",
		"Input Sample Completed (data ready)",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved"
	};

	static const char*	irq1[]	=
	{
		"Idle/None",
		"Input Buffer Threshold Low->High",
		"Input Buffer Threshold High->Low",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved"
	};

	static const char*	negated[]	= { "Negated",	"Asserted"	};

	int	errs;
	s32	ll;

	errs	= ai64_query(fd, -1, 0, AI64_QUERY_LOW_LATENCY, &ll);

	SHOW_FIELD( 2,  0, 1, irq0,		"IRQ0 Select"	);
	SHOW_FIELD( 3,  3, 1, negated,	"IRQ0 Request"	);

	if (ll)
	{
		SHOW_FIELD(31,  4, 1, NULL,		"Reserved"		);
	}
	else
	{
		SHOW_FIELD( 6,  4, 1, irq1,		"IRQ1 Select"	);
		SHOW_FIELD( 7,  7, 1, negated,	"IRQ1 Request"	);
		SHOW_FIELD(31,  8, 1, NULL,		"Reserved"		);
	}

	return(errs);
}



//*****************************************************************************
static int _idbr_decode(int fd, int supported, u32 value, int width)
{
	int	errs;
	s32	reg;
	int	ret	= 0;

	if (supported)
	{
		errs	= ai64_query(fd, -1, 0, AI64_QUERY_REG_IDBR, &reg);
		ret		= errs ? 0 : (reg ? 1 : 0);
	}
	else
	{
		SHOW_FIELD( 15,  0, 1, NULL,	"Data"		);
		SHOW_FIELD( 31, 16, 1, NULL,	"Reserved"	);
	}

	return(ret);
}



//*****************************************************************************
static int _ibcr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	clear[]		= { "Clear",	"Set"		};
	static const char*	idle[]		= { "Idle",		"Active"	};

	int	errs;
	s32	reg;
	int	ret	= 0;

	if (supported)
	{
		errs	= ai64_query(fd, -1, 0, AI64_QUERY_REG_IBCR, &reg);
		ret		= errs ? 0 : (reg ? 1 : 0);
	}
	else
	{
		SHOW_FIELD( 15,  0, 1, NULL,	"Threshold"			);
		SHOW_FIELD( 16, 16, 1, idle,	"Clear Buffer"		);
		SHOW_FIELD( 17, 17, 1, clear,	"Threshold Flag"	);
		SHOW_FIELD( 31, 18, 1, NULL,	"Reserved"			);
	}

	return(ret);
}



//*****************************************************************************
static int _rgr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	enabled[]	= { "Enabled",	"Disabled"	};

	SHOW_FIELD( 15,  0, 1, NULL,	"Nrate"		);
	SHOW_FIELD( 16, 16, 1, enabled,	"Disable"	);
	SHOW_FIELD( 31, 17, 1, NULL,	"Reserved"	);
	return(0);
}



//*****************************************************************************
static int _ragr_decode(int fd, int supported, u32 value, int width)
{
	int	errs;
	s32	reg;
	int	ret;

	if (supported)
	{
		errs	= ai64_query(fd, -1, 0, AI64_QUERY_REG_RAGR, &reg);
		ret		= errs ? 0 : (reg ? 1 : 0);
	}
	else
	{
		ret	= _rgr_decode(fd, supported, value, width);
	}

	return(ret);
}



//*****************************************************************************
static int _rbgr_decode(int fd, int supported, u32 value, int width)
{
	int	errs;
	s32	reg;
	int	ret;

	if (supported)
	{
		errs	= ai64_query(fd, -1, 0, AI64_QUERY_REG_RBGR, &reg);
		ret		= errs ? 0 : (reg ? 1 : 0);
	}
	else
	{
		ret	= _rgr_decode(fd, supported, value, width);
	}

	return(ret);
}



//*****************************************************************************
static int _bsr_decode(int fd, int supported, u32 value, int width)
{
	int	errs;
	s32	reg;
	int	ret	= 0;

	if (supported)
	{
		errs	= ai64_query(fd, -1, 0, AI64_QUERY_REG_BSR, &reg);
		ret		= errs ? 0 : (reg ? 1 : 0);
	}
	else
	{
		SHOW_FIELD( 15,  0, 1, NULL,	"Filled Locations"	);
		SHOW_FIELD( 31, 16, 1, NULL,	"Reserved"			);
	}

	return(ret);
}



//*****************************************************************************
static int _ccr_decode(int fd, int supported, u32 value, int width)
{
	int	errs;
	s32	reg;
	int	ret	= 0;

	if (supported)
	{
		errs	= ai64_query(fd, -1, 0, AI64_QUERY_REG_CCR, &reg);
		ret		= errs ? 0 : (reg ? 1 : 0);
	}
	else
	{
		SHOW_FIELD( 19,  0, 1, NULL,	"Conversions"	);
		SHOW_FIELD( 31, 20, 1, NULL,	"Reserved"		);
	}

	return(ret);
}



//*****************************************************************************
static int _sscr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	clk_src[]	=
	{
		"Rate-A Generator",
		"Rate-B Generator",
		"External Sync Input",
		"Software (BCR D12)"
	};

	static const char*	scan_size[]	=
	{
		"Single Channel Mode",
		"2 Channels (0-1)",
		"4 Channels (0-3)",
		"8 Channels (0-7)",
		"16 Channels (0-15)",
		"32 Channels (0-31)",
		"64 Channels (0-63)",
		"Reserved"
	};

	static const char*	b_src[]		= { "Master Clock",	"Rate-A Generator"	};
	static const char*	disable[]	= { "Disabled",		"Enabled"			};
	static const char*	idle[]		= { "Idle",			"Asserted"			};

	int	errs;
	s32	ll;

	errs	= ai64_query(fd, -1, 0, AI64_QUERY_LOW_LATENCY, &ll);


	if (ll)
	{
		SHOW_FIELD( 2,  0, 1, scan_size,	"Scan Size"			);
		SHOW_FIELD( 4,  3, 1, NULL,			"Reserved"			);
		SHOW_FIELD( 5,  5, 1, disable,		"Scan Enable"		);
		SHOW_FIELD(11,  6, 1, NULL,			"Reserved"			);
		SHOW_FIELD(17, 12, 1, NULL,			"Single Channel"	);
		SHOW_FIELD(18, 18, 1, idle,			"Reset Inputs"		);
		SHOW_FIELD(31, 19, 1, NULL,			"Reserved"			);
	}
	else
	{
		SHOW_FIELD( 2,  0, 1, scan_size,	"Scan Size"			);
		SHOW_FIELD( 4,  3, 1, clk_src,		"AI Clock Source"	);
		SHOW_FIELD( 9,  5, 1, NULL,			"Reserved"			);
		SHOW_FIELD(10, 10, 1, b_src,		"Rate-B Gen. Source");
		SHOW_FIELD(11, 11, 1, NULL,			"Reserved"			);
		SHOW_FIELD(17, 12, 1, NULL,			"Single Channel"	);
		SHOW_FIELD(31, 18, 1, NULL,			"Reserved"			);
	}

	return(errs);
}



//*****************************************************************************
static int _frr_decode(int fd, int supported, u32 value, int width)
{
	SHOW_FIELD(15,  0, 1, NULL,	"Firmware Revision"	);
	SHOW_FIELD(31, 16, 1, NULL,	"Reserved"			);
	return(0);
}



// variables ******************************************************************

static gsc_reg_def_t	_gsc[]	=
{
	{ _GSC_REG(BCR),	0,	_bcr_decode,	"Board Control Register"		},
	{ _GSC_REG(ICR),	0,	_icr_decode,	"Interrupt Control Register"	},
	{ _GSC_REG(IDBR),	1,	_idbr_decode,	"Input Data Buffer Register"	},
	{ _GSC_REG(IBCR),	1,	_ibcr_decode,	"Input Buffer Control Register"	},
	{ _GSC_REG(RAGR),	1,	_ragr_decode,	"Rate-A Generator Register"		},
	{ _GSC_REG(RBGR),	1,	_rbgr_decode,	"Rate-B Generator Register"		},
	{ _GSC_REG(BSR),	1,	_bsr_decode,	"Buffer Size Register"			},
	{ _GSC_REG(CCR),	1,	_ccr_decode,	"Conversion Counter Register"	},
	{ _GSC_REG(SSCR),	0,	_sscr_decode,	"Scan & Sync Control Register"	},
	{ _GSC_REG(FRR),	0,	_frr_decode,	"Firmware Revision Register"	},
	{ NULL, 0, 0, 0, 0,	NULL	}
};



//*****************************************************************************
static const gsc_reg_def_t* _find_reg(u32 reg, const gsc_reg_def_t* list)
{
	const gsc_reg_def_t*	def	= NULL;
	int						i;

	for (i = 0; list[i].name; i++)
	{
		if (reg == list[i].reg)
		{
			def	= &list[i];
			break;
		}
	}

	return(def);
}



/******************************************************************************
*
*	Function:	ai64_reg_get_def_id
*
*	Purpose:
*
*		Retrieve the register definition structure given the register id.
*
*	Arguments:
*
*		reg		The id of the register to access.
*
*	Returned:
*
*		NULL	The register id wasn't found.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* ai64_reg_get_def_id(u32 reg)
{
	const gsc_reg_def_t*	def;

	def	= _find_reg(reg, _gsc);

	return(def);
}



/******************************************************************************
*
*	Function:	ai64_reg_get_def_index
*
*	Purpose:
*
*		Retrieve the register definition structure based on an index.
*
*	Arguments:
*
*		index	The index of the register to access.
*
*	Returned:
*
*		NULL	The index doesn't correspond to a known register.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* ai64_reg_get_def_index(int index)
{
	const gsc_reg_def_t*	def	= NULL;

	if ((index >= 0) && (index < (SIZEOF_ARRAY(_gsc) - 1)))
		def	= &_gsc[index];

	return(def);
}



/******************************************************************************
*
*	Function:	ai64_reg_get_desc
*
*	Purpose:
*
*		Retrieve the description of the specified register.
*
*	Arguments:
*
*		reg		The register whose description is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* ai64_reg_get_desc(u32 reg)
{
	const gsc_reg_def_t*	def;
	const char*				desc;

	def	= _find_reg(reg, _gsc);

	if (def)
		desc	= def->desc;
	else
		desc	= "UNKNOWN";

	return(desc);
}



/******************************************************************************
*
*	Function:	ai64_reg_get_name
*
*	Purpose:
*
*		Retrieve the name of the specified register.
*
*	Arguments:
*
*		reg		The register whose name is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* ai64_reg_get_name(u32 reg)
{
	const gsc_reg_def_t*	def;
	const char*				name;

	def	= _find_reg(reg, _gsc);

	if (def)
		name	= def->name;
	else
		name	= "UNKNOWN";

	return(name);
}



/******************************************************************************
*
*	Function:	ai64_reg_list
*
*	Purpose:
*
*		List the GSC registers and their values.
*
*	Arguments:
*
*		fd		The handle to access the device.
*
*		detail	List the register details?
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int ai64_reg_list(int fd, int detail)
{
	int	errs;

	errs	= gsc_reg_list(fd, _gsc, detail, ai64_reg_read);
	return(errs);
}



