// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI64/16AI64_Linux_3.x.x.x_DN/utils/query.c $
// $Rev: 55142 $
// $Date: 2024-08-30 10:26:38 -0500 (Fri, 30 Aug 2024) $

// 16AI64: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	ai64_query
*
*	Purpose:
*
*		Provide a visual wrapper for the AI64_IOCTL_QUERY service.
*
*	Arguments:
*
*		fd		Use this handle to access the device.
*
*		index	The index of the device to access. Ignore if < 0.
*				This is for display purposes only.
*
*		verbose	Work verbosely? 0 = no, !0 = yes
*
*		set		This is the query option to access.
*
*		get		The results are reported here. This may be NULL.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int ai64_query(int fd, int index, int verbose, s32 set, s32* get)
{
	char		buf[128];
	int			errs	= 0;
	const char*	ptr;
	s32			query	= set;
	int			ret;
	s32			tmp;

	switch (query)
	{
		default:

			errs++;
			ptr	= buf;
			sprintf(buf, "Query Error (#%ld)", (long) query);
			break;

		case AI64_QUERY_AUTOCAL_MS:		ptr	= "Autocal Period";	break;
		case AI64_QUERY_CHANNEL_MAX:	ptr	= "Max Channels";		break;
		case AI64_QUERY_CHANNEL_QTY:	ptr	= "Input Channels";		break;
		case AI64_QUERY_COUNT:			ptr	= "Query Options";		break;
		case AI64_QUERY_DEVICE_TYPE:	ptr	= "Device Type";		break;
		case AI64_QUERY_FIFO_SIZE:		ptr	= "FIFO Size";			break;
		case AI64_QUERY_FSAMP_MAX:		ptr	= "Fsamp Max";			break;
		case AI64_QUERY_FSAMP_MIN:		ptr	= "Fsamp Min";			break;
		case AI64_QUERY_INIT_MS:		ptr	= "Initialize Period";	break;
		case AI64_QUERY_IRQ1:			ptr	= "IRQ1 Supported";		break;
		case AI64_QUERY_LOW_LATENCY:	ptr = "Low Latency";		break;
		case AI64_QUERY_MASTER_CLOCK:	ptr	= "Master Clock";		break;
		case AI64_QUERY_NRATE_MAX:		ptr = "Nrate Max";			break;
		case AI64_QUERY_NRATE_MIN:		ptr = "Nrate Min";			break;
		case AI64_QUERY_RATE_GEN_QTY:	ptr = "Rate Generators";	break;
		case AI64_QUERY_REG_BSR:		ptr = "Register: BSR";		break;
		case AI64_QUERY_REG_CCR:		ptr = "Register: CCR";		break;
		case AI64_QUERY_REG_IBCR:		ptr = "Register: IBCR";		break;
		case AI64_QUERY_REG_IDBR:		ptr = "Register: IDBR";		break;
		case AI64_QUERY_REG_RAGR:		ptr = "Register: RAGR";		break;
		case AI64_QUERY_REG_RBGR:		ptr = "Register: RBGR";		break;
	}

	if (verbose)
		gsc_label_index(ptr, index);

	ret		= ai64_ioctl(fd, AI64_IOCTL_QUERY, &set);
	errs	+= ret ? 1 : 0;

	switch (query)
	{
		default:

			errs++;
			sprintf(buf, "%ld", (long) set);
			break;

		case AI64_QUERY_CHANNEL_MAX:
		case AI64_QUERY_CHANNEL_QTY:
		case AI64_QUERY_COUNT:
		case AI64_QUERY_NRATE_MAX:
		case AI64_QUERY_NRATE_MIN:
		case AI64_QUERY_RATE_GEN_QTY:

			sprintf(buf, "%ld", (long) set);
			break;

		case AI64_QUERY_MASTER_CLOCK:

			sprintf(buf, "%ld Hz", (long) set);
			break;

		case AI64_QUERY_AUTOCAL_MS:
		case AI64_QUERY_INIT_MS:

			sprintf(buf, "%ld ms", (long) set);
			break;

		case AI64_QUERY_FSAMP_MAX:
		case AI64_QUERY_FSAMP_MIN:

			sprintf(buf, "%ld S/S", (long) set);
			break;

		case AI64_QUERY_LOW_LATENCY:
		case AI64_QUERY_REG_BSR:
		case AI64_QUERY_REG_CCR:
		case AI64_QUERY_REG_IBCR:
		case AI64_QUERY_REG_IDBR:
		case AI64_QUERY_REG_RAGR:
		case AI64_QUERY_REG_RBGR:
		case AI64_QUERY_IRQ1:

			sprintf(buf, "%s", set ? "Supported" : "Not Supported");
			break;

		case AI64_QUERY_FIFO_SIZE:

			ptr	= "";
			tmp	= set;

			if ((tmp) && ((tmp % 1024) == 0))
			{
				ptr	= "K";
				tmp	/= 1024;
			}

			sprintf(buf, "%ld%s Words", (long) tmp, ptr);
			break;

		case AI64_QUERY_DEVICE_TYPE:

			if (set == GSC_DEV_TYPE_16AI64)
			{
				strcpy(buf, "16AI64");
			}
			else
			{
				errs++;
				sprintf(buf, "INVALID: %ld", (long) set);
			}

			break;
	}

	if (verbose)
	{
		if (errs)
			printf("FAIL <---  (%s)\n", buf);
		else
			printf("%s\n", buf);
	}

	if (get)
		get[0]	= set;

	return(errs);
}


