// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI64/16AI64_Linux_3.x.x.x_DN/utils/fsamp_ai.c $
// $Rev: 55142 $
// $Date: 2024-08-30 10:26:38 -0500 (Fri, 30 Aug 2024) $

// 16AI64: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	ABS(v)					(((v) < 0) ? (-(v)) : (v))
#define	LIMIT_MAX(v,max)		(((v) > (max)) ? (max) : (v))
#define	LIMIT_MIN(v,min)		(((v) < (min)) ? (min) : (v))
#define	LIMIT_RANGE(v,min,max)	(LIMIT_MIN((LIMIT_MAX((v),(max))),(min)))



// data types *****************************************************************

typedef struct
{
	s32		fsamp_want;
	double	fsamp_got;

	double	delta;
	s32		enable_a;
	s32		enable_b;
	s32		master;
	s32		fsamp_max;
	s32		fsamp_min;
	s32		nrate_a;
	s32		nrate_b;
	s32		nrate_max;
	s32		nrate_min;
	s32		src;
	s32		src_b;
} fsamp_t;



//*****************************************************************************
static int _channels(int fd, s32* first, s32* last)
{
	int	errs	= 0;
	s32	qty;
	s32	single;
	s32	size;

	first[0]	= -1;
	last[0]		= -1;

	errs	+= ai64_query		(fd, -1, 0, AI64_QUERY_CHANNEL_QTY, &qty);
	errs	+= ai64_ai_scan_size(fd, -1, 0, -1, &size);
	errs	+= ai64_chan_single	(fd, -1, 0, -1, &single);

	switch (size)
	{
		default:

			errs++;
			break;

		case AI64_AI_SCAN_SIZE_0_1:		first[0]	= 0;		last[0]	= 1;		break;
		case AI64_AI_SCAN_SIZE_0_3:		first[0]	= 0;		last[0]	= 3;		break;
		case AI64_AI_SCAN_SIZE_0_7:		first[0]	= 0;		last[0]	= 7;		break;
		case AI64_AI_SCAN_SIZE_0_15:	first[0]	= 0;		last[0]	= 15;		break;
		case AI64_AI_SCAN_SIZE_0_31:	first[0]	= 0;		last[0]	= 31;		break;
		case AI64_AI_SCAN_SIZE_0_63:	first[0]	= 0;		last[0]	= 63;		break;
		case AI64_AI_SCAN_SIZE_SINGLE:	first[0]	= single;	last[0]	= single;	break;
	}

	return(errs);
}



//*****************************************************************************
static int _rag_disabled(int verbose, s32* sps)
{
	sps[0]	= 0;

	if (verbose)
	{
		gsc_label_long_comma((long) sps[0]);
		printf(" S/S  (Rate-A Generatoer is disabled.)\n");
	}

	return(0);
}



//*****************************************************************************
static int _rag_nrate_invalid(int verbose, s32 nrate, s32* sps)
{
	sps[0]	= 0;

	if (verbose)
	{
		gsc_label_long_comma((long) sps[0]);
		printf(	" S/S  (Rate-A Generatoer Nrate is invalid: %ld)\n",
				(long) nrate);
	}

	return(0);
}



//*****************************************************************************
static int _rbg_disabled(int verbose, s32* sps)
{
	sps[0]	= 0;

	if (verbose)
	{
		gsc_label_long_comma((long) sps[0]);
		printf(" S/S  (Rate-B Generatoer is disabled.)\n");
	}

	return(0);
}



//*****************************************************************************
static int _rbg_nrate_invalid(int verbose, s32 nrate, s32* sps)
{
	sps[0]	= 0;

	if (verbose)
	{
		gsc_label_long_comma((long) sps[0]);
		printf(	" S/S  (Rate-B Generatoer Nrate is invalid: %ld)\n",
				(long) nrate);
	}

	return(0);
}



//*****************************************************************************
static int _ai_clk_src_rag(int fd, int verbose, s32 fref, s32* sps)
{
	s32	a_enable;
	s32	a_nrate;
	int	errs		= 0;

	errs	+= ai64_rag_enable	(fd, -1, 0, -1, &a_enable);
	errs	+= ai64_rag_nrate	(fd, -1, 0, -1, &a_nrate);

	for (;;)	// A convenience loop.
	{
		if (a_enable == AI64_GEN_ENABLE_NO)
		{
			errs	+= _rag_disabled(verbose, sps);
			break;
		}

		if (a_nrate <= 0)
		{
			errs	+= _rag_nrate_invalid(verbose, a_nrate, sps);
			break;
		}

		sps[0]	= fref / a_nrate;

		if (verbose)
		{
			gsc_label_long_comma((long) sps[0]);
			printf(	" S/S  (Fref %ld, A Nrate %ld)\n",
					(long) fref,
					(long) a_nrate);
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _ai_clk_src_rbg_mc(int fd, int verbose, s32 fref, s32* sps)
{
	s32	b_enable;
	s32	b_nrate;
	int	errs		= 0;

	errs	+= ai64_rbg_enable	(fd, -1, 0, -1, &b_enable);
	errs	+= ai64_rbg_nrate	(fd, -1, 0, -1, &b_nrate);

	for (;;)	// A convenience loop.
	{
		if (b_enable == AI64_GEN_ENABLE_NO)
		{
			errs	+= _rbg_disabled(verbose, sps);
			break;
		}

		if (b_nrate <= 0)
		{
			errs	+= _rbg_nrate_invalid(verbose, b_nrate, sps);
			break;
		}

		sps[0]	= fref / b_nrate;

		if (verbose)
		{
			gsc_label_long_comma((long) sps[0]);
			printf(	" S/S  (Fref %ld, B Nrate %ld)\n",
					(long) fref,
					(long) b_nrate);
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _ai_clk_src_rbg_rag(int fd, int verbose, s32 fref, s32* sps)
{
	s32	a_enable;
	s32	a_nrate;
	s32	b_enable;
	s32	b_nrate;
	int	errs		= 0;

	errs	+= ai64_rag_enable	(fd, -1, 0, -1, &a_enable);
	errs	+= ai64_rag_nrate	(fd, -1, 0, -1, &a_nrate);
	errs	+= ai64_rbg_enable	(fd, -1, 0, -1, &b_enable);
	errs	+= ai64_rbg_nrate	(fd, -1, 0, -1, &b_nrate);

	for (;;)	// A convenience loop.
	{
		if (a_enable == AI64_GEN_ENABLE_NO)
		{
			errs	+= _rag_disabled(verbose, sps);
			break;
		}

		if (b_enable == AI64_GEN_ENABLE_NO)
		{
			errs	+= _rbg_disabled(verbose, sps);
			break;
		}

		if (a_nrate <= 0)
		{
			errs	+= _rag_nrate_invalid(verbose, a_nrate, sps);
			break;
		}

		if (b_nrate <= 0)
		{
			errs	+= _rbg_nrate_invalid(verbose, b_nrate, sps);
			break;
		}

		sps[0]	= (s32) ((float) fref / a_nrate / b_nrate + 0.5);

		if (verbose)
		{
			gsc_label_long_comma((long) sps[0]);
			printf(	" S/S  (Fref %ld, A Nrate %ld, B Nrate %ld)\n",
					(long) fref,
					(long) a_nrate,
					(long) b_nrate);
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _ai_clk_src_rbg(int fd, int verbose, s32 fref, s32* sps)
{
	int	errs;
	s32	rbg_src;

	errs	= ai64_rbg_clk_src(fd, -1, 0, -1, &rbg_src);

	if (errs == 0)
	{
		switch (rbg_src)
		{
			default:

				if (errs == 0)
					printf("FAIL <---  (invalid RBG CLK SRC selection: %ld)\n", (long) rbg_src);

				sps[0]	= 0;
				errs	= 1;
				break;

			case AI64_RBG_CLK_SRC_MASTER:

				errs	= _ai_clk_src_rbg_mc(fd, verbose, fref, sps);
				break;

			case AI64_RBG_CLK_SRC_RAG:

				errs	= _ai_clk_src_rbg_rag(fd, verbose, fref, sps);
				break;
		}
	}

	return(errs);
}



//*****************************************************************************
static int _ai_clk_src_ext(int verbose, s32* sps)
{
	sps[0]	= 0;

	if (verbose)
	{
		gsc_label_long_comma((long) sps[0]);
		printf(" S/S  (Externally Clocked Sampling)\n");
	}

	return(0);
}



//*****************************************************************************
static int _ai_clk_src_bcr(int verbose, s32* sps)
{
	sps[0]	= 0;

	if (verbose)
	{
		gsc_label_long_comma((long) sps[0]);
		printf(" S/S  (Software Clocked Sampling)\n");
	}

	return(0);
}



//*****************************************************************************
static int _fsamp_ai_report(int fd, int verbose, char* buf, s32* sps)
{
	s32	clk_src;
	int	errs;
	s32	fref;
	s32	ll;

	for (;;)
	{
		errs	= ai64_query(fd, -1, 0, AI64_QUERY_MASTER_CLOCK, &fref);

		if (errs)
		{
			sprintf(buf, "FAIL <---  (Master Clock Query)");
			break;
		}

		errs	= ai64_ai_clk_src(fd, -1, 0, -1, &clk_src);

		if (errs)
		{
			sprintf(buf, "FAIL <---  (AI CLock Source Query)");
			break;
		}

		errs	= ai64_query(fd, -1, 0, AI64_QUERY_LOW_LATENCY, &ll);

		if (errs)
		{
			sprintf(buf, "FAIL <---  (Low Latency Query)");
			break;
		}

		if (ll)
		{
			sps[0]	= 0;
			printf("0 S/S  (Low Latency Board)\n");
			break;
		}

		switch (clk_src)
		{
			default:

				sps[0]	= 0;
				errs	= 1;
				sprintf(buf,
						"FAIL <---  (invalid AI CLK SRC selection: %ld)\n",
						(long) clk_src);
				break;

			case AI64_AI_CLK_SRC_RAG:

				errs	= _ai_clk_src_rag(fd, verbose, fref, sps);
				break;

			case AI64_AI_CLK_SRC_RBG:

				errs	= _ai_clk_src_rbg(fd, verbose, fref, sps);
				break;

			case AI64_AI_CLK_SRC_EXT:

				errs	= _ai_clk_src_ext(verbose, sps);
				break;

			case AI64_AI_CLK_SRC_BCR:

				errs	= _ai_clk_src_bcr(verbose, sps);
				break;
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static void _fsamp_compute_a(fsamp_t* data)
{
	float	delta;
	float	fsamp_got;
	float	fsamp_want	= data->fsamp_want;
	int		init		= 0;	// Have we performed the first run initialization?
	s32		master		= data->master;
	s32		nr;
	s32		nr_max;
	s32		nr_min;
	s32		nrate_a;
	s32		nrate_max	= data->nrate_max;
	s32		nrate_min	= data->nrate_min;

	data->src		= AI64_AI_CLK_SRC_RAG;
	data->src_b		= AI64_RBG_CLK_SRC_MASTER;
	data->enable_a	= AI64_GEN_ENABLE_YES;
	data->enable_b	= AI64_GEN_ENABLE_NO;

	nr		= master / fsamp_want;
	nr		= (nr < nrate_min) ? nrate_min
			: (nr > nrate_max) ? nrate_max : nr;

	nr_min	= nr - 5;
	nr_max	= nr + 5;
	nr_min	= (nr_min < nrate_min) ? nrate_min
			: (nr_min > nrate_max) ? nrate_max : nr_min;
	nr_max	= (nr_max < nrate_min) ? nrate_min
			: (nr_max > nrate_max) ? nrate_max : nr_max;

	for (nrate_a = nr_min; nrate_a <= nr_max; nrate_a++)
	{
		fsamp_got	= ((float) master) / nrate_a;
		delta		= fsamp_got - fsamp_want;

		if ((init == 0) || ((ABS(delta)) < (ABS(data->delta))))
		{
			init			= 1;
			data->delta		= delta;
			data->fsamp_got	= fsamp_got;
			data->nrate_a	= nrate_a;
		}
	}
}



//*****************************************************************************
static void _fsamp_compute_ab(fsamp_t* data)
{
	float	delta;
	float	fsamp_got;
	float	fsamp_want	= data->fsamp_want;
	int		init		= 0;	// Have we performed the first run initialization?
	s32		master		= data->master;
	s32		nr;
	s32		nr_max;
	s32		nr_min;
	s32		nrate_a;
	s32		nrate_b;
	s32		nrate_max	= data->nrate_max;
	s32		nrate_min	= data->nrate_min;

	data->src		= AI64_AI_CLK_SRC_RBG;
	data->src_b		= AI64_RBG_CLK_SRC_RAG;
	data->enable_a	= AI64_GEN_ENABLE_YES;
	data->enable_b	= AI64_GEN_ENABLE_YES;

	for (nrate_a = nrate_min; nrate_a <= nrate_max; nrate_a++)
	{
		nr		= master / fsamp_want / nrate_a;
		nr		= (nr < nrate_min) ? nrate_min
				: (nr > nrate_max) ? nrate_max : nr;

		nr_min	= nr - 5;
		nr_max	= nr + 5;
		nr_min	= (nr_min < nrate_min) ? nrate_min
				: (nr_min > nrate_max) ? nrate_max : nr_min;
		nr_max	= (nr_max < nrate_min) ? nrate_min
				: (nr_max > nrate_max) ? nrate_max : nr_max;

		for (nrate_b = nr_min; nrate_b <= nr_max; nrate_b++)
		{
			fsamp_got	= (float) master / ((float) nrate_b * nrate_a);
			delta		= fsamp_got - fsamp_want;

			if ((init == 0) || ((ABS(delta)) < (ABS(data->delta))))
			{
				init			= 1;
				data->delta		= delta;
				data->fsamp_got	= fsamp_got;
				data->nrate_a	= nrate_a;
				data->nrate_b	= nrate_b;
			}
		}
	}
}



/******************************************************************************
*
*	Function:	ai64_fsamp_ai_report
*
*	Purpose:
*
*		Determine and report the sample rate for the specified channel.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	0: errors only, !0: all output
*
*		chan	The index of the channel of interest.
*
*		sps		Store the sample rate here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int ai64_fsamp_ai_report(int fd, int index, int verbose, int chan, s32* sps)
{
	char	buf[128];
	int		errs;
	s32		first;
	s32		last;
	s32		mode;
	s32		qty;
	s32		rate;

	if (sps == NULL)
		sps	= &rate;

	sps[0]	= 0;

	if (verbose)
	{
		if (chan < 0)
			sprintf(buf, "Sample Rate");
		else
			sprintf(buf, "Channel %d", chan);

		gsc_label_index(buf, index);

		if (chan < 0)
			chan	= 0;
	}

	for (;;)
	{
		buf[0]	= 0;
		errs	= ai64_ai_mode(fd, -1, 0, -1, &mode);

		if (errs)
		{
			sprintf(buf, "FAIL <---  (AI Mode selection: %d)", (int) mode);
			break;
		}

		errs	= ai64_query(fd, -1, 0, AI64_QUERY_CHANNEL_QTY, &qty);

		if (errs)
		{
			sprintf(buf, "FAIL <---  (Channel Qty Query)");
			break;
		}

		if ((chan < 0) || (chan > (qty - 1)))
		{
			errs	= 1;
			sprintf(buf, "FAIL <---  (invalid channel index: %d)", chan);
			break;
		}

		errs	= _channels(fd, &first, &last);

		if (errs)
		{
			sprintf(buf, "FAIL <---  (Channel configuration selection)");
			break;
		}

		if ((chan < first) || (chan > last))
		{
			printf("SKIPPED  (disabled)");
			break;
		}

		errs	= _fsamp_ai_report(fd, verbose, buf, sps);
		break;
	}

	if ((verbose) && (buf[0]))
			printf("%s\n", buf);

	return(errs);
}



/******************************************************************************
*
*	Function:	ai64_fsamp_ai_report_all
*
*	Purpose:
*
*		Determine and report the sample rate for all active channels.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely? 0 = no, !0 = yes
*
*		sps		Store the sample rate here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int ai64_fsamp_ai_report_all(int fd, int index, int verbose, s32* sps)
{
	int	errs	= 0;
	s32	first	= 0;
	int	i;
	s32	last	= 0;
	s32	limit;
	s32	ll;
	s32	mode;
	s32	rate;
	s32	size;
	s32	total	= 0;

	if (verbose)
		gsc_label_index("Sample Rates", index);

	errs	+= _channels(fd, &first, &last);
	errs	+= ai64_query	(fd, -1, 0, AI64_QUERY_LOW_LATENCY, &ll);
	errs	+= ai64_ai_mode	(fd, -1, 0, -1, &mode);
	size	= last - first + 1;
	limit	= (size > 1) ? 350052L : 500000L;

	if (errs)
	{
	}
	else if (ll)
	{
		if (verbose)
			printf("0 S/S  (Low Latency Board)\n");
	}
	else
	{
		if (verbose)
		{
			printf("\n");
			gsc_label_level_inc();
		}

		for (i = first; i <= last; i++)
		{
			errs	+= ai64_fsamp_ai_report(fd, index, verbose, i, &rate);
			total	+= rate;
		}

		if (verbose)
		{
			gsc_label("Overall Rate");
			gsc_label_long_comma(total);
			printf(" S/S");

			if (total > limit)
				printf("  <--- Exceeds Device Capabilities");

			printf("\n");
			gsc_label_level_dec();
		}
	}

	if (sps)
		sps[0]	= total;

	return(errs);
}



//*****************************************************************************
int	ai64_fsamp_ai_compute(
	int		fd,
	int		index,
	int		verbose,
	s32		fsamp,
	s32*	src,
	s32*	src_b,
	s32*	nrate_a,
	s32*	nrate_b,
	s32*	enable_a,
	s32*	enable_b,
	double*	rate)
{
	fsamp_t	data;
	fsamp_t	data_a;
	fsamp_t	data_ab;
	int		errs		= 0;

	if (verbose)
	{
		gsc_label_index("AI Sample Rate", index);
		gsc_label_long_comma(fsamp);
		printf(" S/S");
	}

	memset(&data, 0, sizeof(data));
	errs	+= ai64_query(fd, -1, 0, AI64_QUERY_FSAMP_MAX, &data.fsamp_max);
	errs	+= ai64_query(fd, -1, 0, AI64_QUERY_FSAMP_MIN, &data.fsamp_min);
	errs	+= ai64_query(fd, -1, 0, AI64_QUERY_MASTER_CLOCK, &data.master);
	errs	+= ai64_query(fd, -1, 0, AI64_QUERY_NRATE_MAX, &data.nrate_max);
	errs	+= ai64_query(fd, -1, 0, AI64_QUERY_NRATE_MIN, &data.nrate_min);

	data.delta		= FLT_MAX;
	data.fsamp_want	= LIMIT_RANGE(fsamp, data.fsamp_min, data.fsamp_max);
	data.nrate_a	= data.nrate_max;
	data.nrate_b	= data.nrate_max;

	data_a	= data;
	data_ab	= data;

	_fsamp_compute_a(&data_a);
	_fsamp_compute_ab(&data_ab);

	if ((ABS(data_a.delta)) < (ABS(data_ab.delta)))
	{
		data	= data_a;

		if (verbose)
		{
			printf("%s  (", errs ? " FAIL <---" : "");
			printf("Fref ");
			gsc_label_long_comma((long) data.master);
			printf(", Nrate-A ");
			gsc_label_long_comma((long) data.nrate_a);
			printf(", Fsamp %.3f", data.fsamp_got);
			printf(")\n");
		}
	}
	else
	{
		data	= data_ab;

		if (verbose)
		{
			printf("%s  (", errs ?  " FAIL <---" : "");
			printf("Fref ");
			gsc_label_long_comma((long) data.master);
			printf(", Nrate-A ");
			gsc_label_long_comma((long) data.nrate_a);
			printf(", Nrate-B ");
			gsc_label_long_comma((long) data.nrate_b);
			printf(", Fsamp %.3f", data.fsamp_got);
			printf(")\n");
		}
	}

	if (src)
		src[0]		= data.src;

	if (src_b)
		src_b[0]	= data.src_b;

	if (nrate_a)
		nrate_a[0]	= data.nrate_a;

	if (nrate_b)
		nrate_b[0]	= data.nrate_b;

	if (enable_a)
		enable_a[0]	= data.enable_a;

	if (enable_b)
		enable_b[0]	= data.enable_b;

	if (rate)
		rate[0]		= data.fsamp_got;

	return(errs);
}


