// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI64/16AI64_Linux_3.x.x.x_DN/samples/savedata/perform.c $
// $Rev: 55151 $
// $Date: 2024-08-30 10:35:48 -0500 (Fri, 30 Aug 2024) $

// 16AI64: Sample Application: source file

#include "main.h"



// macros *********************************************************************

#define	_1M					(1024L * 1024L)	// same as 64 * 16K
#define	UNUSED_MASK			0xFFFF0000



// variables ******************************************************************

static	u32	_buffer[_1M];



//*****************************************************************************
static int _channels(const args_t* args, s32* chans)
{
	s32		active	= -1;
	char	buf[64];
	int		errs	= 0;
	int		ret;
	s32		single	= -1;

	gsc_label("Input Channels");

	ret		= ai64_ioctl(args->fd, AI64_IOCTL_AI_SCAN_SIZE,	&active	);
	errs	+= ret ? 1 : 0;
	ret		= ai64_ioctl(args->fd, AI64_IOCTL_CHAN_SINGLE,	&single	);
	errs	+= ret ? 1 : 0;

	if (errs == 0)
	{
		switch (active)
		{
			default:

				errs++;
				printf("FAIL <---  (invalid ACTIVE selection: %ld)\n", (long) active);
				break;

			case AI64_AI_SCAN_SIZE_SINGLE:

				chans[0]	= 1;
				sprintf(buf, "# %ld", (long) single);
				break;

			case AI64_AI_SCAN_SIZE_0_1:

				chans[0]	= 2;
				strcpy(buf, "#s 0-1");
				break;

			case AI64_AI_SCAN_SIZE_0_3:

				chans[0]	= 4;
				strcpy(buf, "#s 0-3");
				break;

			case AI64_AI_SCAN_SIZE_0_7:

				chans[0]	= 8;
				strcpy(buf, "#s 0-7");
				break;

			case AI64_AI_SCAN_SIZE_0_15:

				chans[0]	= 16;
				strcpy(buf, "#s 0-15");
				break;

			case AI64_AI_SCAN_SIZE_0_31:

				chans[0]	= 32;
				strcpy(buf, "#s 0-31");
				break;

			case AI64_AI_SCAN_SIZE_0_63:

				chans[0]	= 64;
				strcpy(buf, "#s 0-63");
				break;
		}

		if (errs == 0)
		{
			printf(	"%ld Channel%s  (%s)\n",
					(long) chans[0],
					(chans[0] == 1) ? "" : "s",
					buf);
		}
	}

	return(errs);
}



//*****************************************************************************
// Some boards may initially provide invalid data for a period. This function
// is called to consume the leading invalid data. Some board models may need
// this while others won't. Any type of required settling delay should be
// documented in the board user manual.
static int _data_preread(const args_t* args)
{
	int	errs	= 0;
	int	get		= 1024;	// bytes
	int	got;
	int	i;
	int	j;
	u32	last	= 0;
	int	tests	= 0;

	for (i = 1;; i++)
	{
		if (i >= 1000)
		{
			errs++;
			printf(" FAIL <--- (preread failed)\n");
			fflush(stdout);
			break;
		}

		memset(_buffer, 0, get);
		errs	= ai64_ai_buf_clear(args->fd, -1, 0);

		if (errs)
		{
			printf(" FAIL <--- (preread clear buffer failed)\n");
			fflush(stdout);
			break;
		}

		got	= ai64_read(args->fd, _buffer, get);

		if (got < 0)
		{
			errs++;
			printf(" FAIL <--- (preread returned %d)\n", got);
			fflush(stdout);
			break;
		}
		else if (got != get)
		{
			errs++;
			printf(" FAIL <--- (preread requested ");
			gsc_label_long_comma(get);
			printf(" bytes, received ");
			gsc_label_long_comma(got);
			printf(")\n");
			fflush(stdout);
			break;
		}

		// Find the first channel 0 entry.
		// Since this board doesn't impliment channel tags we're going to
		// assume that the first data element is the first channel.
		j	= 0;

		if (i == 1)
		{
			// This is the first sample from the first channel.
			last	= _buffer[j];
		}
		else
		{
			if (_buffer[j] != last)
			{
				tests++;

				if (tests >= 5)	// Repeat this before accepting the results.
					break;
			}

			last	= _buffer[j];
		}
	}

	return(errs);
}



//*****************************************************************************
static int _read_data(const args_t* args)
{
	char	buf[80];
	int		errs	= 0;
	int		get		= sizeof(_buffer);
	int		got;
	int		repeat;

	for (repeat = 0; (errs == 0) && (repeat < args->repeat);)
	{
		repeat++;
		strcpy(buf, "Reading");

		if (args->repeat > 1)
		{
			strcat(buf, " #");
			gsc_label_long_comma_buf(repeat, buf + strlen(buf));
		}

		gsc_label(buf);

		memset(_buffer, 0, sizeof(_buffer));
		errs	= _data_preread(args);

		if (errs)
			break;

		errs	= ai64_ai_buf_clear(args->fd, -1, 0);

		if (errs)
		{
			printf("FAIL <---  (setup errors)\n");
			break;
		}

		got	= ai64_read(args->fd, _buffer, get);

		if (got < 0)
		{
			errs++;
			printf("FAIL <---  (read error: %d)\n", got);
			break;
		}

		if (got != get)
		{
			errs	= 1;
			printf(	"FAIL <---  (got %ld bytes, requested %ld)\n",
					(long) got,
					(long) get);
			break;
		}

		printf("PASS  (");
		gsc_label_long_comma(get / 4);
		printf(" samples)\n");
	}

	return(errs);
}



//*****************************************************************************
static int _save_eol(const char* name, FILE* file)
{
	int	errs	= 0;
	int	i;

	i	= fprintf(file, "\r\n");

	if (i != 2)
	{
		printf("FAIL <---  (fprintf() failure to %s)\n", name);
		errs	= 1;
	}

	return(errs);
}



//*****************************************************************************
static int _save_args(const args_t* args, const char* name, FILE* file)
{
	int			arg;
	int			errs	= 0;
	int			i;
	int			len;
	const char*	space	= "# ";

	for (arg = 0; arg < args->argc; arg++)
	{
		// space ==========================================
		len	= strlen(space);
		i	= fprintf(file, "%s", space);

		if (i != len)
		{
			printf("FAIL <---  (fprintf() failure to %s)\n", name);
			errs++;
		}

		// argument =======================================
		space	= " ";
		len		= strlen(args->argv[arg]);
		i		= fprintf(file, "%s", args->argv[arg]);

		if (i != len)
		{
			printf("FAIL <---  (fprintf() failure to %s)\n", name);
			errs++;
		}
	}

	errs	+= _save_eol(name, file);
	return(errs);
}



//*****************************************************************************
static int _save_value(const args_t* args, double voltage, const char* name, FILE* file, u32 value)
{
	char		buf1[40];
	char		buf2[80];
	int			errs	= 0;
	const char*	format	= "";
	int			i;
	int			len;
	int			max;
	int			mid;
	int			vi;
	int			width	= 8;

	if (args->format == FORMAT_HEX)
	{
		sprintf(buf1, "%08lX", (long) value);
	}
	else if (args->format == FORMAT_DEC)
	{
		sprintf(buf1, "%ld", (long) value);
	}
	else if (args->format == FORMAT_DEC_0)
	{
		mid	= 0x8000;
		vi	= value - mid;
		sprintf(buf1, "%ld", (long) vi);
	}
	else if (args->format == FORMAT_VOLT)
	{
		max		= 0xFFFF;
		format	= "%10.6lf";
		voltage	= (double) value / max * voltage * 2 - voltage;
		sprintf(buf1, format, voltage);
	}
	else
	{
		sprintf(buf1, "FORMAT %d", (int) args->format);
	}

	sprintf(buf2, "  %*s", width, buf1);
	len	= strlen(buf2);
	i	= fprintf(file, "%8s", buf2);

	if (i != len)
	{
		printf("FAIL <---  (fprintf() failure to %s)\n", name);
		errs	= 1;
	}

	return(errs);
}



//*****************************************************************************
static int _validate_data(long index, u32 data)
{
	int	errs	= 0;

	if (data & UNUSED_MASK)
	{
		errs	= 1;
		printf(" FAIL <---  (Upper bits are invalid: [%ld] = 0x%08lX)\n", index, (long) data);
	}

	return(errs);
}



//*****************************************************************************
static int _save_data(const args_t* args, double voltage, int chans, int errs)
{
	char		buf[64];
	FILE*		file;
	long		index;
	const char*	name	= "data.txt";
	long		samples	= sizeof(_buffer) / 4;

	strcpy(buf, "Saving");

	if (args->force_save)
	{
		errs	= 0;
		strcat(buf, " (Forced)");
	}

	gsc_label(buf);

	for (;;)
	{
		if ((errs) && (args->force_save == 0))
		{
			printf("SKIPPED  (errors)\n");
			errs	= 0;
			break;
		}

		file	= fopen(name, "w+b");

		if (file == NULL)
		{
			printf("FAIL <---  (unable to create %s)\n", name);
			errs	= 1;
			break;
		}

		errs	+= _save_args(args, name, file);

		for (index = 0; index < samples; index++)
		{
			if ((index > 0) && ((index % chans) == 0))
				errs	+= _save_eol(name, file);

			errs	+= _save_value(args, voltage, name, file, _buffer[index]);
			errs	+= _validate_data(index, _buffer[index]);
		}

		fclose(file);

		if (errs == 0)
			printf("PASS  (%s)\n", name);

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _config(const args_t* args, s32* ll, double* voltage)
{
	int	errs	= 0;
	s32	range	= AI64_AI_RANGE_10V;

	gsc_label("Configure");
	printf("\n");
	gsc_label_level_inc();

	errs	+= ai64_config_ai	(args->fd, -1, 1, AI64_AI_SCAN_SIZE_0_31, args->fsamp);
	errs	+= ai64_rx_io_mode	(args->fd, -1, 1, args->io_mode, NULL);
	errs	+= ai64_ai_range	(args->fd, -1, 1, -1, &range);

	if (range == AI64_AI_RANGE_2_5V)
		voltage[0]	= 2.5;
	else if (range == AI64_AI_RANGE_5V)
		voltage[0]	= 5.0;
	else
		voltage[0]	= 10.0;

	errs	+= ai64_query(args->fd, -1, 1, AI64_QUERY_LOW_LATENCY, ll);

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
int perform_tests(const args_t* args)
{
	s32		chans	= 64;	// assume the maximum
	int		errs	= 0;
	s32		ll		= 0;	// Low Latency
	double	voltage	= 10.0;	// assume for now

	gsc_label("Capture & Save");
	printf("\n");
	gsc_label_level_inc();

	errs	+= _config(args, &ll, &voltage);

	if (ll)
	{
		gsc_label("Testing");
		printf("SKIPPED  (Low Latency Board)\n");
	}
	else
	{
		errs	+= _channels(args, &chans);
		errs	+= ai64_ai_buf_clear(args->fd, -1, 1);
		errs	+= _read_data(args);
		errs	+= _save_data(args, voltage, chans, errs);
	}

	gsc_label_level_dec();
	return(errs);
}



