// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI64/16AI64_Linux_3.x.x.x_DN/samples/savedata/main.c $
// $Rev: 55151 $
// $Date: 2024-08-30 10:35:48 -0500 (Fri, 30 Aug 2024) $

// 16AI64: Sample Application: source file

#include "main.h"



// macros *********************************************************************

#define	DIFF			AI64_AI_MODE_DIFF
#define	SINGLE			AI64_AI_MODE_SINGLE
#define	VREF			AI64_AI_MODE_VREF
#define	ZERO			AI64_AI_MODE_ZERO

#define	BMDMA			GSC_IO_MODE_BMDMA
#define	PIO				GSC_IO_MODE_PIO



//*****************************************************************************
static int _parse_args(int argc, char** argv, args_t* args)
{
	const gsc_arg_item_t	list[]	=
	{
		//	type				var						arg			values				desc
		{	GSC_ARG_S32_FLAG,	&args->io_mode,			"-bmdma",	{ BMDMA },			"Transfer data using Block Mode DMA."					},
		{	GSC_ARG_S32_FLAG,	&args->continuous,		"-c",		{ -1 },				"Continue testing until an error occurs."				},
		{	GSC_ARG_S32_FLAG,	&args->continuous,		"-C",		{ +1 },				"Continue testing even if errors occur."				},
		{	GSC_ARG_S32_FLAG,	&args->ai_mode,			"-diff",	{ DIFF },			"Sample data with the Differential input."				},
		{	GSC_ARG_S32_FLAG,	&args->format,			"-dec",		{ FORMAT_DEC },		"Save data as decimal numbers."							},
		{	GSC_ARG_S32_FLAG,	&args->format,			"-dec0",	{ FORMAT_DEC_0 },	"Center decimal data around zero instead of mid scale."	},
		{	GSC_ARG_S32_FLAG,	&args->force_save,		"-f",		{ 1 },				"Force data to be saved even if errors occur."			},
		{	GSC_ARG_S32_RANGE,	&args->fsamp,			"-fsamp",	{ 1, 500000 },		"Sample data at # S/S (default: 10,000)."				},
		{	GSC_ARG_S32_FLAG,	&args->format,			"-hex",		{ FORMAT_HEX },		"Save data as hex numbers (default)."					},
		{	GSC_ARG_S32_MIN,	&args->minute_limit,	"-m",		{ 1 },				"Limit continuous testing to # minutes."				},
		{	GSC_ARG_S32_MIN,	&args->test_limit,		"-n",		{ 1 },				"Limit continuous testing to # iterations."				},
		{	GSC_ARG_S32_FLAG,	&args->io_mode,			"-pio",		{ PIO },			"Transfer data using PIO (default)."					},
		{	GSC_ARG_S32_RANGE,	&args->repeat,			"-r",		{ 1, 10000 },		"Repeat data retreaval # times before saving the data."	},
		{	GSC_ARG_S32_FLAG,	&args->ai_mode,			"-single",	{ SINGLE },			"Sample data with the Single Ended inputs."				},
		{	GSC_ARG_S32_FLAG,	&args->format,			"-volt",	{ FORMAT_VOLT },	"Save data as floating point voltage levels."			},
		{	GSC_ARG_S32_FLAG,	&args->ai_mode,			"-vref",	{ VREF },			"Sample data from the onboard Vref source."				},
		{	GSC_ARG_S32_FLAG,	&args->ai_mode,			"-vzero",	{ ZERO },			"Sample data from the onboard zero voltage source."		},
		{	GSC_ARG_DEV_INDEX,	&args->index,			"index",	{ 1 },				"The zero based index of the device to access."			}
	};

	const gsc_arg_set_t	set	=
	{
		/* name	*/	"savedata",
		/* desc	*/	"Capture and save data to disk.",
		/* qty	*/	SIZEOF_ARRAY(list),
		/* list	*/	list
	};

	int	errs;

	memset(args, 0, sizeof(args_t));
	args->argc			= argc;
	args->argv			= argv;
	args->ai_mode		= -1;
	args->fsamp			= 10000;
	args->io_mode		= GSC_IO_MODE_PIO;
	args->repeat		= 1;

	errs	= gsc_args_parse(argc, argv, &set);

	return(errs);
}



//*****************************************************************************
static void _show_access_index(const args_t* args)
{
	gsc_label("Accessing Device");
	printf("%d\n", args->index);
}



//*****************************************************************************
static void _show_time_stamp(void)
{
	const char*	psz;
	struct tm*	stm;
	time_t		tt;

	time(&tt);
	stm	= localtime(&tt);
	psz	= (char*) asctime(stm);
	gsc_label("Time Stamp");
	printf("%s", psz);
}



/******************************************************************************
*
*	Function:	main
*
*	Purpose:
*
*		Control the overall flow of the application.
*
*	Arguments:
*
*		argc			The number of command line arguments.
*
*		argv			The list of command line arguments.
*
*	Returned:
*
*		EXIT_SUCCESS	No errors were encounterred.
*		EXIT_FAILURE	One or more problems were encounterred.
*
******************************************************************************/

int main(int argc, char** argv)
{
	args_t	args;
	int		errs;
	time_t	exec		= time(NULL);
	int		failures	= 0;
	long	hours;
	long	mins;
	time_t	now;
	int		passes		= 0;
	int		ret;
	long	secs;
	time_t	t_limit;
	time_t	test;
	int		tests		= 0;

	for (;;)	// A convenience loop.
	{
		gsc_label_init(30);
		test	= time(NULL);
		errs	= _parse_args(argc, argv, &args);

		// Introduction

		t_limit	= exec + (args.minute_limit * 60);
		os_id_host();
		_show_time_stamp();
		_show_access_index(&args);
		errs	+= ai64_init_util(1);
		errs	+= os_id_driver(ai64_open, ai64_read, ai64_close);
		errs	+= ai64_count_boards(1, &args.qty);

		if (args.qty <= 0)
			errs++;

		// Open access to device

		if (errs == 0)
			errs	= ai64_open_util(args.index, 0, -1, 1, &args.fd);

		if (errs)
			args.continuous	= 0;

		if (errs == 0)
		{
			// Device identification

			errs	= ai64_id_device(args.fd, -1, 1);

			// Perform testing

			errs	+= perform_tests(&args);

			// Close device access

			errs	+= ai64_close_util(args.fd, -1, 1);
		}

		// End of processing.

		now	= time(NULL);
		tests++;

		if (errs)
		{
			failures++;
			printf(	"\nRESULTS: FAIL <---  (%d error%s)",
					errs,
					(errs == 1) ? "" : "s");
		}
		else
		{
			passes++;
			printf("\nRESULTS: PASS");
		}

		secs	= (long) (now - test);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(" (duration %ld:%ld:%02ld)\n", hours, mins, secs);

		secs	= (long) (now - exec);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(	"SUMMARY: tests %d, pass %d, fail %d"
				" (duration %ld:%ld:%02ld)\n\n",
				tests,
				passes,
				failures,
				hours,
				mins,
				secs);

		if (args.continuous == 0)
			break;

		if ((args.continuous < 0) && (errs))
			break;

		if ((args.minute_limit > 0) && (now >= t_limit))
			break;

		if ((args.test_limit > 0) && (tests >= args.test_limit))
			break;
	}

	ret	= ((args.qty <= 0) || (errs) || (failures)) ? EXIT_FAILURE : EXIT_SUCCESS;
	return(ret);
}


