// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI64/16AI64_Linux_3.x.x.x_DN/driver/read.c $
// $Rev: 51842 $
// $Date: 2022-10-17 17:32:36 -0500 (Mon, 17 Oct 2022) $

// 16AI64: Device Driver: source file

#include "main.h"



//*****************************************************************************
static void _dev_io_sw_init(dev_data_t* dev, dev_io_t* io)
{
	io->io_mode				= AI64_IO_MODE_DEFAULT;
	io->pio_threshold		= 32;
	io->timeout_s			= AI64_IO_TIMEOUT_DEFAULT;
}



//*****************************************************************************
static void _dev_io_close(dev_data_t* dev, dev_io_t* io)
{
	io->dma_channel	= NULL;
}



//*****************************************************************************
static void _dev_io_open(dev_data_t* dev, dev_io_t* io)
{
	_dev_io_sw_init(dev, io);
}



//*****************************************************************************
static int _dev_io_startup(dev_data_t* dev, dev_io_t* io)
{
	int	ret;

	if (dev->cache.low_latency)
		ret	= -ENOSYS;
	else
		ret	= 0;

	return(ret);
}



//*****************************************************************************
static long _dev_pio_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	u32	bsr;

	bsr		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsr_32);
	bytes	= (bsr & 0xFFFF) * 4;
	return(bytes);
}



//*****************************************************************************
static long _dev_pio_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	long	qty;

	qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	return(qty);
}



//*****************************************************************************
static long _dev_bmdma_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	u32	avail;
	u32	bsr;
	u32	ibcr;
	u32	thresh;

	for (;;)
	{
		// Read from the current content, if there is enough data.
		bsr		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsr_32);
		avail	= (bsr & 0xFFFF) * 4;

		if (bytes <= avail)
		{
			// Read from the available data.
			break;
		}

		// Read from the current content, if the threshold has been met.
		ibcr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ibcr_32);
		thresh	= (ibcr & 0xFFFF) * 4;

		if (avail >= thresh)
		{
			// Read upto the threshold level.
			bytes	= thresh;
			break;
		}

		// Wait until additional data becomes available.
		bytes	= 0;
		break;
	}

	return(bytes);
}



//*****************************************************************************
static long _dev_bmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	long			qty;
	long			samples	= bytes / 4;
	gsc_dma_setup_t	setup;

	if (samples < io->pio_threshold)
	{
		qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	}
	else
	{
		memset(&setup, 0, sizeof(gsc_dma_setup_t));
		setup.alt		= dev;
		setup.dev		= dev;
		setup.io		= io;
		setup.mem		= mem;
		setup.st_end	= st_end;
		setup.bytes		= bytes;
		setup.ability	= GSC_DMA_CAP_BMDMA_READ;

		setup.mode		= GSC_DMA_MODE_BLOCK_DMA
						| GSC_DMA_MODE_SIZE_32_BITS
						| GSC_DMA_MODE_INPUT_ENABLE
						| GSC_DMA_MODE_BURSTING_LOCAL
						| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
						| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
						| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		setup.dpr		= GSC_DMA_DPR_BOARD_TO_HOST
						| GSC_DMA_DPR_END_OF_CHAIN
						| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty	= gsc_dma_perform(&setup);
	}

	return(qty);
}



//*****************************************************************************
static long _dev_dmdma_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	// DMDMA is not supported.
	return(-1);
}



//*****************************************************************************
static long _dev_dmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	// DMDMA is not supported.
	return(-1);
}



/******************************************************************************
*
*	Function:	dev_read_create
*
*	Purpose:
*
*		Perform a one-tine setup of the Read Analog Input streaming I/O structure.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*		io		The I/O structure for this stream.
*
*	Returned:
*
*		None.
*
******************************************************************************/

int dev_read_create(dev_data_t* dev, dev_io_t* io)
{
	int	ret;

	io->bytes_per_sample	= 4;
	io->io_reg_offset		= GSC_REG_OFFSET(AI64_GSC_IDBR);
	io->io_reg_vaddr		= dev->vaddr.gsc_idbr_32;

	io->dev_io_sw_init		= _dev_io_sw_init;
	io->dev_io_close		= _dev_io_close;
	io->dev_io_open			= _dev_io_open;
	io->dev_io_startup		= _dev_io_startup;
	io->dev_pio_available	= _dev_pio_available;
	io->dev_bmdma_available	= _dev_bmdma_available;
	io->dev_dmdma_available	= _dev_dmdma_available;
	io->dev_pio_xfer		= _dev_pio_xfer;
	io->dev_bmdma_xfer		= _dev_bmdma_xfer;
	io->dev_dmdma_xfer		= _dev_dmdma_xfer;

	io->wait.abort			= AI64_WAIT_IO_RX_ABORT;
	io->wait.done			= AI64_WAIT_IO_RX_DONE;
	io->wait.error			= AI64_WAIT_IO_RX_ERROR;
	io->wait.timeout		= AI64_WAIT_IO_RX_TIMEOUT;

	ret	= gsc_io_create(dev, io, 64L * 512 * 4);

	return(ret);
}



