// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI64/16AI64_Linux_3.x.x.x_DN/driver/ioctl.c $
// $Rev: 55155 $
// $Date: 2024-08-30 10:38:53 -0500 (Fri, 30 Aug 2024) $

// 16AI64: Device Driver: source file

#include "main.h"



// macros *********************************************************************

#define	AUTOCAL_START			D13		// BCR
#define	AUTOCAL_PASS			D14		// BCR
#define	INIT_START				D15		// BCR

#define	IRQ0_INIT_DONE			0x0000	// ICR
#define	IRQ0_AUTOCAL_DONE		0x0001	// ICR
#define	IRQ0_MASK				0x0007	// ICR
#define	IRQ0_ACTIVE				0x0008	// ICR
#define	IRQ1_ACTIVE				0x0080	// ICR



//*****************************************************************************
static int _query(dev_data_t* dev, s32* arg)
{
	switch (arg[0])
	{
		default:						arg[0]	= AI64_IOCTL_QUERY_ERROR;	break;
		case AI64_QUERY_AUTOCAL_MS:		arg[0]	= dev->cache.autocal_ms;	break;
		case AI64_QUERY_CHANNEL_MAX:	arg[0]	= dev->cache.channels_max;	break;
		case AI64_QUERY_CHANNEL_QTY:	arg[0]	= dev->cache.channel_qty;	break;
		case AI64_QUERY_COUNT:			arg[0]	= AI64_IOCTL_QUERY_LAST;	break;
		case AI64_QUERY_DEVICE_TYPE:	arg[0]	= dev->board_type;			break;
		case AI64_QUERY_FIFO_SIZE:		arg[0]	= dev->cache.fifo_size;		break;
		case AI64_QUERY_FSAMP_MAX:		arg[0]	= dev->cache.fsamp_max;		break;
		case AI64_QUERY_FSAMP_MIN:		arg[0]	= dev->cache.fsamp_min;		break;
		case AI64_QUERY_INIT_MS:		arg[0]	= dev->cache.initialize_ms;	break;
		case AI64_QUERY_IRQ1:			arg[0]	= dev->cache.irq1;			break;
		case AI64_QUERY_LOW_LATENCY:	arg[0]	= dev->cache.low_latency;	break;
		case AI64_QUERY_MASTER_CLOCK:	arg[0]	= dev->cache.master_clock;	break;
		case AI64_QUERY_NRATE_MAX:		arg[0]	= dev->cache.nrate_max;		break;
		case AI64_QUERY_NRATE_MIN:		arg[0]	= dev->cache.nrate_min;		break;
		case AI64_QUERY_RATE_GEN_QTY:	arg[0]	= dev->cache.rate_gen_qty;	break;
		case AI64_QUERY_REG_BSR:		arg[0]	= dev->cache.reg_bsr;		break;
		case AI64_QUERY_REG_CCR:		arg[0]	= dev->cache.reg_ccr;		break;
		case AI64_QUERY_REG_IBCR:		arg[0]	= dev->cache.reg_ibcr;		break;
		case AI64_QUERY_REG_IDBR:		arg[0]	= dev->cache.reg_idbr;		break;
		case AI64_QUERY_REG_RAGR:		arg[0]	= dev->cache.reg_ragr;		break;
		case AI64_QUERY_REG_RBGR:		arg[0]	= dev->cache.reg_rbgr;		break;
	}

	return(0);
}



//*****************************************************************************
static int _init_start(dev_data_t* dev, void* arg)
{
	// Initiate initialization.
	os_reg_mem_tx_u32(NULL, dev->vaddr.gsc_bcr_32, INIT_START);
	return(0);
}



//*****************************************************************************
int initialize_ioctl(dev_data_t* dev, void* arg)
{
	int			i;
	int			mask;
	long		ms			= dev->cache.initialize_ms + 5000;
	long		ms_total	= ms;
	u32			reg;
	int			ret			= 0;
	os_sem_t	sem;
	int			tmp;
	VADDR_T		va			= dev->vaddr.gsc_icr_32;
	int			value;
	gsc_wait_t	wt;

	if ((dev->irq.opened) && (gsc_global.driver_unloading == 0))
	{
		ms_total	*= 2;

		// Safely select the Initialize Done interrupt.
		mask	= IRQ0_MASK | IRQ0_ACTIVE | IRQ1_ACTIVE;
		value	= IRQ0_INIT_DONE;
		os_reg_mem_mx_u32(dev, va, value, mask);

		// Wait for the local interrupt.
		os_sem_create(&sem);	// dummy, required for wait operations.
		memset(&wt, 0, sizeof(wt));
		wt.flags		= GSC_WAIT_FLAG_INTERNAL;
		wt.gsc			= AI64_WAIT_GSC_INIT_DONE;
		wt.timeout_ms	= ms;
		ret				= gsc_wait_event(dev, &wt, _init_start, NULL, &sem);
		os_sem_destroy(&sem);

		if (wt.flags & GSC_WAIT_FLAG_TIMEOUT)
		{
			ret	= ret ? ret : -ETIMEDOUT;
			printf(	"%s: INITILIZE DONE IRQ TIMED OUT AFTER %ld ms.\n",
					dev->model,
					ms);
		}
	}
	else
	{
		_init_start(dev, NULL);
	}

	// Manually wait for completion as the IRQ wait may have ended early.
	va	= dev->vaddr.gsc_bcr_32;
	tmp	= gsc_poll_u32(dev, ms, va, INIT_START, 0);

	if (tmp)
	{
		ret	= ret ? ret : tmp;
		printf(	"%s: INITIALIZATION DID NOT COMPLETE WITHIN %ld ms.\n",
				dev->model,
				ms_total);
	}

	// Manually check for completion as the IRQ wait may have ended early.
	reg	= os_reg_mem_rx_u32(dev, va);

	if (reg & INIT_START)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: INITILIZE STILL ACTIVE AFTER %ld ms.\n",
				dev->model,
				ms_total);
	}

	// Initialize the software settings.

	for (i = 0; i < DEV_IO_STREAM_QTY; i++)
	{
		if (dev->io.io_streams[i])
		{
			if (dev->io.io_streams[i]->dev_io_sw_init)
			{
				(dev->io.io_streams[i]->dev_io_sw_init)(dev, dev->io.io_streams[i]);
			}
		}
	}

	return(ret);
}



//*****************************************************************************
static int _autocal_start(dev_data_t* dev, void* arg)
{
	// Initiate autocalibration.
	os_reg_mem_mx_u32(NULL, dev->vaddr.gsc_bcr_32, AUTOCAL_START, AUTOCAL_START);
	return(0);
}



//*****************************************************************************
static int _autocal(dev_data_t* dev, void* arg)
{
	u32			mask;
	long		ms		= dev->cache.autocal_ms + 5000;
	u32			reg;
	int			ret;
	os_sem_t	sem;
	int			tmp;
	VADDR_T		va		= dev->vaddr.gsc_icr_32;
	u32			value;
	gsc_wait_t	wt;

	// Safely select the Autocalibration Done interrupt.
	mask	= IRQ0_MASK | IRQ0_ACTIVE | IRQ1_ACTIVE;
	value	= IRQ0_AUTOCAL_DONE;
	os_reg_mem_mx_u32(dev, va, value, mask);

	// Wait for the local interrupt.
	os_sem_create(&sem);	// dummy, required for wait operations.
	memset(&wt, 0, sizeof(wt));
	wt.flags		= GSC_WAIT_FLAG_INTERNAL;
	wt.gsc			= AI64_WAIT_GSC_AUTOCAL_DONE;
	wt.timeout_ms	= ms;
	ret				= gsc_wait_event(dev, &wt, _autocal_start, NULL, &sem);
	os_sem_destroy(&sem);

	if (wt.flags & GSC_WAIT_FLAG_TIMEOUT)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATE DONE IRQ TIMED OUT AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually wait for completion in case something terminates our wait early.
	va	= dev->vaddr.gsc_bcr_32;
	tmp	= gsc_poll_u32(dev, ms, va, AUTOCAL_START, 0);

	if (tmp)
	{
		ms	*= 2;
		ret	= ret ? ret : tmp;
		printf(	"%s: AUTOCALIBRATION DID NOT COMPLETE WITHIN %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually check for completion as the IRQ wait may have ended early.
	reg	= os_reg_mem_rx_u32(dev, va);

	if (reg & AUTOCAL_START)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATION STILL ACTIVE AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Final results.
	reg	= os_reg_mem_rx_u32(dev, va);

	if ((reg & AUTOCAL_PASS) == 0)
	{
		ret	= ret ? ret : -EIO;
		printf(	"%s: AUTOCALIBRATION FAILED (%ld ms).\n",
				dev->model,
				ms);
	}

	return(ret);
}



//*****************************************************************************
static int _autocal_status(dev_data_t* dev, s32* arg)
{
	u32	reg;

	reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bcr_32);

	if (reg & AUTOCAL_START)
		arg[0]	= AI64_AUTOCAL_STATUS_ACTIVE;
	else if (reg & AUTOCAL_PASS)
		arg[0]	= AI64_AUTOCAL_STATUS_PASS;
	else
		arg[0]	= AI64_AUTOCAL_STATUS_FAIL;

	return(0);
}



//*****************************************************************************
static int _ai_buf_clear(dev_data_t* dev, void* arg)
{
	#define	CLEAR	D16		// IBCR
	int		i;
	u32		reg;
	int		ret	= 0;
	VADDR_T	va	= dev->vaddr.gsc_ibcr_32;

	if (dev->cache.low_latency)
	{
		// Quietly do nothing.
	}
	else
	{
		// Clear the buffer.
		os_reg_mem_mx_u32(dev, va, CLEAR, CLEAR);

		for (i = 0;; i++)
		{
			reg	= os_reg_mem_rx_u32(dev, va);

			if ((reg & CLEAR) == 0)
				break;

			if (i >= 250)
			{
				ret	= -EINVAL;
				printf(	"%s: The analog input buffer took too long to clear.\n",
						DEV_NAME);
				break;
			}
		}
	}

	return(ret);
}



//*****************************************************************************
static int _ai_buf_level(dev_data_t* dev, s32* arg)
{
	if (dev->cache.low_latency)
	{
		arg[0]	= -1;
	}
	else
	{
		arg[0]	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsr_32);
		arg[0]	&= 0xFFFF;
	}

	return(0);
}



//*****************************************************************************
static int _ai_buf_thr_lvl(dev_data_t* dev, s32* arg)
{
	int	ret;

	if (dev->cache.low_latency)
	{
		ret		= 0;
		arg[0]	= -1;
	}
	else
	{
		ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFF, dev->vaddr.gsc_ibcr_32, 15, 0);
	}

	return(ret);
}



//*****************************************************************************
static int _ai_buf_thr_sts(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AI64_AI_BUF_THR_STS_CLEAR,
		AI64_AI_BUF_THR_STS_SET,
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;

	if (dev->cache.low_latency)
	{
		ret	= 0;
	}
	else
	{
		ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_ibcr_32, 17, 17);
	}

	return(ret);
}



//*****************************************************************************
static int _ai_clk_src(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AI64_AI_CLK_SRC_RAG,
		AI64_AI_CLK_SRC_RBG,
		AI64_AI_CLK_SRC_EXT,
		AI64_AI_CLK_SRC_BCR,
		-1	// terminate list
	};

	int	ret;

	if (dev->cache.low_latency)
	{
		ret		= 0;
		arg[0]	= -1;
	}
	else
	{
		ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_sscr_32, 4, 3);
	}

	return(ret);
}



//*****************************************************************************
static int _ai_mode(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AI64_AI_MODE_SINGLE,
		AI64_AI_MODE_DIFF,
		AI64_AI_MODE_ZERO,
		AI64_AI_MODE_VREF,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 2, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_range(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AI64_AI_RANGE_2_5V,
		AI64_AI_RANGE_5V,
		AI64_AI_RANGE_10V,
		-1	// terminate list
	};

	int	ret;
	s32	v	= arg[0];

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 5, 4);

	if ((v == -1) && (arg[0] == 3))
		arg[0]	= AI64_AI_RANGE_10V;

	return(ret);
}



//*****************************************************************************
static int _ai_scan_size(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AI64_AI_SCAN_SIZE_SINGLE,
		AI64_AI_SCAN_SIZE_0_1,
		AI64_AI_SCAN_SIZE_0_3,
		AI64_AI_SCAN_SIZE_0_7,
		AI64_AI_SCAN_SIZE_0_15,
		AI64_AI_SCAN_SIZE_0_31,
		AI64_AI_SCAN_SIZE_0_63,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_sscr_32, 2, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_sync(dev_data_t* dev, void* arg)
{
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bcr_32, D12, D12);
	return(0);
}



//*****************************************************************************
static int _chan_single(dev_data_t* dev, void* arg)
{
	u32	last;
	int	ret;

	last	= dev->cache.channel_qty - 1;
	ret		= gsc_s32_range_reg(dev, arg, 0, last, dev->vaddr.gsc_sscr_32, 17, 12);
	return(ret);
}



//*****************************************************************************
static int _data_format(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AI64_DATA_FORMAT_2S_COMP,
		AI64_DATA_FORMAT_OFF_BIN,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 6, 6);
	return(ret);
}



//*****************************************************************************
static int _ext_sync_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AI64_EXT_SYNC_ENABLE_NO,
		AI64_EXT_SYNC_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 7, 7);
	return(ret);
}



//*****************************************************************************
static int _irq0_sel(dev_data_t* dev, s32* arg)
{
	u32	mask;
	u32	reg;
	int	ret	= 0;

	if (arg[0] == -1)
	{
		// Retrieve the current setting.
		reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_icr_32);
		arg[0]	= GSC_FIELD_DECODE(reg, 2, 0);
	}
	else
	{
		// Validate the option value passed in.

		switch (arg[0])
		{
			default:

				ret	= -EINVAL;
				break;

			case AI64_IRQ0_INIT_DONE:
			case AI64_IRQ0_AUTOCAL_DONE:
			case AI64_IRQ0_SYNC_START:
			case AI64_IRQ0_SYNC_DONE:

				break;
		}

		if (ret == 0)
		{
			// Clear the IRQ0 status bit and apply the new setting.
			mask	= (dev->cache.icr_anomaly) ? 0x8F : 0x07;
			os_reg_mem_mx_u32(dev, dev->vaddr.gsc_icr_32, arg[0], mask);
		}
	}

	return(ret);
}



//*****************************************************************************
static int _irq1_sel(dev_data_t* dev, s32* arg)
{
	u32	mask;
	u32	reg;
	int	ret	= 0;

	if (dev->cache.irq1 == 0)
	{
		arg[0]	= -1;
	}
	else if (arg[0] == -1)
	{
		// Retrieve the current setting.
		reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_icr_32);
		arg[0]	= GSC_FIELD_DECODE(reg, 6, 4);
	}
	else
	{
		// Validate the option value passed in.

		switch (arg[0])
		{
			default:

				ret	= -EINVAL;
				break;

			case AI64_IRQ1_NONE:
			case AI64_IRQ1_IN_BUF_THR_L2H:
			case AI64_IRQ1_IN_BUF_THR_H2L:

				break;
		}

		if (ret == 0)
		{
			// Clear the IRQ1 status bit and apply the new setting.
			mask	= (dev->cache.icr_anomaly) ? 0xF8 : 0x70;
			os_reg_mem_mx_u32(dev, dev->vaddr.gsc_icr_32, arg[0] << 4, mask);
		}
	}

	return(ret);
}



//*****************************************************************************
static int _rag_enable(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AI64_GEN_ENABLE_NO,
		AI64_GEN_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	if (dev->cache.low_latency)
	{
		ret		= 0;
		arg[0]	= -1;
	}
	else
	{
		ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_ragr_32, 16, 16);
	}

	return(ret);
}



//*****************************************************************************
static int _rag_nrate(dev_data_t* dev, s32* arg)
{
	int	ret;

	if (dev->cache.low_latency)
	{
		ret		= 0;
		arg[0]	= -1;
	}
	else
	{
		ret	= gsc_s32_range_reg(
				dev,
				arg,
				dev->cache.nrate_min,
				dev->cache.nrate_max,
				dev->vaddr.gsc_ragr_32,
				15,
				0);
	}

	return(ret);
}



//*****************************************************************************
static int _rbg_clk_src(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AI64_RBG_CLK_SRC_MASTER,
		AI64_RBG_CLK_SRC_RAG,
		-1	// terminate list
	};

	int	ret;

	if (dev->cache.low_latency)
	{
		ret		= 0;
		arg[0]	= -1;
	}
	else
	{
		ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_sscr_32, 10, 10);
	}

	return(ret);
}



//*****************************************************************************
static int _rbg_enable(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AI64_GEN_ENABLE_NO,
		AI64_GEN_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	if (dev->cache.low_latency)
	{
		ret		= 0;
		arg[0]	= -1;
	}
	else
	{
		ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_rbgr_32, 16, 16);
	}

	return(ret);
}



//*****************************************************************************
static int _rbg_nrate(dev_data_t* dev, s32* arg)
{
	int	ret;

	if (dev->cache.low_latency)
	{
		ret		= 0;
		arg[0]	= -1;
	}
	else
	{
		ret	= gsc_s32_range_reg(
				dev,
				arg,
				dev->cache.nrate_min,
				dev->cache.nrate_max,
				dev->vaddr.gsc_rbgr_32,
				15,
				0);
	}

	return(ret);
}



//*****************************************************************************
static int _rx_io_abort(dev_data_t* dev, s32* arg)
{
	arg[0]	= gsc_read_abort_active_xfer(dev, &dev->io.rx);
	return(0);
}



//*****************************************************************************
static int _rx_io_mode(dev_data_t* dev, s32* arg)
{
	static const s32	list[]	=
	{
		GSC_IO_MODE_PIO,
		GSC_IO_MODE_BMDMA,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.rx.io_mode);
	return(ret);
}



//*****************************************************************************
static int _rx_io_timeout(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_var(
			arg,
			AI64_IO_TIMEOUT_MIN,
			AI64_IO_TIMEOUT_INFINITE,
			&dev->io.rx.timeout_s);
	return(ret);
}



//*****************************************************************************
static int _conv_count(dev_data_t* dev, s32* arg)
{
	int	ret;

	if (dev->cache.low_latency)
	{
		ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFFF, dev->vaddr.gsc_ccr_32, 19, 0);
	}
	else
	{
		ret		= 0;
		arg[0]	= -1;
	}

	return(ret);
}



//*****************************************************************************
static int _scan_enable(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AI64_SCAN_ENABLE_NO,
		AI64_SCAN_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	if (dev->cache.low_latency)
	{
		ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_sscr_32, 5, 5);
	}
	else
	{
		ret		= 0;
		arg[0]	= -1;
	}

	return(ret);
}



//*****************************************************************************
static int _low_lat_read(dev_data_t* dev, ai64_ll_t* arg)
{
	int		begin;
	int		end;
	int		i;
	u32		reg;
	int		ret		= 0;
	VADDR_T	va;

	memset(arg, 0, sizeof(arg[0]));

	if (dev->cache.low_latency == 0)
	{
		// The Low Latency feature is unavailable.
		arg->ret	= -1;
	}
	else
	{
		// Initialize ...
		begin	= 0;
		end		= dev->cache.channel_qty - 1;
		va		= dev->vaddr.gsc_llr_32;

		// Sequentially read the Low Latency registers.

		for (i = begin; i <= end; i++)
		{
			reg				= os_reg_mem_rx_u32(dev, va);
			arg->data[i]	= (u16) (0xFFFF & reg);
			va				= (VADDR_T) (void*) ((unsigned long) va + 4);
		}
	}

	return(ret);
}



//*****************************************************************************
static int _reset_inputs(dev_data_t* dev, void* arg)
{
	if (dev->cache.low_latency)
	{
		os_reg_mem_mx_u32(dev, dev->vaddr.gsc_sscr_32, D18, D18);
	}
	else
	{
		// Quietly do nothing.
	}

	return(0);
}



// variables ******************************************************************

const gsc_ioctl_t	dev_ioctl_list[]	=
{
	{ AI64_IOCTL_REG_READ,			(void*) gsc_reg_read_ioctl		},
	{ AI64_IOCTL_REG_WRITE,			(void*) gsc_reg_write_ioctl		},
	{ AI64_IOCTL_REG_MOD,			(void*) gsc_reg_mod_ioctl		},
	{ AI64_IOCTL_QUERY,				(void*) _query					},
	{ AI64_IOCTL_INITIALIZE,		(void*) initialize_ioctl		},
	{ AI64_IOCTL_AUTOCAL,			(void*) _autocal				},
	{ AI64_IOCTL_AUTOCAL_STATUS,	(void*) _autocal_status			},
	{ AI64_IOCTL_AI_BUF_CLEAR,		(void*) _ai_buf_clear			},
	{ AI64_IOCTL_AI_BUF_LEVEL,		(void*) _ai_buf_level			},
	{ AI64_IOCTL_AI_BUF_THR_LVL,	(void*) _ai_buf_thr_lvl			},
	{ AI64_IOCTL_AI_BUF_THR_STS,	(void*) _ai_buf_thr_sts			},
	{ AI64_IOCTL_AI_CLK_SRC,		(void*) _ai_clk_src				},
	{ AI64_IOCTL_AI_MODE,			(void*) _ai_mode				},
	{ AI64_IOCTL_AI_RANGE,			(void*) _ai_range				},
	{ AI64_IOCTL_AI_SCAN_SIZE,		(void*) _ai_scan_size			},
	{ AI64_IOCTL_AI_SYNC,			(void*) _ai_sync				},
	{ AI64_IOCTL_CHAN_SINGLE,		(void*) _chan_single			},
	{ AI64_IOCTL_DATA_FORMAT,		(void*) _data_format			},
	{ AI64_IOCTL_EXT_SYNC_ENABLE,	(void*) _ext_sync_enable		},
	{ AI64_IOCTL_IRQ0_SEL,			(void*) _irq0_sel				},
	{ AI64_IOCTL_IRQ1_SEL,			(void*) _irq1_sel				},
	{ AI64_IOCTL_RAG_ENABLE,		(void*) _rag_enable				},
	{ AI64_IOCTL_RAG_NRATE,			(void*) _rag_nrate				},
	{ AI64_IOCTL_RBG_CLK_SRC,		(void*) _rbg_clk_src			},
	{ AI64_IOCTL_RBG_ENABLE,		(void*) _rbg_enable				},
	{ AI64_IOCTL_RBG_NRATE,			(void*) _rbg_nrate				},
	{ AI64_IOCTL_RX_IO_ABORT,		(void*) _rx_io_abort			},
	{ AI64_IOCTL_RX_IO_MODE,		(void*) _rx_io_mode				},
	{ AI64_IOCTL_RX_IO_TIMEOUT,		(void*) _rx_io_timeout			},
	{ AI64_IOCTL_WAIT_CANCEL,		(void*) gsc_wait_cancel_ioctl	},
	{ AI64_IOCTL_WAIT_EVENT,		(void*) gsc_wait_event_ioctl	},
	{ AI64_IOCTL_WAIT_STATUS,		(void*) gsc_wait_status_ioctl	},
	{ AI64_IOCTL_CONV_COUNT,		(void*) _conv_count				},
	{ AI64_IOCTL_SCAN_ENABLE,		(void*) _scan_enable			},
	{ AI64_IOCTL_LOW_LAT_READ,		(void*) _low_lat_read			},
	{ AI64_IOCTL_RESET_INPUTS,		(void*) _reset_inputs			},
	{ -1, NULL }
};


