// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI64/16AI64_Linux_3.x.x.x_DN/driver/device.c $
// $Rev: 55155 $
// $Date: 2024-08-30 10:38:53 -0500 (Fri, 30 Aug 2024) $

// 16AI64: Device Driver: source file

#include "main.h"



// variables ******************************************************************

const gsc_dev_id_t	dev_id_list[]	=
{
	// model	Vendor	Device	SubVen	SubDev	type
	{ "16AI64",	0x10B5, 0x9080, 0x10B5, 0x2407,	GSC_DEV_TYPE_16AI64	},
	{ NULL }
};



//*****************************************************************************
static void _channel_qty_compute(dev_data_t* dev)
{
	dev->cache.channel_qty		= 64;
	dev->cache.channels_max		= 64;
}



//*****************************************************************************
static void _low_latency_compute(dev_data_t* dev)
{
	if ((dev->cache.gsc_frr_32 & 0xF00) == 0x600)
		dev->cache.low_latency	= 1;
	else
		dev->cache.low_latency	= 0;
}



//*****************************************************************************
static void _regs_compute(dev_data_t* dev)
{
	int	ll	= dev->cache.low_latency ? 1 : 0;
	int	std	= dev->cache.low_latency ? 0 : 1;

	dev->cache.reg_bsr	=	std;
	dev->cache.reg_ccr	=	ll;
	dev->cache.reg_ibcr	=	std;
	dev->cache.reg_idbr	=	std;
	dev->cache.reg_ragr	=	std;
	dev->cache.reg_rbgr	=	std;
}



/******************************************************************************
*
*	Function:	dev_device_create
*
*	Purpose:
*
*		Do everything needed to setup and use the given device.
*
*	Arguments:
*
*		dev		The structure to initialize.
*
*	Returned:
*
*		0		All is well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int dev_device_create(dev_data_t* dev)
{
	static const gsc_bar_maps_t	bar_map	=
	{
		{
			// mem	io	rw
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 0: PLX registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 1: PLX registers, I/O mapped
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RW },	// BAR 2: GSC registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 3: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 4: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO }	// BAR 5: unused
		}
	};

	u32	dma;
	int	ret;

	for (;;)	// A convenience loop.
	{
		// Verify some macro contents.
		ret	= gsc_macro_test_base_name(AI64_BASE_NAME);
		if (ret)	break;

		ret	= gsc_macro_test_model();
		if (ret)	break;

		// PCI setup.
		ret	= os_pci_dev_enable(&dev->pci);
		if (ret)	break;

		ret	= os_pci_master_set(&dev->pci);
		if (ret)	break;

		// Control ISR access to the device and data structure.
		ret	= os_spinlock_create(&dev->spinlock);
		if (ret)	break;

		// Control access to the device and data structure.
		ret	= os_sem_create(&dev->sem);
		if (ret)	break;

		// Access the BAR regions.
		ret	= gsc_bar_create(dev, &dev->bar, &bar_map);
		if (ret)	break;

		// Firmware access.
		dev->vaddr.gsc_bcr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI64_GSC_BCR));
		dev->vaddr.gsc_bsr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI64_GSC_BSR));
		dev->vaddr.gsc_ccr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI64_GSC_CCR));
		dev->vaddr.gsc_frr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI64_GSC_FRR));
		dev->vaddr.gsc_ibcr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI64_GSC_IBCR));
		dev->vaddr.gsc_icr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI64_GSC_ICR));
		dev->vaddr.gsc_idbr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI64_GSC_IDBR));
		dev->vaddr.gsc_ragr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI64_GSC_RAGR));
		dev->vaddr.gsc_rbgr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI64_GSC_RBGR));
		dev->vaddr.gsc_sscr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI64_GSC_SSCR));
		dev->vaddr.gsc_llr_32		= GSC_VADDR(dev, 0x100);

		// Data cache initialization.
		dev->cache.gsc_frr_32		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_frr_32);

		_low_latency_compute(dev);	// must be 1st or 2nd
		_channel_qty_compute(dev);	// must be 1st or 2nd
		_regs_compute(dev);

		dev->cache.autocal_ms		= 4000;
		dev->cache.fifo_size		= dev->cache.low_latency ? 0 : _64K;
		dev->cache.fsamp_max		= 500000L;	// ~350,000 for multi-channel scans
		dev->cache.fsamp_min		= 1L;
		dev->cache.icr_anomaly		= 1;
		dev->cache.initialize_ms	= 3;
		dev->cache.master_clock		= 24000000L;
		dev->cache.nrate_max		= 0xFFFF;
		dev->cache.nrate_min		= dev->cache.master_clock / dev->cache.fsamp_max;
		dev->cache.rate_gen_qty		= dev->cache.low_latency ? 1 : 2;
		dev->cache.irq1				= dev->cache.low_latency ? 0 : 1;

		// Initialize additional resources.
		ret	= dev_irq_create(dev);
		if (ret)	break;

		ret	= dev_io_create(dev);
		if (ret)	break;

		dma	= GSC_DMA_SEL_STATIC
			| GSC_DMA_CAP_BMDMA_READ;
		ret	= gsc_dma_create(dev, dma, dma);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	dev_device_destroy
*
*	Purpose:
*
*		Do everything needed to release the referenced device.
*
*	Arguments:
*
*		dev		The partial data for the device of interest.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void dev_device_destroy(dev_data_t* dev)
{
	if (dev)
	{
		gsc_dma_destroy(dev);
		dev_io_destroy(dev);
		dev_irq_destroy(dev);
		gsc_bar_destroy(&dev->bar);
		os_sem_destroy(&dev->sem);
		os_spinlock_destroy(&dev->spinlock);
		os_pci_master_clear(&dev->pci);
		os_pci_dev_disable(&dev->pci);
	}
}


