// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/utils/util_reg.c $
// $Rev: 54749 $
// $Date: 2024-07-01 09:34:12 -0500 (Mon, 01 Jul 2024) $

// 16AI32SSC: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	_GSC_REG(a)			"GSC " #a, AI32SSC_GSC_##a, 0, 0
#define	NAME_WIDTH			21



//*****************************************************************************
static int _bctlr_decode(int fd, int supported, u32 value, int width)
{
	static const char * const	aim[]	=
	{
		"Differential",
		"Reserved",
		"Zero Test",
		"+Vref Test",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved"
	};

	static const char * const	range[]	=
	{
		"+-2.5 Volts",
		"+-5 Volts",
		"+-10 Volts",
		"+-10 Volts"
	};

	static const char * const	disabled[]	= { "Disabled",			"Enabled"		};
	static const char * const	enabled[]	= { "Enabled",			"Disabled"		};
	static const char * const	idle[]		= { "Idle",				"Active"		};
	static const char * const	no[]		= { "No",				"Yes"			};
	static const char * const	twos[]		= { "Twos Compliment",	"Offset Binary"	};

	int	errs;
	int	nw		= NAME_WIDTH;
	s32	query;

	width	+= 10;

	// D31-D21
	gsc_reg_field_show(width, nw, value, 31, 21, 1, NULL,	"Reserved");

	// D20
	errs	= ai32ssc_query(fd, -1, 0, AI32SSC_QUERY_TIME_TAG, &query);

	if (query)
		gsc_reg_field_show(width, nw, value, 20, 20, 1, disabled,	"Time Tag");
	else
		gsc_reg_field_show(width, nw, value, 20, 20, 1, NULL,		"Reserved");

	// D19-D0
	gsc_reg_field_show(width, nw, value, 19, 19, 1, NULL,		"Reserved"			);
	gsc_reg_field_show(width, nw, value, 18, 18, 1, disabled,	"Data Packing"		);
	gsc_reg_field_show(width, nw, value, 17, 17, 1, no,			"Buffer Overflow"	);
	gsc_reg_field_show(width, nw, value, 16, 16, 1, no,			"Buffer Underflow"	);
	gsc_reg_field_show(width, nw, value, 15, 15, 1, idle,		"Initialize"		);
	gsc_reg_field_show(width, nw, value, 14, 14, 1, no,			"Autocal Pass"		);
	gsc_reg_field_show(width, nw, value, 13, 13, 1, idle,		"Autocal"			);
	gsc_reg_field_show(width, nw, value, 12, 12, 1, idle,		"Input Sync"		);
	gsc_reg_field_show(width, nw, value, 11, 11, 1, enabled,	"Scan Marker"		);
	gsc_reg_field_show(width, nw, value, 10,  7, 1, NULL,		"Reserved"			);
	gsc_reg_field_show(width, nw, value,  6,  6, 1, twos,		"Data Format"		);
	gsc_reg_field_show(width, nw, value,  5,  4, 1, range,		"Range"				);
	gsc_reg_field_show(width, nw, value,  3,  3, 1, NULL,		"Reserved"			);
	gsc_reg_field_show(width, nw, value,  2,  0, 1, aim,		"Input Mode"		);

	return(errs);
}



//*****************************************************************************
static int _icr_decode(int fd, int supported, u32 value, int width)
{
	static const char * const	irq0[]	=
	{
		"Initialization Complete",
		"Autocal Completed",
		"Input Sample Initiated (Sync)",
		"Input Sample Completed (data ready)",
		"Trigger Burst Initiated (Burst Busy -> High",
		"Trigger Burst Completed (Burst Busy -> Low",
		"Reserved",
		"Reserved"
	};

	static const char * const	irq1[]	=
	{
		"Idle/None",
		"Input Buffer Threshold Low->High",
		"Input Buffer Threshold High->Low",
		"Input Buffer Overflow/Underflow",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved"
	};

	static const char * const	negated[]	= { "Negated",	"Asserted"	};

	int	nw	= NAME_WIDTH;

	width	+= 10;

	gsc_reg_field_show(width, nw, value, 31,  8, 1, NULL,		"Reserved"		);
	gsc_reg_field_show(width, nw, value,  7,  7, 1, negated,	"IRQ1 Request"	);
	gsc_reg_field_show(width, nw, value,  6,  4, 1, irq1,		"IRQ1 Select"	);
	gsc_reg_field_show(width, nw, value,  3,  3, 1, negated,	"IRQ0 Request"	);
	gsc_reg_field_show(width, nw, value,  2,  0, 1, irq0,		"IRQ0 Select"	);

	return(0);
}



//*****************************************************************************
static int _ibdr_decode(int fd, int supported, u32 value, int width)
{
	static const char * const	tag[]	= { "Not First",	"First"	};

	int	nw	= NAME_WIDTH;

	width	+= 10;

	gsc_reg_field_show(width, nw, value, 31, 31, 1, tag,	"Channel Tag"	);
	gsc_reg_field_show(width, nw, value, 30, 16, 1, NULL,	"Reserved"		);
	gsc_reg_field_show(width, nw, value, 15,  0, 1, NULL,	"Data"			);

	return(0);
}



//*****************************************************************************
static int _idcr_decode(int fd, int supported, u32 value, int width)
{
	static const char * const	clear[]	= { "Clear",	"Set"		};
	static const char * const	idle[]	= { "Idle",		"Active"	};

	int	nw	= NAME_WIDTH;

	width	+= 10;

	gsc_reg_field_show(width, nw, value, 31, 20, 1, NULL,	"Reserved"			);
	gsc_reg_field_show(width, nw, value, 19, 19, 1, clear,	"Threshold Flag"	);
	gsc_reg_field_show(width, nw, value, 18, 18, 1, idle,	"Clear Buffer"		);
	gsc_reg_field_show(width, nw, value, 17,  0, 1, NULL,	"Threshold"			);

	return(0);
}



//*****************************************************************************
static int _rgr_decode(int fd, int supported, u32 value, int width)
{
	static const char * const	enabled[]	= { "Enabled",	"Disabled"	};

	int	nw	= NAME_WIDTH;

	width	+= 10;

	gsc_reg_field_show(width, nw, value, 31, 17, 1, NULL,		"Reserved"	);
	gsc_reg_field_show(width, nw, value, 16, 16, 1, enabled,	"Disable"	);
	gsc_reg_field_show(width, nw, value, 15,  0, 1, NULL,		"Nrate"		);

	return(0);
}



//*****************************************************************************
static int _bufsr_decode(int fd, int supported, u32 value, int width)
{
	int	nw	= NAME_WIDTH;

	width	+= 10;

	gsc_reg_field_show(width, nw, value, 31, 19, 1, NULL,	"Reserved"			);
	gsc_reg_field_show(width, nw, value, 18,  0, 1, NULL,	"Filled Locations"	);

	return(0);
}



//*****************************************************************************
static int _bsizr_decode(int fd, int supported, u32 value, int width)
{
	int	nw	= NAME_WIDTH;

	width	+= 10;

	gsc_reg_field_show(width, nw, value, 31, 19, 1, NULL,	"Reserved"		);
	gsc_reg_field_show(width, nw, value, 19,  0, 1, NULL,	"Burst Size"	);

	return(0);
}



//*****************************************************************************
static int _sscr_decode(int fd, int supported, u32 value, int width)
{
	static const char * const	burst[]	=
	{
		"Disabled",
		"Rate-B Generator",
		"Ext Sync Input (target Mode)",
		"Software (BCTLR Input Sync Bit)"
	};

	static const char * const	clk_src[]	=
	{
		"External Clock (clock Target mode)",
		"Rate-A Generator",
		"Rate-B Generator",
		"Software (BCTLR Input Sync Bit)"
	};

	static const char * const	channels[]	=
	{
		"Single Channel Mode",
		"2 Channels (0-1)",
		"4 Channels (0-3)",
		"8 Channels (0-7)",
		"16 Channels (0-15)",
		"32 Channels (0-31)",
		"Reserved",
		"Channel Range (x-y)"
	};

	static const char * const	b_src[]		= { "Master Clock",	"Rate-A Generator"	};
	static const char * const	disabled[]	= { "Disabled",		"Enabled"			};
	static const char * const	idle[]		= { "Idle",			"Active"			};
	static const char * const	no[]		= { "No",			"Yes"				};

	int	nw	= NAME_WIDTH;

	width	+= 10;

	gsc_reg_field_show(width, nw, value, 31, 18, 1, NULL,		"Reserved"				);
	gsc_reg_field_show(width, nw, value, 17, 12, 1, NULL,		"Single Channel"		);
	gsc_reg_field_show(width, nw, value, 11, 11, 1, no,			"Invert I/O"			);
	gsc_reg_field_show(width, nw, value, 10, 10, 1, b_src,		"Rate-B Gen. Source"	);
	gsc_reg_field_show(width, nw, value,  9,  8, 1, burst,		"Bursting Source"		);
	gsc_reg_field_show(width, nw, value,  7,  7, 1, idle,		"Bursting"				);
	gsc_reg_field_show(width, nw, value,  6,  6, 1, NULL,		"Reserved"				);
	gsc_reg_field_show(width, nw, value,  5,  5, 1, disabled,	"ADC Clocking"			);
	gsc_reg_field_show(width, nw, value,  4,  3, 1, clk_src,	"Sample Clock Source"	);
	gsc_reg_field_show(width, nw, value,  2,  0, 1, channels,	"Active Channel"		);

	return(0);
}



//*****************************************************************************
static int _acar_decode(int fd, int supported, u32 value, int width)
{
	int	nw	= NAME_WIDTH;

	width	+= 10;

	gsc_reg_field_show(width, nw, value, 31, 16, 1, NULL,	"Reserved"		);
	gsc_reg_field_show(width, nw, value, 15,  8, 1, NULL,	"Last Channel"	);
	gsc_reg_field_show(width, nw, value,  7,  0, 1, NULL,	"First Channel"	);

	return(0);
}



//*****************************************************************************
static int _bcfgr_decode(int fd, int supported, u32 value, int width)
{
	static const char * const	channels[]	=
	{
		"32 Channels",
		"16 Channels",
		"Reserved",
		"Reserved"
	};

	static const char * const	custom[]	=
	{
		"None",
		"Timestamp",
		"Reserved",
		"Reserved"
	};

	static const char * const	master[]	=
	{
		"50 MHz",
		"Reserved",
		"Reserved",
		"Reserved"
	};

	static const char * const	id[]	= { "ERROR",	"16AI32SSC"	};

	int	nw	= NAME_WIDTH;

	width	+= 10;

	gsc_reg_field_show(width, nw, value, 31, 22, 1, NULL,		"Reserved"			);
	gsc_reg_field_show(width, nw, value, 21, 20, 1, custom,		"Custom Feature"	);
	gsc_reg_field_show(width, nw, value, 19, 18, 1, master,		"Master Clock"		);
	gsc_reg_field_show(width, nw, value, 17, 16, 1, channels,	"Channels"			);
	gsc_reg_field_show(width, nw, value, 15, 15, 1, id,			"Product Id"		);
	gsc_reg_field_show(width, nw, value, 14, 12, 1, NULL,		"Reserved"			);
	gsc_reg_field_show(width, nw, value, 11,  0, 1, NULL,		"Firmware Rev"		);

	return(0);
}



//*****************************************************************************
static int _asiocr_decode(int fd, int supported, u32 value, int width)
{
	static const char * const	clock[]	=
	{
		"Inactive",
		"Active Input (Low-to-High Edge)",
		"Active Output (Positive Pulse)",
		"Reserved"
	};

	static const char * const	noise[]	= { "100ns-135ns",		"1.5us-2.0us"		};
	static const char * const	high[]	= { "High Pulses",		"Low Pulses"		};
	static const char * const	low[]	= { "Low-to-High Edge",	"High-to-Low Edge"	};

	int	nw	= NAME_WIDTH;

	width	+= 10;

	gsc_reg_field_show(width, nw, value, 31, 11, 1, NULL,	"Reserved"			);
	gsc_reg_field_show(width, nw, value, 10, 10, 1, noise,	"Noise Supression"	);
	gsc_reg_field_show(width, nw, value,  9,  9, 1, high,	"Output Polarity"	);
	gsc_reg_field_show(width, nw, value,  8,  8, 1, low,	"Input Polarity"	);
	gsc_reg_field_show(width, nw, value,  7,  4, 1, NULL,	"Reserved"			);
	gsc_reg_field_show(width, nw, value,  3,  2, 1, clock,	"Aux Sync Mode"		);
	gsc_reg_field_show(width, nw, value,  1,  0, 1, clock,	"Aux Clock Mode"	);

	return(0);
}



//*****************************************************************************
static int _smwr_decode(int fd, int supported, u32 value, int width)
{
	int	nw	= NAME_WIDTH;

	width	+= 10;

	gsc_reg_field_show(width, nw, value, 31, 12, 1, NULL,	"Reserved"			);
	gsc_reg_field_show(width, nw, value, 11,  6, 1, NULL,	"Release Channel"	);
	gsc_reg_field_show(width, nw, value,  5,  0, 1, NULL,	"Hold Channel"		);

	return(0);
}



//*****************************************************************************
static int _ttcr_decode(int fd, int supported, u32 value, int width)
{
	static const char * const	clock[]	=
	{
		"Rate-A Generator",
		"SAMP CLK INP",
		"REF CLK INP",
		"Reserved"
	};

	static const char * const	disabled[]	= { "Disabled",				"Enabled"			};
	static const char * const	idle[]		= { "Idle",					"Reset"				};
	static const char * const	logging[]	= { "Triggerred Channels",	"Active Channels"	};
	static const char * const	reference[]	= { "Auto-Update",			"Unchanged"			};
	static const char * const	source[]	= { "Internal",				"REF CLK INP"		};
	static const char * const	trigger[]	= { "Free Run",				"Reference-Trigger"	};

	int	errs	= 0;
	int	nw		= NAME_WIDTH;
	s32	query;

	width	+= 10;

	if (supported)
	{
		errs	= ai32ssc_query(fd, -1, 0, AI32SSC_QUERY_TIME_TAG, &query);

		if (errs)
			errs	= -1;
		else
			errs	= query ? 1 : 0;
	}
	else
	{
		gsc_reg_field_show(width, nw, value, 31, 12, 1, NULL,		"Reserved"			);
		gsc_reg_field_show(width, nw, value, 11, 11, 1, disabled,	"Tagging"			);
		gsc_reg_field_show(width, nw, value, 10, 10, 1, disabled,	"External Reset"	);
		gsc_reg_field_show(width, nw, value,  9,  9, 1, idle,		"Time Tag Reset"	);
		gsc_reg_field_show(width, nw, value,  8,  8, 1, source,		"Clock Source"		);
		gsc_reg_field_show(width, nw, value,  7,  7, 1, NULL,		"Reserved"			);
		gsc_reg_field_show(width, nw, value,  6,  6, 1, logging,	"Logging"			);
		gsc_reg_field_show(width, nw, value,  5,  5, 1, reference,	"Reference Value"	);
		gsc_reg_field_show(width, nw, value,  4,  4, 1, trigger,	"Reference Trigger"	);
		gsc_reg_field_show(width, nw, value,  3,  3, 1, NULL,		"Reserved"			);
		gsc_reg_field_show(width, nw, value,  2,  2, 1, disabled,	"ADC Clocking"		);
		gsc_reg_field_show(width, nw, value,  2,  2, 1, clock,		"Clocking Source"	);
	}

	return(errs);
}



//*****************************************************************************
static int _ttacmr_decode(int fd, int supported, u32 value, int width)
{
	static const char * const	no[]	= { "No",	"Yes"	};

	char	buf[64];
	s32		chans	= 32;
	int		errs;
	int		i;
	int		nw		= NAME_WIDTH;
	s32		query;

	width	+= 10;

	if (supported)
	{
		errs	= ai32ssc_query(fd, -1, 0, AI32SSC_QUERY_TIME_TAG, &query);

		if (errs)
			errs	= -1;
		else
			errs	= query ? 1 : 0;
	}
	else
	{
		errs	= ai32ssc_query(fd, -1, 0, AI32SSC_QUERY_CHANNEL_QTY, &chans);
		chans	= (chans < 1) ? 1 : ((chans > 32) ? 32 : chans);

		for (i = 31; i >= 0; i--)
		{
			if ((i + 1) > chans)
			{
				gsc_reg_field_show(width, nw, value, i, i, 1, NULL,	"Reserved"	);
			}
			else
			{
				sprintf(buf, "Acquire Chan %d", i);
				gsc_reg_field_show(width, nw, value, i, i, 1, no,	buf	);
			}
		}
	}

	return(errs);
}



//*****************************************************************************
static int _ttclr_decode(int fd, int supported, u32 value, int width)
{
	int	errs	= 0;
	int	nw		= NAME_WIDTH;
	s32	query;

	width	+= 10;

	if (supported)
	{
		errs	= ai32ssc_query(fd, -1, 0, AI32SSC_QUERY_TIME_TAG, &query);

		if (errs)
			errs	= -1;
		else
			errs	= query ? 1 : 0;
	}
	else
	{
		gsc_reg_field_show(width, nw, value, 31, 0, 1, NULL, "Lower 32-bits");
	}

	return(errs);
}



//*****************************************************************************
static int _ttcur_decode(int fd, int supported, u32 value, int width)
{
	int	errs	= 0;
	int	nw		= NAME_WIDTH;
	s32	query;

	width	+= 10;

	if (supported)
	{
		errs	= ai32ssc_query(fd, -1, 0, AI32SSC_QUERY_TIME_TAG, &query);

		if (errs)
			errs	= -1;
		else
			errs	= query ? 1 : 0;
	}
	else
	{
		gsc_reg_field_show(width, nw, value, 31, 16, 1, NULL, "Reserved"		);
		gsc_reg_field_show(width, nw, value, 15,  0, 1, NULL, "Upper 16-bits"	);
	}

	return(errs);
}



//*****************************************************************************
static int _ttrdr_decode(int fd, int supported, u32 value, int width)
{
	int	errs	= 0;
	int	nw		= NAME_WIDTH;
	s32	query;

	width	+= 10;

	if (supported)
	{
		errs	= ai32ssc_query(fd, -1, 0, AI32SSC_QUERY_TIME_TAG, &query);

		if (errs)
			errs	= -1;
		else
			errs	= query ? 1 : 0;
	}
	else
	{
		gsc_reg_field_show(width, nw, value, 31, 20, 1, NULL, "Reserved"	);
		gsc_reg_field_show(width, nw, value, 19,  0, 1, NULL, "Divider"		);
	}

	return(errs);
}



//*****************************************************************************
static int _ttbsr_decode(int fd, int supported, u32 value, int width)
{
	int	errs	= 0;
	int	nw		= NAME_WIDTH;
	s32	query;

	width	+= 10;

	if (supported)
	{
		errs	= ai32ssc_query(fd, -1, 0, AI32SSC_QUERY_TIME_TAG, &query);

		if (errs)
			errs	= -1;
		else
			errs	= query ? 1 : 0;
	}
	else
	{
		gsc_reg_field_show(width, nw, value, 31, 0, 1, NULL, "Burst Size");
	}

	return(errs);
}



//*****************************************************************************
static int _ttctrr_decode(int fd, int supported, u32 value, int width)
{
	int	errs	= 0;
	int	nw		= NAME_WIDTH;
	s32	query;

	width	+= 10;

	if (supported)
	{
		errs	= ai32ssc_query(fd, -1, 0, AI32SSC_QUERY_TIME_TAG, &query);

		if (errs)
			errs	= -1;
		else
			errs	= query ? 1 : 0;
	}
	else
	{
		gsc_reg_field_show(width, nw, value, 31, 16, 1, NULL, "Trigger Threshold");
		gsc_reg_field_show(width, nw, value, 15,  0, 1, NULL, "Trigger Reference");
	}

	return(errs);
}



// macros *********************************************************************

static gsc_reg_def_t	_gsc[]	=
{
	{ _GSC_REG(BCTLR),		0,	_bctlr_decode,	"Board Control Register"				},
	{ _GSC_REG(ICR),		0,	_icr_decode,	"Interrupt Control Register"			},
	{ _GSC_REG(IBDR),		0,	_ibdr_decode,	"Input Buffer Data Register"			},
	{ _GSC_REG(IBCR),		0,	_idcr_decode,	"Input Buffer Control Register"			},
	{ _GSC_REG(RAGR),		0,	_rgr_decode,	"Rate-A Generator Register"				},
	{ _GSC_REG(RBGR),		0,	_rgr_decode,	"Rate-B Generator Register"				},
	{ _GSC_REG(BUFSR),		0,	_bufsr_decode,	"Buffer Size Register"					},
	{ _GSC_REG(BSIZR),		0,	_bsizr_decode,	"Burst Size Register"					},
	{ _GSC_REG(SSCR),		0,	_sscr_decode,	"Scan & Sync Control Register"			},
	{ _GSC_REG(ACAR),		0,	_acar_decode,	"Active Channel Assignment Register"	},
	{ _GSC_REG(BCFGR),		0,	_bcfgr_decode,	"Board Configuration Register"			},
	{ _GSC_REG(AVR),		0,	NULL,			"Autocal Values Register"				},
	{ _GSC_REG(AWRR),		0,	NULL,			"Auxiliary R/W Register"				},
	{ _GSC_REG(ASIOCR),		0,	_asiocr_decode,	"Auxiliary Sync I/O Control Register"	},
	{ _GSC_REG(SMUWR),		0,	_smwr_decode,	"Scan Marker Upper Word Register"		},
	{ _GSC_REG(SMLWR),		0,	_smwr_decode,	"Scan Marker Lower Word Register"		},

	// Time Tag Registers
	{ _GSC_REG(TTCR),		1,	_ttcr_decode,	"Time Tag Configuration Register"		},
	{ _GSC_REG(TTACMR),		1,	_ttacmr_decode,	"Time Tag Active Channel Mask Register"	},
	{ _GSC_REG(TTCLR),		1,	_ttclr_decode,	"Time Tag Counter Lower Register"		},
	{ _GSC_REG(TTCUR),		1,	_ttcur_decode,	"Time Tag Counter Upper Register"		},
	{ _GSC_REG(TTRDR),		1,	_ttrdr_decode,	"Time Tag Rate Divider Register"		},
	{ _GSC_REG(TTBSR),		1,	_ttbsr_decode,	"Time Tag Burst Size Register"			},

	{ _GSC_REG(TTC00TRR),	1,	_ttctrr_decode,	"Time Tag Channel 00 Threshold/Reference Register"	},
	{ _GSC_REG(TTC01TRR),	1,	_ttctrr_decode,	"Time Tag Channel 01 Threshold/Reference Register"	},
	{ _GSC_REG(TTC02TRR),	1,	_ttctrr_decode,	"Time Tag Channel 02 Threshold/Reference Register"	},
	{ _GSC_REG(TTC03TRR),	1,	_ttctrr_decode,	"Time Tag Channel 03 Threshold/Reference Register"	},
	{ _GSC_REG(TTC04TRR),	1,	_ttctrr_decode,	"Time Tag Channel 04 Threshold/Reference Register"	},
	{ _GSC_REG(TTC05TRR),	1,	_ttctrr_decode,	"Time Tag Channel 05 Threshold/Reference Register"	},
	{ _GSC_REG(TTC06TRR),	1,	_ttctrr_decode,	"Time Tag Channel 06 Threshold/Reference Register"	},
	{ _GSC_REG(TTC07TRR),	1,	_ttctrr_decode,	"Time Tag Channel 07 Threshold/Reference Register"	},
	{ _GSC_REG(TTC08TRR),	1,	_ttctrr_decode,	"Time Tag Channel 08 Threshold/Reference Register"	},
	{ _GSC_REG(TTC09TRR),	1,	_ttctrr_decode,	"Time Tag Channel 09 Threshold/Reference Register"	},

	{ _GSC_REG(TTC10TRR),	1,	_ttctrr_decode,	"Time Tag Channel 10 Threshold/Reference Register"	},
	{ _GSC_REG(TTC11TRR),	1,	_ttctrr_decode,	"Time Tag Channel 11 Threshold/Reference Register"	},
	{ _GSC_REG(TTC12TRR),	1,	_ttctrr_decode,	"Time Tag Channel 12 Threshold/Reference Register"	},
	{ _GSC_REG(TTC13TRR),	1,	_ttctrr_decode,	"Time Tag Channel 13 Threshold/Reference Register"	},
	{ _GSC_REG(TTC14TRR),	1,	_ttctrr_decode,	"Time Tag Channel 14 Threshold/Reference Register"	},
	{ _GSC_REG(TTC15TRR),	1,	_ttctrr_decode,	"Time Tag Channel 15 Threshold/Reference Register"	},
	{ _GSC_REG(TTC16TRR),	1,	_ttctrr_decode,	"Time Tag Channel 16 Threshold/Reference Register"	},
	{ _GSC_REG(TTC17TRR),	1,	_ttctrr_decode,	"Time Tag Channel 17 Threshold/Reference Register"	},
	{ _GSC_REG(TTC18TRR),	1,	_ttctrr_decode,	"Time Tag Channel 18 Threshold/Reference Register"	},
	{ _GSC_REG(TTC19TRR),	1,	_ttctrr_decode,	"Time Tag Channel 19 Threshold/Reference Register"	},

	{ _GSC_REG(TTC20TRR),	1,	_ttctrr_decode,	"Time Tag Channel 20 Threshold/Reference Register"	},
	{ _GSC_REG(TTC21TRR),	1,	_ttctrr_decode,	"Time Tag Channel 21 Threshold/Reference Register"	},
	{ _GSC_REG(TTC22TRR),	1,	_ttctrr_decode,	"Time Tag Channel 22 Threshold/Reference Register"	},
	{ _GSC_REG(TTC23TRR),	1,	_ttctrr_decode,	"Time Tag Channel 23 Threshold/Reference Register"	},
	{ _GSC_REG(TTC24TRR),	1,	_ttctrr_decode,	"Time Tag Channel 24 Threshold/Reference Register"	},
	{ _GSC_REG(TTC25TRR),	1,	_ttctrr_decode,	"Time Tag Channel 25 Threshold/Reference Register"	},
	{ _GSC_REG(TTC26TRR),	1,	_ttctrr_decode,	"Time Tag Channel 26 Threshold/Reference Register"	},
	{ _GSC_REG(TTC27TRR),	1,	_ttctrr_decode,	"Time Tag Channel 27 Threshold/Reference Register"	},
	{ _GSC_REG(TTC28TRR),	1,	_ttctrr_decode,	"Time Tag Channel 28 Threshold/Reference Register"	},
	{ _GSC_REG(TTC29TRR),	1,	_ttctrr_decode,	"Time Tag Channel 29 Threshold/Reference Register"	},

	{ _GSC_REG(TTC30TRR),	1,	_ttctrr_decode,	"Time Tag Channel 30 Threshold/Reference Register"	},
	{ _GSC_REG(TTC31TRR),	1,	_ttctrr_decode,	"Time Tag Channel 31 Threshold/Reference Register"	},

	{ NULL, 0, 0, 0, 0,		NULL	}
};



//*****************************************************************************
static const gsc_reg_def_t* _find_reg(u32 reg, const gsc_reg_def_t* list)
{
	const gsc_reg_def_t*	def	= NULL;
	int						i;

	for (i = 0; list[i].name; i++)
	{
		if (reg == list[i].reg)
		{
			def	= &list[i];
			break;
		}
	}

	return(def);
}



/******************************************************************************
*
*	Function:	ai32ssc_reg_get_def_id
*
*	Purpose:
*
*		Retrieve the register definition structure given the register id.
*
*	Arguments:
*
*		reg		The id of the register to access.
*
*	Returned:
*
*		NULL	The register id wasn't found.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* ai32ssc_reg_get_def_id(u32 reg)
{
	const gsc_reg_def_t*	def;

	def	= _find_reg(reg, _gsc);
	return(def);
}



/******************************************************************************
*
*	Function:	ai32ssc_reg_get_def_index
*
*	Purpose:
*
*		Retrieve the register definition structure based on an index.
*
*	Arguments:
*
*		index	The index of the register to access.
*
*	Returned:
*
*		NULL	The index doesn't correspond to a known register.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* ai32ssc_reg_get_def_index(int index)
{
	const gsc_reg_def_t*	def;

	if (index < 0)
		def	= NULL;
	else if (index >= (SIZEOF_ARRAY(_gsc) - 1))
		def	= NULL;
	else
		def	= &_gsc[index];

	return(def);
}



/******************************************************************************
*
*	Function:	ai32ssc_reg_get_desc
*
*	Purpose:
*
*		Retrieve the description of the specified register.
*
*	Arguments:
*
*		reg		The register whose description is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* ai32ssc_reg_get_desc(u32 reg)
{
	const gsc_reg_def_t*	def;
	const char*				desc;

	def	= _find_reg(reg, _gsc);

	if (def)
		desc	= def->desc;
	else
		desc	= "UNKNOWN";

	return(desc);
}



/******************************************************************************
*
*	Function:	ai32ssc_reg_get_name
*
*	Purpose:
*
*		Retrieve the name of the specified register.
*
*	Arguments:
*
*		reg		The register whose name is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* ai32ssc_reg_get_name(u32 reg)
{
	const gsc_reg_def_t*	def;
	const char*				name;

	def	= _find_reg(reg, _gsc);

	if (def)
		name	= def->name;
	else
		name	= "UNKNOWN";

	return(name);
}



/******************************************************************************
*
*	Function:	ai32ssc_reg_list
*
*	Purpose:
*
*		List the GSC registers and their values.
*
*	Arguments:
*
*		fd		The handle to access the device.
*
*		detail	List the register details?
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int ai32ssc_reg_list(int fd, int detail)
{
	int	errs;

	errs	= gsc_reg_list(fd, _gsc, detail, ai32ssc_reg_read);
	return(errs);
}


