// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/utils/util_query.c $
// $Rev: 53690 $
// $Date: 2023-09-07 13:47:54 -0500 (Thu, 07 Sep 2023) $

// 16AI32SSC: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	ai32ssc_query
*
*	Purpose:
*
*		Provide a visual wrapper for the AI32SSC_IOCTL_QUERY service.
*
*	Arguments:
*
*		fd		Use this handle to access the device.
*
*		index	The index of the device to access. Ignore if < 0.
*				This is for display purposes only.
*
*		verbose	Work verbosely?
*
*		set		This is the query option to access.
*
*		get		The results are reported here. This may be NULL.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int	ai32ssc_query(int fd, int index, int verbose, s32 set, s32* get)
{
	char		buf[128];
	int			errs	= 0;
	const char*	ptr;
	s32			query	= set;
	int			ret;
	s32			tmp;

	switch (query)
	{
		default:

			errs++;
			ptr	= buf;
			sprintf(buf, "Query Error (#%ld)", (long) query);
			break;

		case AI32SSC_QUERY_AUTOCAL_MS:		ptr	= "Autocal Period";		break;
		case AI32SSC_QUERY_CHANNEL_MAX:		ptr	= "Max Channels";		break;
		case AI32SSC_QUERY_CHANNEL_QTY:		ptr	= "Input Channels";		break;
		case AI32SSC_QUERY_COUNT:			ptr	= "Query Options";		break;
		case AI32SSC_QUERY_DEVICE_TYPE:		ptr	= "Device Type";		break;
		case AI32SSC_QUERY_FGEN_MAX:		ptr	= "Fgen Max";			break;
		case AI32SSC_QUERY_FGEN_MIN:		ptr	= "Fgen Min";			break;
		case AI32SSC_QUERY_FIFO_SIZE:		ptr	= "FIFO Size";			break;
		case AI32SSC_QUERY_FSAMP_MAX:		ptr	= "Fsamp Max";			break;
		case AI32SSC_QUERY_FSAMP_MIN:		ptr	= "Fsamp Min";			break;
		case AI32SSC_QUERY_INIT_MS:			ptr	= "Initialize Period";	break;
		case AI32SSC_QUERY_LOW_LATENCY:		ptr = "Low Latency";		break;
		case AI32SSC_QUERY_MASTER_CLOCK:	ptr	= "Master Clock";		break;
		case AI32SSC_QUERY_NRATE_MASK:		ptr = "Nrate Mask";			break;
		case AI32SSC_QUERY_NRATE_MAX:		ptr = "Nrate Max";			break;
		case AI32SSC_QUERY_NRATE_MIN:		ptr = "Nrate Min";			break;
		case AI32SSC_QUERY_RATE_GEN_QTY:	ptr = "Rate Generators";	break;
		case AI32SSC_QUERY_RBG_SYNC_OUTPUT:	ptr = "RBG Sync Out";		break;
		case AI32SSC_QUERY_TIME_TAG:		ptr = "Time Tag";			break;
	}

	if (verbose)
		gsc_label_index(ptr, index);

	ret		= ai32ssc_ioctl(fd, AI32SSC_IOCTL_QUERY, &set);
	errs	+= ret ? 1 : 0;

	switch (query)
	{
		default:

			errs++;
			sprintf(buf, "%ld", (long) set);
			break;

		case AI32SSC_QUERY_CHANNEL_MAX:
		case AI32SSC_QUERY_CHANNEL_QTY:
		case AI32SSC_QUERY_COUNT:
		case AI32SSC_QUERY_NRATE_MAX:
		case AI32SSC_QUERY_NRATE_MIN:
		case AI32SSC_QUERY_RATE_GEN_QTY:

			gsc_label_long_comma_buf(set, buf);
			break;

		case AI32SSC_QUERY_NRATE_MASK:

			sprintf(buf, "0x%lX", (long) set);
			break;

		case AI32SSC_QUERY_FGEN_MAX:
		case AI32SSC_QUERY_FGEN_MIN:
		case AI32SSC_QUERY_MASTER_CLOCK:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " Hz");
			break;

		case AI32SSC_QUERY_AUTOCAL_MS:
		case AI32SSC_QUERY_INIT_MS:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " ms");
			break;

		case AI32SSC_QUERY_FSAMP_MAX:
		case AI32SSC_QUERY_FSAMP_MIN:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " S/S");
			break;

		case AI32SSC_QUERY_LOW_LATENCY:
		case AI32SSC_QUERY_RBG_SYNC_OUTPUT:
		case AI32SSC_QUERY_TIME_TAG:

			sprintf(buf, "%s", set ? "Supported" : "Not Supported");
			break;

		case AI32SSC_QUERY_FIFO_SIZE:

			ptr	= "";
			tmp	= set;

			if ((tmp) && ((tmp % 1024) == 0))
			{
				ptr	= "K";
				tmp	/= 1024;
			}

			gsc_label_long_comma_buf(tmp, buf);
			strcat(buf, ptr);
			strcat(buf, " Words");
			break;

		case AI32SSC_QUERY_DEVICE_TYPE:

			if (set == GSC_DEV_TYPE_16AI32SSC)
			{
				strcpy(buf, "16AI32SSC");
			}
			else
			{
				errs++;
				sprintf(buf, "INVALID: %ld", (long) set);
			}

			break;
	}

	if (verbose)
	{
		if (errs)
			printf("FAIL <---  (%s)\n", buf);
		else
			printf("%s\n", buf);
	}

	if (get)
		get[0]	= set;

	return(errs);
}


