// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/utils/util_fsamp_ai.c $
// $Rev: 53690 $
// $Date: 2023-09-07 13:47:54 -0500 (Thu, 07 Sep 2023) $

// 16AI32SSC: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	ABS(v)					(((v) < 0) ? (-(v)) : (v))
#define	LIMIT_MAX(v,max)		(((v) > (max)) ? (max) : (v))
#define	LIMIT_MIN(v,min)		(((v) < (min)) ? (min) : (v))
#define	LIMIT_RANGE(v,min,max)	(LIMIT_MIN((LIMIT_MAX((v),(max))),(min)))



// data types *****************************************************************

typedef struct
{
	s32		fsamp_want;
	double	fsamp_got;

	double	delta;
	s32		enable_a;
	s32		enable_b;
	s32		master;
	s32		fsamp_max;
	s32		fsamp_min;
	s32		nrate_a;
	s32		nrate_b;
	s32		nrate_max;
	s32		nrate_min;
	s32		src;
	s32		src_b;
} fsamp_t;



/******************************************************************************
*
*	Function:	_channels
*
*	Purpose:
*
*		Check to see which channels are active.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*		mask	The mask of active channels goes here.
*
*		tt		Tell the caller if Time Tagging is active.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

static int _channels(int fd, u32* mask, s32* tt)
{
	s32	active	= -1;
	int	errs	= 0;
	s32	first	= -1;
	s32	last	= -1;
	s32	single	= -1;
	s32	tt_en	= -1;	// Time Tag Enable
	u32	tt_mask	= 0;	// Time Tag Channel mask

	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_CHAN_ACTIVE,	&active	);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_CHAN_FIRST,	&first	);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_CHAN_LAST,	&last	);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_CHAN_SINGLE,	&single	);
	errs	+= ai32ssc_query(fd, -1, 0, AI32SSC_QUERY_TIME_TAG, tt);

	if (tt[0])
	{
		errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_TT_ENABLE, &tt_en);
		errs	+= ai32ssc_reg_read(fd, -1, 0, AI32SSC_GSC_TTACMR, &tt_mask);
		tt[0]	= (tt_en == AI32SSC_TT_ENABLE_YES) ? 1 : 0;
	}

	if (errs)
	{
	}
	else if (tt[0])
	{
		mask[0]	= tt_mask;
	}
	else
	{
		switch (active)
		{
			default:

				errs++;
				printf("FAIL <---  (invalid ACTIVE selection: %ld)\n", (long) active);
				break;

			case AI32SSC_CHAN_ACTIVE_0_1:		mask[0]	= 0x00000003;		break;
			case AI32SSC_CHAN_ACTIVE_0_3:		mask[0]	= 0x0000000F;		break;
			case AI32SSC_CHAN_ACTIVE_0_7:		mask[0]	= 0x000000FF;		break;
			case AI32SSC_CHAN_ACTIVE_0_15:		mask[0]	= 0x0000FFFF;		break;
			case AI32SSC_CHAN_ACTIVE_0_31:		mask[0]	= 0xFFFFFFFF;		break;
			case AI32SSC_CHAN_ACTIVE_SINGLE:	mask[0]	= 0x1L << single;	break;
			case AI32SSC_CHAN_ACTIVE_RANGE:

				mask[0]	= GSC_FIELD_ENCODE(0xFFFFFFFF, last, first);
				break;
		}
	}

	return(errs);
}



//*****************************************************************************
static int _rag_disabled(s32* sps, char* buf)
{
	s32	fsamp	= 0;

	strcpy(buf, "Rate-A Generator Disabled");

	if (sps)
		sps[0]	= fsamp;

	return(0);
}



//*****************************************************************************
static int _rbg_disabled(s32* sps, char* buf)
{
	s32	fsamp	= 0;

	strcpy(buf, "Rate-B Generator Disabled");

	if (sps)
		sps[0]	= fsamp;

	return(0);
}



//*****************************************************************************
static int _adc_bcr(s32* sps, char* buf)
{
	s32	fsamp	= 0;

	strcpy(buf, "Software Clocked Sampling");

	if (sps)
		sps[0]	= fsamp;

	return(0);
}



//*****************************************************************************
static int _adc_disabled(s32* sps, char* buf)
{
	s32	fsamp	= 0;

	strcpy(buf, "ADC Disabled");

	if (sps)
		sps[0]	= fsamp;

	return(0);
}



//*****************************************************************************
static int _adc_ext(s32* sps, char* buf)
{
	s32	fsamp	= 0;

	strcpy(buf, "Externally Clocked Sampling");

	if (sps)
		sps[0]	= fsamp;

	return(0);
}



//*****************************************************************************
static int _adc_rag_mc(int fd, s32* sps, char* buf)
{
	s32	a_enable	= -1;
	s32	a_nrate		= -1;
	s32	adc_enable	= -1;
	int	errs		= 0;
	s32	fsamp		= 0;
	s32	master;

	errs	+= ai32ssc_query(fd, -1, 0,	AI32SSC_QUERY_MASTER_CLOCK, &master);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_ADC_ENABLE,	&adc_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RAG_ENABLE,	&a_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RAG_NRATE,	&a_nrate);

	for (;;)	// A convenience loop.
	{
		if (adc_enable == AI32SSC_ADC_ENABLE_NO)
		{
			errs	+= _adc_disabled(&fsamp, buf);
			break;
		}

		if (a_enable == AI32SSC_GEN_ENABLE_NO)
		{
			errs	+= _rag_disabled(&fsamp, buf);
			break;
		}

		sprintf(buf, "Fref %ld, A Nrate %ld", (long) master, (long) a_nrate);

		// Rate-A Generator calculations
		fsamp	= master / a_nrate;
		break;
	}

	if ((errs == 0) && (fsamp > 200000L))
		strcat(buf, "  <--- UNSTABLE");

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static int _adc_rbg_mc(int fd, s32* sps, char* buf)
{
	s32	adc_enable	= -1;
	s32	b_enable	= -1;
	s32	b_nrate		= -1;
	int	errs		= 0;
	s32	fsamp		= 0;
	s32	master;

	errs	+= ai32ssc_query(fd, -1, 0,	AI32SSC_QUERY_MASTER_CLOCK, &master);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_ADC_ENABLE,	&adc_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_ENABLE,	&b_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_NRATE,	&b_nrate);

	for (;;)	// A convenience loop.
	{
		if (adc_enable == AI32SSC_ADC_ENABLE_NO)
		{
			errs	+= _adc_disabled(&fsamp, buf);
			break;
		}

		if (b_enable == AI32SSC_GEN_ENABLE_NO)
		{
			errs	+= _rbg_disabled(sps, buf);
			break;
		}

		sprintf(buf, "Fref %ld, B Nrate %ld", (long) master, (long) b_nrate);

		// Rate-A Generator calculations
		fsamp	= master / b_nrate;
		break;
	}

	if ((errs == 0) && (fsamp > 200000L))
		strcat(buf, "  <--- UNSTABLE");

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static int _adc_rbg_rag_mc(int fd, s32* sps, char* buf)
{
	s32	a_enable	= -1;
	s32	a_nrate		= -1;
	s32	adc_enable	= -1;
	s32	b_enable	= -1;
	s32	b_nrate		= -1;
	int	errs		= 0;
	s32	fsamp		= 0;
	s32	master;

	errs	+= ai32ssc_query(fd, -1, 0,	AI32SSC_QUERY_MASTER_CLOCK, &master);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_ADC_ENABLE,	&adc_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RAG_ENABLE,	&a_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RAG_NRATE,	&a_nrate);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_ENABLE,	&b_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_NRATE,	&b_nrate);

	for (;;)	// A convenience loop.
	{
		if (adc_enable == AI32SSC_ADC_ENABLE_NO)
		{
			errs	+= _adc_disabled(&fsamp, buf);
			break;
		}

		if (a_enable == AI32SSC_GEN_ENABLE_NO)
		{
			errs	+= _rag_disabled(&fsamp, buf);
			break;
		}

		if (b_enable == AI32SSC_GEN_ENABLE_NO)
		{
			errs	+= _rbg_disabled(sps, buf);
			break;
		}

		sprintf(buf,
				"Fref %ld, A Nrate %ld, B Nrate %ld",
				(long) master,
				(long) a_nrate,
				(long) b_nrate);

		// Rate-A/B Generator calculations
		fsamp	= (s32) (master / ((long) a_nrate * b_nrate));
		break;
	}

	if ((errs == 0) && (fsamp > 200000L))
		strcat(buf, "  <--- UNSTABLE");

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static int _adc_rbg(int fd, s32* sps, char* buf)
{
	int	errs;
	s32	fsamp	= 0;
	s32	rbg		= -1;

	errs	= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_CLK_SRC, &rbg);

	if (errs == 0)
	{
		switch (rbg)
		{
			default:

				sprintf(buf, "%d. INTERNAL ERROR: %ld", __LINE__, (long) rbg);
				break;

			case AI32SSC_RBG_CLK_SRC_MASTER:

				errs	+= _adc_rbg_mc(fd, &fsamp, buf);
				break;

			case AI32SSC_RBG_CLK_SRC_RAG:

				errs	+= _adc_rbg_rag_mc(fd, &fsamp, buf);
				break;
		}
	}

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static int _burst_bcr(s32* sps, char* buf)
{
	s32	fsamp	= 0;

	strcpy(buf, "Software Triggered Bursting");

	if (sps)
		sps[0]	= fsamp;

	return(0);
}



//*****************************************************************************
static int _burst_disable(int fd, s32* sps, char* buf)
{
	s32	adc		= -1;
	int	errs;
	s32	fsamp	= 0;

	errs	= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_ADC_CLK_SRC, &adc);

	if (errs == 0)
	{
		switch (adc)
		{
			default:

				sprintf(buf, "%d. INTERNAL ERROR: %ld", __LINE__, (long) adc);
				break;

			case AI32SSC_ADC_CLK_SRC_BCR:

				errs	+= _adc_bcr(&fsamp, buf);
				break;

			case AI32SSC_ADC_CLK_SRC_EXT:

				errs	+= _adc_ext(&fsamp, buf);
				break;

			case AI32SSC_ADC_CLK_SRC_RAG:

				errs	+= _adc_rag_mc(fd, &fsamp, buf);
				break;

			case AI32SSC_ADC_CLK_SRC_RBG:

				errs	+= _adc_rbg(fd, &fsamp, buf);
				break;
		}
	}

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static int _burst_ext(s32* sps, char* buf)
{
	s32	fsamp	= 0;

	strcpy(buf, "Externally Triggered Bursting");

	if (sps)
		sps[0]	= fsamp;

	return(0);
}



//*****************************************************************************
static int _burst_rbg_mc_adc_rag_mc(int fd, s32* sps, char* buf)
{
	s32			a_enable	= -1;
	s32			a_nrate		= -1;
	s32			b_enable	= -1;
	s32			b_nrate		= -1;
	float		b_tb;				// Burst: Time Burst
	long long	b_tb_ns;			// Burst: Time Burst Nano Seconds.
	long long	b_tt_ns;			// Burst: Time Total Nano Seconds.
	float		bps;				// Bursts Per Second
	s32			adc_enable	= -1;
	int			errs		= 0;
	s32			fsamp		= 0;
	s32			master;
	long		qty;
	long long	s_tb_ns;			// Sample: Time Burst Nano Seconds.
	float		s_ts;				// Sample: Time Scan
	long long	s_ts_ns;			// Sample: Time Scan Nano Seconds.
	s32			size		= -1;

	errs	+= ai32ssc_query(fd, -1, 0,	AI32SSC_QUERY_MASTER_CLOCK, &master);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_ADC_ENABLE,	&adc_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_BURST_SIZE,	&size);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RAG_ENABLE,	&a_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RAG_NRATE,	&a_nrate);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_ENABLE,	&b_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_NRATE,	&b_nrate);

	for (;;)	// A convenience loop.
	{
		if (adc_enable == AI32SSC_ADC_ENABLE_NO)
		{
			fsamp	= 0;
			sprintf(buf, "ADC disabled");
			break;
		}

		if (a_enable == AI32SSC_GEN_ENABLE_NO)
		{
			fsamp	= 0;
			sprintf(buf, "Rate-A Generator disabled");
			break;
		}

		if (b_enable == AI32SSC_GEN_ENABLE_NO)
		{
			fsamp	= 0;
			sprintf(buf, "Rate-B Generator disabled");
			break;
		}

		sprintf(buf,
				"Fref %ld, A Nrate %ld, B Nrate %ld, Burst Size %ld",
				(long) master,
				(long) a_nrate,
				(long) b_nrate,
				(long) size);

		if (size == 0)
		{
			fsamp	= master / a_nrate;
			break;
		}

		// Sample clocking
		s_ts	= (float) a_nrate / master;
		s_ts_ns	= (long long) (s_ts * 1000000000L + 0.5);
		s_tb_ns	= s_ts_ns * size;

		// Burst Clocking
		b_tb	= (float) b_nrate / master;
		b_tb_ns	= (long long) (b_tb * 1000000000L + 0.5);
		qty		= (long) ((s_tb_ns + s_ts_ns - 1) / b_tb_ns);
		qty		= (qty < 1) ? 1 : qty;
		b_tt_ns	= b_tb_ns * qty;

		// Sample rate.
		bps		= 1.0 / (1.0 * b_tt_ns / 1000000000L);
		fsamp	= (s32) (bps * size);
		break;
	}

	if (fsamp > 200000L)
		strcat(buf, "  <--- UNSTABLE");

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static int _burst_rbg_mc_adc_rbg_mc(int fd, s32* sps, char* buf)
{
	s32			b_enable	= -1;
	s32			b_nrate		= -1;
	float		b_tb;				// Burst: Time Burst
	long long	b_tb_ns;			// Burst: Time Burst Nano Seconds.
	long long	b_tt_ns;			// Burst: Time Total Nano Seconds.
	float		bps;				// Bursts Per Second
	s32			adc_enable	= -1;
	int			errs		= 0;
	s32			fsamp		= 0;
	s32			master;
	long		qty;
	long long	s_tb_ns;			// Sample: Time Burst Nano Seconds.
	float		s_ts;				// Sample: Time Scan
	long long	s_ts_ns;			// Sample: Time Scan Nano Seconds.
	s32			size		= -1;

	errs	+= ai32ssc_query(fd, -1, 0,	AI32SSC_QUERY_MASTER_CLOCK, &master);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_ADC_ENABLE,	&adc_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_BURST_SIZE,	&size);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_ENABLE,	&b_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_NRATE,	&b_nrate);

	for (;;)	// A convenience loop.
	{
		if (adc_enable == AI32SSC_ADC_ENABLE_NO)
		{
			fsamp	= 0;
			sprintf(buf, "ADC disabled");
			break;
		}

		if (b_enable == AI32SSC_GEN_ENABLE_NO)
		{
			fsamp	= 0;
			sprintf(buf, "Rate-B Generator disabled");
			break;
		}

		sprintf(buf,
				"Fref %ld, B Nrate %ld, Burst Size %ld",
				(long) master,
				(long) b_nrate,
				(long) size);

		if (size == 0)
		{
			fsamp	= master / b_nrate;
			break;
		}

		// Sample clocking
		s_ts	= (float) b_nrate / master;
		s_ts_ns	= (long long) (s_ts * 1000000000L + 0.5);
		s_tb_ns	= s_ts_ns * size;

		// Burst Clocking
		b_tb	= (float) b_nrate / master;
		b_tb_ns	= (long long) (b_tb * 1000000000L + 0.5);
		qty		= (long) ((s_tb_ns + s_ts_ns - 1) / b_tb_ns);
		qty		= (qty < 1) ? 1 : qty;
		b_tt_ns	= b_tb_ns * qty;

		// Sample rate.
		bps		= 1.0 / (1.0 * b_tt_ns / 1000000000L);
		fsamp	= (s32) (bps * size);
		break;
	}

	if (fsamp > 200000L)
		strcat(buf, "  <--- UNSTABLE");

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static int _burst_rbg_mc(int fd, s32* sps, char* buf)
{
	s32	adc		= -1;
	int	errs;
	s32	fsamp	= 0;

	errs	= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_ADC_CLK_SRC, &adc);

	if (errs == 0)
	{
		switch (adc)
		{
			default:

				sprintf(buf, "%d. INTERNAL ERROR: %ld", __LINE__, (long) adc);
				break;

			case AI32SSC_ADC_CLK_SRC_BCR:

				errs	+= _adc_bcr(&fsamp, buf);
				break;

			case AI32SSC_ADC_CLK_SRC_EXT:

				errs	+= _adc_ext(&fsamp, buf);
				break;

			case AI32SSC_ADC_CLK_SRC_RAG:

				errs	+= _burst_rbg_mc_adc_rag_mc(fd, &fsamp, buf);
				break;

			case AI32SSC_ADC_CLK_SRC_RBG:

				errs	+= _burst_rbg_mc_adc_rbg_mc(fd, &fsamp, buf);
				break;
		}
	}

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static int _burst_rbg_rag_mc_adc_rag_mc(int fd, s32* sps, char* buf)
{
	s32			a_enable	= -1;
	s32			a_nrate		= -1;
	s32			b_enable	= -1;
	s32			b_nrate		= -1;
	float		b_tb;				// Burst: Time Burst
	long long	b_tb_ns;			// Burst: Time Burst Nano Seconds.
	long long	b_tt_ns;			// Burst: Time Total Nano Seconds.
	float		bps;				// Bursts Per Second
	s32			adc_enable	= -1;
	int			errs		= 0;
	s32			fsamp		= 0;
	s32			master;
	long		qty;
	long long	s_tb_ns;			// Sample: Time Burst Nano Seconds.
	float		s_ts;				// Sample: Time Scan
	long long	s_ts_ns;			// Sample: Time Scan Nano Seconds.
	s32			size		= -1;

	errs	+= ai32ssc_query(fd, -1, 0,	AI32SSC_QUERY_MASTER_CLOCK, &master);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_ADC_ENABLE,	&adc_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_BURST_SIZE,	&size);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RAG_ENABLE,	&a_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RAG_NRATE,	&a_nrate);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_ENABLE,	&b_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_NRATE,	&b_nrate);

	for (;;)	// A convenience loop.
	{
		if (adc_enable == AI32SSC_ADC_ENABLE_NO)
		{
			fsamp	= 0;
			sprintf(buf, "ADC disabled");
			break;
		}

		if (a_enable == AI32SSC_GEN_ENABLE_NO)
		{
			fsamp	= 0;
			sprintf(buf, "Rate-A Generator disabled");
			break;
		}

		if (b_enable == AI32SSC_GEN_ENABLE_NO)
		{
			fsamp	= 0;
			sprintf(buf, "Rate-B Generator disabled");
			break;
		}

		sprintf(buf,
				"Fref %ld, A Nrate %ld, B Nrate %ld, Burst Size %ld",
				(long) master,
				(long) a_nrate,
				(long) b_nrate,
				(long) size);

		if (size == 0)
		{
			fsamp	= master / a_nrate;
			break;
		}

		// Sample clocking
		s_ts	= (float) a_nrate / master;
		s_ts_ns	= (long long) (s_ts * 1000000000L + 0.5);
		s_tb_ns	= s_ts_ns * size;

		// Burst Clocking
		b_tb	= ((float) a_nrate * b_nrate) / master;
		b_tb_ns	= (long long) (b_tb * 1000000000L + 0.5);
		qty		= (long) ((s_tb_ns + s_ts_ns - 1) / b_tb_ns);
		qty		= (qty < 1) ? 1 : qty;
		b_tt_ns	= b_tb_ns * qty;

		// Sample rate.
		bps		= 1.0 / (1.0 * b_tt_ns / 1000000000L);
		fsamp	= (s32) (bps * size);
		break;
	}

	if (fsamp > 200000L)
		strcat(buf, "  <--- UNSTABLE");

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static int _burst_rbg_rag_mc_adc_rbg_rag_mc(int fd, s32* sps, char* buf)
{
	s32			a_enable	= -1;
	s32			a_nrate		= -1;
	s32			b_enable	= -1;
	s32			b_nrate		= -1;
	float		b_tb;				// Burst: Time Burst
	long long	b_tb_ns;			// Burst: Time Burst Nano Seconds.
	long long	b_tt_ns;			// Burst: Time Total Nano Seconds.
	float		bps;				// Bursts Per Second
	s32			adc_enable	= -1;
	int			errs		= 0;
	s32			fsamp		= 0;
	s32			master;
	long		qty;
	long long	s_tb_ns;			// Sample: Time Burst Nano Seconds.
	float		s_ts;				// Sample: Time Scan
	long long	s_ts_ns;			// Sample: Time Scan Nano Seconds.
	s32			size		= -1;

	errs	+= ai32ssc_query(fd, -1, 0,	AI32SSC_QUERY_MASTER_CLOCK, &master);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_ADC_ENABLE,	&adc_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_BURST_SIZE,	&size);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RAG_ENABLE,	&a_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RAG_NRATE,	&a_nrate);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_ENABLE,	&b_enable);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_NRATE,	&b_nrate);

	for (;;)	// A convenience loop.
	{
		if (adc_enable == AI32SSC_ADC_ENABLE_NO)
		{
			fsamp	= 0;
			sprintf(buf, "ADC disabled");
			break;
		}

		if (a_enable == AI32SSC_GEN_ENABLE_NO)
		{
			fsamp	= 0;
			sprintf(buf, "Rate-A Generator disabled");
			break;
		}

		if (b_enable == AI32SSC_GEN_ENABLE_NO)
		{
			fsamp	= 0;
			sprintf(buf, "Rate-B Generator disabled");
			break;
		}

		sprintf(buf,
				"Fref %ld, A Nrate %ld, B Nrate %ld, Burst Size %ld",
				(long) master,
				(long) a_nrate,
				(long) b_nrate,
				(long) size);

		if (size == 0)
		{
			fsamp	= master / a_nrate;
			break;
		}

		// Sample clocking
		s_ts	= ((float) a_nrate * b_nrate) / master;
		s_ts_ns	= (long long) (s_ts * 1000000000L + 0.5);
		s_tb_ns	= s_ts_ns * size;

		// Burst Clocking
		b_tb	= ((float) a_nrate * b_nrate) / master;
		b_tb_ns	= (long long) (b_tb * 1000000000L + 0.5);
		qty		= (long) ((s_tb_ns + s_ts_ns - 1) / b_tb_ns);
		qty		= (qty < 1) ? 1 : qty;
		b_tt_ns	= b_tb_ns * qty;

		// Sample rate.
		bps		= 1.0 / (1.0 * b_tt_ns / 1000000000L);
		fsamp	= (s32) (bps * size);
		break;
	}

	if (fsamp > 200000L)
		strcat(buf, "  <--- UNSTABLE");

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static int _burst_rbg_rag_mc(int fd, s32* sps, char* buf)
{
	s32	adc		= -1;
	int	errs;
	s32	fsamp	= 0;

	errs	= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_ADC_CLK_SRC, &adc);

	if (errs == 0)
	{
		switch (adc)
		{
			default:

				sprintf(buf, "%d. INTERNAL ERROR: %ld", __LINE__, (long) adc);
				break;

			case AI32SSC_ADC_CLK_SRC_BCR:

				errs	+= _adc_bcr(&fsamp, buf);
				break;

			case AI32SSC_ADC_CLK_SRC_EXT:

				errs	+= _adc_ext(&fsamp, buf);
				break;

			case AI32SSC_ADC_CLK_SRC_RAG:

				errs	+= _burst_rbg_rag_mc_adc_rag_mc(fd, &fsamp, buf);
				break;

			case AI32SSC_ADC_CLK_SRC_RBG:

				errs	+= _burst_rbg_rag_mc_adc_rbg_rag_mc(fd, &fsamp, buf);
				break;
		}
	}

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static int _burst_rbg(int fd, s32* sps, char* buf)
{
	int	errs;
	s32	fsamp	= 0;
	s32	rbg		= -1;

	errs	= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RBG_CLK_SRC, &rbg);

	if (errs == 0)
	{
		switch (rbg)
		{
			default:

				sprintf(buf, "%d. INTERNAL ERROR: %ld", __LINE__, (long) rbg);
				break;

			case AI32SSC_RBG_CLK_SRC_MASTER:

				errs	+= _burst_rbg_mc(fd, &fsamp, buf);
				break;

			case AI32SSC_RBG_CLK_SRC_RAG:

				errs	+= _burst_rbg_rag_mc(fd, &fsamp, buf);
				break;
		}
	}

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static void _fsamp_compute_a(fsamp_t* data)
{
	float	delta;
	float	fsamp_got;
	float	fsamp_want	= data->fsamp_want;
	int		init		= 0;	// Have we performed the first run initialization?
	s32		master		= data->master;
	s32		nr;
	s32		nr_max;
	s32		nr_min;
	s32		nrate_a;
	s32		nrate_max	= data->nrate_max;
	s32		nrate_min	= data->nrate_min;

	data->src		= AI32SSC_ADC_CLK_SRC_RAG;
	data->src_b		= AI32SSC_RBG_CLK_SRC_MASTER;
	data->enable_a	= AI32SSC_GEN_ENABLE_YES;
	data->enable_b	= AI32SSC_GEN_ENABLE_NO;

	nr		= master / fsamp_want;
	nr		= (nr < nrate_min) ? nrate_min
			: (nr > nrate_max) ? nrate_max : nr;

	nr_min	= nr - 5;
	nr_max	= nr + 5;
	nr_min	= (nr_min < nrate_min) ? nrate_min
			: (nr_min > nrate_max) ? nrate_max : nr_min;
	nr_max	= (nr_max < nrate_min) ? nrate_min
			: (nr_max > nrate_max) ? nrate_max : nr_max;

	for (nrate_a = nr_min; nrate_a <= nr_max; nrate_a++)
	{
		fsamp_got	= ((float) master) / nrate_a;
		delta		= fsamp_got - fsamp_want;

		if ((init == 0) || ((ABS(delta)) < (ABS(data->delta))))
		{
			init			= 1;
			data->delta		= delta;
			data->fsamp_got	= fsamp_got;
			data->nrate_a	= nrate_a;
		}
	}
}



//*****************************************************************************
static void _fsamp_compute_ab(fsamp_t* data)
{
	float	delta;
	float	fsamp_got;
	float	fsamp_want	= data->fsamp_want;
	int		init		= 0;	// Have we performed the first run initialization?
	s32		master		= data->master;
	s32		nr;
	s32		nr_max;
	s32		nr_min;
	s32		nrate_a;
	s32		nrate_b;
	s32		nrate_max	= data->nrate_max;
	s32		nrate_min	= data->nrate_min;

	data->src		= AI32SSC_ADC_CLK_SRC_RBG;
	data->src_b		= AI32SSC_RBG_CLK_SRC_RAG;
	data->enable_a	= AI32SSC_GEN_ENABLE_YES;
	data->enable_b	= AI32SSC_GEN_ENABLE_YES;

	for (nrate_a = nrate_min; nrate_a <= nrate_max; nrate_a++)
	{
		nr		= master / fsamp_want / nrate_a;
		nr		= (nr < nrate_min) ? nrate_min
				: (nr > nrate_max) ? nrate_max : nr;

		nr_min	= nr - 5;
		nr_max	= nr + 5;
		nr_min	= (nr_min < nrate_min) ? nrate_min
				: (nr_min > nrate_max) ? nrate_max : nr_min;
		nr_max	= (nr_max < nrate_min) ? nrate_min
				: (nr_max > nrate_max) ? nrate_max : nr_max;

		for (nrate_b = nr_min; nrate_b <= nr_max; nrate_b++)
		{
			fsamp_got	= (float) master / ((float) nrate_b * nrate_a);
			delta		= fsamp_got - fsamp_want;

			if ((init == 0) || ((ABS(delta)) < (ABS(data->delta))))
			{
				init			= 1;
				data->delta		= delta;
				data->fsamp_got	= fsamp_got;
				data->nrate_a	= nrate_a;
				data->nrate_b	= nrate_b;
			}
		}
	}
}



//*****************************************************************************
static int _tt_rag(int fd, s32* sps, char* buf)
{
	int	errs		= 0;
	s32	fsamp		= 0;
	s32	master;
	s32	nrate_a		= -1;
	s32	nrate_tt	= -1;
	s32	rag			= -1;

	errs	+= ai32ssc_query(fd, -1, 0,	AI32SSC_QUERY_MASTER_CLOCK, &master);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RAG_ENABLE, &rag);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_RAG_NRATE, &nrate_a);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_TT_NRATE, &nrate_tt);

	if (errs)
	{
	}
	else if (rag == AI32SSC_GEN_ENABLE_NO)
	{
		strcpy(buf, "RAG Disabled");
		fsamp	= 0;
	}
	else
	{
		fsamp	= (s32) (((float) master) / nrate_a / nrate_tt);
		sprintf(buf,
				"Fref %ld, A Nrate %ld, TT Nrate %ld",
				(long) master,
				(long) nrate_a,
				(long) nrate_tt);
	}

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static int _tt(int fd, s32* sps, char* buf)
{
	s32	adc		= -1;
	int	errs	= 0;
	s32	fsamp	= 0;
	s32	src		= -1;

	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_TT_ADC_ENABLE, &adc);
	errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_TT_ADC_CLK_SRC, &src);

	if (errs)
	{
	}
	else if (adc == AI32SSC_TT_ADC_ENABLE_NO)
	{
		strcpy(buf, "TT ADC Disabled");
		fsamp	= 0;
	}
	else if (src == AI32SSC_TT_ADC_CLK_SRC_EXT_SAMP)
	{
		strcpy(buf, "TT External Sample Clock");
		fsamp	= 0;
	}
	else if (src == AI32SSC_TT_ADC_CLK_SRC_EXT_REF)
	{
		strcpy(buf, "TT External Reference Clock");
		fsamp	= 0;
	}
	else if (src == AI32SSC_TT_ADC_CLK_SRC_RAG)
	{
		errs	+= _tt_rag(fd, &fsamp, buf);
	}
	else
	{
		sprintf(buf, "%d. INTERNAL ERROR: %ld", __LINE__, (long) src);
		fsamp	= 0;
	}

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
int	ai32ssc_fsamp_compute(
	int		fd,
	int		index,
	int		verbose,
	s32		fsamp,
	s32*	src,
	s32*	src_b,
	s32*	nrate_a,
	s32*	nrate_b,
	s32*	enable_a,
	s32*	enable_b,
	double*	rate)
{
	fsamp_t	data;
	fsamp_t	data_a;
	fsamp_t	data_ab;
	int		errs		= 0;

	if (verbose)
	{
		gsc_label_index("AI Sample Rate", index);
		gsc_label_long_comma(fsamp);
		printf(" S/S");
	}

	memset(&data, 0, sizeof(data));
	errs	+= ai32ssc_query(fd, -1, 0,	AI32SSC_QUERY_FSAMP_MAX, &data.fsamp_max);
	errs	+= ai32ssc_query(fd, -1, 0,	AI32SSC_QUERY_FSAMP_MIN, &data.fsamp_min);
	errs	+= ai32ssc_query(fd, -1, 0,	AI32SSC_QUERY_MASTER_CLOCK, &data.master);
	errs	+= ai32ssc_query(fd, -1, 0,	AI32SSC_QUERY_NRATE_MAX, &data.nrate_max);
	errs	+= ai32ssc_query(fd, -1, 0,	AI32SSC_QUERY_NRATE_MIN, &data.nrate_min);

	data.delta		= FLT_MAX;
	data.fsamp_want	= LIMIT_RANGE(fsamp, data.fsamp_min, data.fsamp_max);
	data.nrate_a	= data.nrate_max;
	data.nrate_b	= data.nrate_max;

	data_a	= data;
	data_ab	= data;

	_fsamp_compute_a(&data_a);
	_fsamp_compute_ab(&data_ab);

	if ((ABS(data_a.delta)) < (ABS(data_ab.delta)))
	{
		data	= data_a;

		if (verbose)
		{
			printf("%s  (", errs ? " FAIL <---" : "");
			printf("Fref ");
			gsc_label_long_comma((long) data.master);
			printf(", Nrate-A ");
			gsc_label_long_comma((long) data.nrate_a);
			printf(", Fsamp %.3f", data.fsamp_got);
			printf(")\n");
		}
	}
	else
	{
		data	= data_ab;

		if (verbose)
		{
			printf("%s  (", errs ?  " FAIL <---" : "");
			printf("Fref ");
			gsc_label_long_comma((long) data.master);
			printf(", Nrate-A ");
			gsc_label_long_comma((long) data.nrate_a);
			printf(", Nrate-B ");
			gsc_label_long_comma((long) data.nrate_b);
			printf(", Fsamp %.3f", data.fsamp_got);
			printf(")\n");
		}
	}

	if (src)
		src[0]		= data.src;

	if (src_b)
		src_b[0]	= data.src_b;

	if (nrate_a)
		nrate_a[0]	= data.nrate_a;

	if (nrate_b)
		nrate_b[0]	= data.nrate_b;

	if (enable_a)
		enable_a[0]	= data.enable_a;

	if (enable_b)
		enable_b[0]	= data.enable_b;

	if (rate)
		rate[0]	= data.fsamp_got;

	return(errs);
}



/******************************************************************************
*
*	Function:	ai32ssc_fsamp_report
*
*	Purpose:
*
*		Determine and report the sample rate for the specified channel.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		chan	The index of the channel of interest.
*
*		sps		Store the sample rate here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int ai32ssc_fsamp_report(int fd, int index, int verbose, int chan, s32* sps)
{
	char	buf[1024];
	s32		burst		= -1;
	int		errs		= 0;
	s32		fsamp		= 0;
	u32		mask;
	s32		tt;			// Time Tag

	if (chan < 0)
		sprintf(buf, "Sample Rate");
	else
		sprintf(buf, "Channel %d", chan);

	if (verbose)
		gsc_label_index(buf, index);

	if (chan < 0)
		chan	= 0;

	for (;;)
	{
		if ((chan < 0) || (chan > 32))
		{
			errs++;
			printf("FAIL <---  (invalid channel index: %d)\n", chan);
			break;
		}

		errs	+= _channels(fd, &mask, &tt);

		if (errs)
			break;

		if (((1L << chan) & mask) == 0)
		{
			sps[0]	= 0;

			if (verbose)
				printf("SKIPPED  (disabled)\n");

			break;
		}

		if (tt)
		{
			errs	+= _tt(fd, &fsamp, buf);
			break;
		}

		errs	+= ai32ssc_ioctl_util(fd, -1, 0, AI32SSC_IOCTL_BURST_SYNC, &burst);

		if (errs)
			break;

		switch (burst)
		{
			default:

				sprintf(buf, "%d. INTERNAL ERROR: %ld", __LINE__, (long) burst);
				break;

			case AI32SSC_BURST_SYNC_BCR:

				errs	+= _burst_bcr(&fsamp, buf);
				break;

			case AI32SSC_BURST_SYNC_DISABLE:

				errs	+= _burst_disable(fd, &fsamp, buf);
				break;

			case AI32SSC_BURST_SYNC_EXT:

				errs	+= _burst_ext(&fsamp, buf);
				break;

			case AI32SSC_BURST_SYNC_RBG:

				errs	+= _burst_rbg(fd, &fsamp, buf);
				break;
		}

		break;
	}

	if ((errs == 0) && (verbose))
	{
		gsc_label_long_comma((long) fsamp);
		printf(" S/S  (%s)\n", buf);
	}

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



/******************************************************************************
*
*	Function:	ai32ssc_fsamp_report_all
*
*	Purpose:
*
*		Determine and report the sample rate for all channels.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fsamp	Report the overall sample rate here.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int ai32ssc_fsamp_report_all(int fd, int index, int verbose, s32* fsamp)
{
	u32	bit;
	int	chan;
	int	errs;
	u32	mask;
	s32	sps;
	s32	total	= 0;
	s32	tt;

	if (verbose)
		gsc_label_index("Sample Rates", index);

	errs	= _channels(fd, &mask, &tt);

	if (errs == 0)
	{
		if (verbose)
		{
			printf("\n");
			gsc_label_level_inc();
		}

		for (chan = 0, bit = 1; chan < 32; chan++, bit *= 2)
		{
			if (bit & mask)
			{
				sps		= 0;
				errs	+= ai32ssc_fsamp_report(fd, index, verbose, chan, &sps);
				total	+= sps;
			}
		}

		if (verbose)
		{
			gsc_label("Overall Rate");
			gsc_label_long_comma(total);
			printf(" S/S\n");
			gsc_label_level_dec();
		}
	}

	if (fsamp)
		fsamp[0]	= total;

	return(errs);
}


