// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/utils/util_config_tt.c $
// $Rev: 53690 $
// $Date: 2023-09-07 13:47:54 -0500 (Thu, 07 Sep 2023) $

// 16AI32SSC: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	ai32ssc_config_tt
*
*	Purpose:
*
*		Configure the Time Tag services.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		set		The setting to apply.
*
*		get		The current setting is recorded here, if not NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int	ai32ssc_config_tt(int fd, int index, int verbose, s32 nrate_a, s32 nrate_tt)
{
	int	chan;
	int	errs	= 0;
	u32	mask;
	s32	qty;
	s32	tt;

	if (verbose)
		gsc_label_index("Time Tag Configuration", index);

	errs	+= ai32ssc_query(fd, index, verbose, AI32SSC_QUERY_TIME_TAG, &tt);

	if (errs)
	{
	}
	else if (tt == 0)
	{
		if (verbose)
			printf("SKIPPED  (Not supported on this device.)\n");
	}
	else
	{
		if (verbose)
		{
			printf("\n");
			gsc_label_level_inc();
		}

		errs	+= ai32ssc_rx_io_mode		(fd, index, verbose, GSC_IO_MODE_PIO,				NULL);
		errs	+= ai32ssc_rx_io_overflow	(fd, index, verbose, AI32SSC_IO_OVERFLOW_CHECK,		NULL);
		errs	+= ai32ssc_rx_io_timeout	(fd, index, verbose, AI32SSC_IO_TIMEOUT_DEFAULT,	NULL);
		errs	+= ai32ssc_rx_io_underflow	(fd, index, verbose, AI32SSC_IO_UNDERFLOW_CHECK,	NULL);

		errs	+= ai32ssc_initialize		(fd, index, verbose);

		// Settings must be applied after the initialization call!

		errs	+= ai32ssc_ain_buf_thr_lvl	(fd, index, verbose, 250000,						NULL);
		errs	+= ai32ssc_ain_mode			(fd, index, verbose, AI32SSC_AIN_MODE_DIFF,			NULL);
		errs	+= ai32ssc_ain_range		(fd, index, verbose, AI32SSC_AIN_RANGE_10V,			NULL);
		errs	+= ai32ssc_data_format		(fd, index, verbose, AI32SSC_DATA_FORMAT_OFF_BIN,	NULL);
		errs	+= ai32ssc_io_inv			(fd, index, verbose, AI32SSC_IO_INV_HIGH,			NULL);
		errs	+= ai32ssc_rag_nrate		(fd, index, verbose, nrate_a,						NULL);
		errs	+= ai32ssc_rag_enable		(fd, index, verbose, AI32SSC_GEN_ENABLE_YES,		NULL);

		errs	+= ai32ssc_tt_reset			(fd, index, verbose);
		errs	+= ai32ssc_tt_adc_clk_src	(fd, index, verbose, AI32SSC_TT_ADC_CLK_SRC_RAG,	NULL);
		errs	+= ai32ssc_tt_adc_enable	(fd, index, verbose, AI32SSC_TT_ADC_ENABLE_NO,		NULL);
		errs	+= ai32ssc_tt_enable		(fd, index, verbose, AI32SSC_TT_ENABLE_NO,			NULL);
		errs	+= ai32ssc_tt_log_mode		(fd, index, verbose, AI32SSC_TT_LOG_MODE_TRIG,		NULL);
		errs	+= ai32ssc_tt_nrate			(fd, index, verbose, nrate_tt,						NULL);
		errs	+= ai32ssc_tt_ref_clk_src	(fd, index, verbose, AI32SSC_TT_REF_CLK_SRC_INT,	NULL);
		errs	+= ai32ssc_tt_ref_val_mode	(fd, index, verbose, AI32SSC_TT_REF_VAL_MODE_AUTO,	NULL);
		errs	+= ai32ssc_tt_reset_ext		(fd, index, verbose, AI32SSC_TT_RESET_EXT_DISABLE,	NULL);
		errs	+= ai32ssc_tt_tagging		(fd, index, verbose, AI32SSC_TT_TAGGING_ENABLE,		NULL);
		errs	+= ai32ssc_tt_trig_mode		(fd, index, verbose, AI32SSC_TT_TRIG_MODE_CONTIN,	NULL);

		errs	+= ai32ssc_query			(fd, index, verbose, AI32SSC_QUERY_CHANNEL_QTY, &qty);
		mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, qty - 1, 0);
		errs	+= ai32ssc_tt_chan_mask		(fd, index, verbose, mask,							NULL);

		for (chan = 0; chan < qty; chan++)
		{
			errs	+= ai32ssc_tt_ref_xx(fd, index, verbose, chan, 0x8000,						NULL);
			errs	+= ai32ssc_tt_thr_xx(fd, index, verbose, chan, 0x4000,						NULL);
		}

		// Settings affecting the sample rate or voltage range must generally
		// be applied before autocalibration.
		// Refer to the board user manual for clarification.
		errs	+= ai32ssc_autocal			(fd, index, verbose);

		// Interrupt selections should be applied after autocalibration as the
		// driver overwrites the current selection to detect the
		// Autocalibration Done interrupt. If an application intends to detect
		// the Autocalibration Done interrupt, then the interrupt selection may
		// be done beforehand.
		errs	+= ai32ssc_irq0_sel			(fd, index, verbose, AI32SSC_IRQ0_INIT_DONE,		NULL);
		errs	+= ai32ssc_irq1_sel			(fd, index, verbose, AI32SSC_IRQ1_NONE,				NULL);

		// The final step is to clear the buffer. This service clears the FIFO
		// and resets the overflow and underflow status bits.
		errs	+= ai32ssc_ain_buf_clear	(fd, index, verbose, 1);

		if (verbose)
			gsc_label_level_dec();
	}

	return(errs);
}


