// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/utils/util_config_ai.c $
// $Rev: 53690 $
// $Date: 2023-09-07 13:47:54 -0500 (Thu, 07 Sep 2023) $

// 16AI32SSC: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	ai32ssc_config_ai
*
*	Purpose:
*
*		Configure the given device's analog input options.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fsamp	This is the desired Fsamp rate, or -1 to use the default.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int	ai32ssc_config_ai(int fd, int index, int verbose, s32 fsamp)
{
	s32		enable_a;
	s32		enable_b;
	int		errs		= 0;
	s32		qty			= 32;
	s32		nrate_a;
	s32		nrate_b;
	s32		src;
	s32		src_b;

	if (verbose)
	{
		gsc_label_index("Input Configuration", index);
		printf("\n");
		gsc_label_level_inc();
	}

	errs	+= ai32ssc_initialize		(fd, index, verbose);

	// Settings must be applied after the initialization call!

	errs	+= ai32ssc_rx_io_mode		(fd, index, verbose, GSC_IO_MODE_PIO,					NULL);
	errs	+= ai32ssc_rx_io_overflow	(fd, index, verbose, AI32SSC_IO_OVERFLOW_CHECK,			NULL);
	errs	+= ai32ssc_rx_io_timeout	(fd, index, verbose, AI32SSC_IO_TIMEOUT_DEFAULT,		NULL);
	errs	+= ai32ssc_rx_io_underflow	(fd, index, verbose, AI32SSC_IO_UNDERFLOW_CHECK,		NULL);
	errs	+= ai32ssc_ain_buf_thr_lvl	(fd, index, verbose, 250000,							NULL);
	errs	+= ai32ssc_ain_mode			(fd, index, verbose, AI32SSC_AIN_MODE_DIFF,				NULL);
	errs	+= ai32ssc_ain_range		(fd, index, verbose, AI32SSC_AIN_RANGE_10V,				NULL);
	errs	+= ai32ssc_burst_size		(fd, index, verbose, 1,									NULL);
	errs	+= ai32ssc_burst_sync		(fd, index, verbose, AI32SSC_BURST_SYNC_DISABLE,		NULL);
	errs	+= ai32ssc_chan_active		(fd, index, verbose, AI32SSC_CHAN_ACTIVE_RANGE,			NULL);
	errs	+= ai32ssc_data_format		(fd, index, verbose, AI32SSC_DATA_FORMAT_OFF_BIN,		NULL);
	errs	+= ai32ssc_data_packing		(fd, index, verbose, AI32SSC_DATA_PACKING_DISABLE,		NULL);
	errs	+= ai32ssc_io_inv			(fd, index, verbose, AI32SSC_IO_INV_HIGH,				NULL);

	errs	+= ai32ssc_query			(fd, index, verbose, AI32SSC_QUERY_CHANNEL_QTY, &qty);
	errs	+= ai32ssc_chan_first		(fd, index, verbose, 0,									NULL);
	errs	+= ai32ssc_chan_last		(fd, index, verbose, qty - 1,							NULL);
	errs	+= ai32ssc_chan_single		(fd, index, verbose, 0,									NULL);

	errs	+= ai32ssc_fsamp_compute	(fd, index, verbose, fsamp, &src, &src_b, &nrate_a, &nrate_b, &enable_a, &enable_b, NULL);
	errs	+= ai32ssc_adc_clk_src		(fd, index, verbose, src,								NULL);
	errs	+= ai32ssc_rag_nrate		(fd, index, verbose, nrate_a,							NULL);
	errs	+= ai32ssc_rag_enable		(fd, index, verbose, enable_a,							NULL);
	errs	+= ai32ssc_rbg_nrate		(fd, index, verbose, nrate_b,							NULL);
	errs	+= ai32ssc_rbg_clk_src		(fd, index, verbose, src_b,								NULL);
	errs	+= ai32ssc_rbg_enable		(fd, index, verbose, enable_b,							NULL);

	errs	+= ai32ssc_rbg_sync_output	(fd, index, verbose, AI32SSC_RBG_SYNC_OUTPUT_DISABLE,	NULL);
	errs	+= ai32ssc_scan_marker		(fd, index, verbose, AI32SSC_SCAN_MARKER_DISABLE,		NULL);
	errs	+= ai32ssc_adc_enable		(fd, index, verbose, AI32SSC_ADC_ENABLE_YES,			NULL);

	errs	+= ai32ssc_fsamp_report_all	(fd, index, verbose,									NULL);

	// Settings affecting the sample rate or voltage range must generally
	// be applied before autocalibration.
	// Refer to the board user manual for clarification.
	errs	+= ai32ssc_autocal			(fd, index, verbose);	// Cable may need disconnecting for this.

	// Interrupt selections should be applied after autocalibration as the
	// driver overwrites the current selection to detect the Autocalibration
	// Done interrupt. If an application intends to detect the Autocalibration
	// Done interrupt, then the interrupt selection may be made beforehand.
	errs	+= ai32ssc_irq0_sel			(fd, index, verbose, AI32SSC_IRQ0_INIT_DONE,			NULL);
	errs	+= ai32ssc_irq1_sel			(fd, index, verbose, AI32SSC_IRQ1_NONE,					NULL);

	// The final step is to clear the buffer. This service clears the FIFO and
	// resets the overflow and underflow status bits.
	errs	+= ai32ssc_ain_buf_clear	(fd, index, verbose, 1);

	if (verbose)
		gsc_label_level_dec();

	return(errs);
}


