// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/samples/snapshot/wait_read.c $
// $Rev: 54732 $
// $Date: 2024-07-01 08:13:38 -0500 (Mon, 01 Jul 2024) $

// 16AI32SSC: Sample Application: source file

#include "main.h"



//*****************************************************************************
// Wait for the Sync Done interrupt, then read data from the FIFO. If the
// volume of data obtained exceeds a single scan, then clear the FIFO. Data
// appears in the FIFO one scan at a time. This is a very efficient method.
// This method requires an I/O Timeout setting of zero, which guarantees that
// the read call does not block.
//*****************************************************************************
int rx_wait_read(const args_t* args, rx_data_t* rx, gsc_buf_man_t* bm)
{
	s32			arg;
	int			ms;
	int			ret			= 0;
	int			rx_bytes	= 4 * rx->chan_qty;
	int			scan_size	= 4 * rx->chan_qty;
	int			tmp;
	gsc_wait_t	wait;

	if (args->data_pack)
	{
		rx_bytes	= (scan_size / 2) + ((rx->chan_qty & 1) ? 2 : 0);
		scan_size	= scan_size / 2;
	}

	for (;;)	// A convenience loop.
	{
		// Clear the buffer in case the sample rate is too fast.
		tmp	= ai32ssc_ioctl(args->fd, AI32SSC_IOCTL_AIN_BUF_CLEAR, NULL);

		if (tmp)
		{
			ret			= tmp;
			bm->eof		= 1;
			bm->count	= 0;
			rx->errs	= 1;
			sprintf(rx->err_buf, "AI Buffer Clear failure: %d", (int) tmp);
			break;
		}

		// Enable the SYNC Done interrupt.
		arg	= AI32SSC_IRQ0_SYNC_DONE;
		ret	= ai32ssc_ioctl(args->fd, AI32SSC_IOCTL_IRQ0_SEL, &arg);

		if (ret < 0)
		{
			bm->eof		= 1;
			bm->count	= 0;
			rx->errs	= 1;
			sprintf(rx->err_buf, "IRQ0 enable failure: %d", ret);
			break;
		}

		// Compute the Wait Event timeout as about 2ms longer than needed.
		ms	= (int) (1000.0 / rx->fsamp + 2.5);	// round up

		// Wait for the SYNC Done interrupt.
		wait.flags		= 0;
		wait.main		= 0;
		wait.gsc		= AI32SSC_WAIT_GSC_SYNC_DONE;
		wait.alt		= 0;
		wait.io			= 0;
		wait.timeout_ms	= ms;
		wait.count		= 0;
		ret	= ai32ssc_ioctl(args->fd, AI32SSC_IOCTL_WAIT_EVENT, &wait);

		if (ret < 0)
		{
			bm->eof		= 1;
			bm->count	= 0;
			rx->errs	= 1;
			sprintf(rx->err_buf, "Wait Event failure: %d", ret);
			break;
		}

		if (wait.flags & GSC_WAIT_FLAG_DONE)
		{
			rx->wait_done++;
		}
		else if (wait.flags & GSC_WAIT_FLAG_TIMEOUT)
		{
			rx->wait_timeout++;
		}

		// Read in the data. ==============================
		rx->read_calls++;
		ret	= ai32ssc_read(args->fd, bm->buffer, rx_bytes);

		if (ret < 0)
		{
			bm->eof		= 1;
			bm->count	= 0;
			rx->errs	= 1;
			sprintf(rx->err_buf, "Rx read failure: %d", (int) ret);
			break;
		}

		if (ret < scan_size)
		{
			bm->eof		= 1;
			bm->count	= 0;
			rx->errs	= 1;
			sprintf(rx->err_buf, "Rx insufficient read bytes: %d", (int) ret);
			break;
		}

		ret	= scan_size;
		break;
	}

	return(ret);
}


