// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/samples/snapshot/tx.c $
// $Rev: 54732 $
// $Date: 2024-07-01 08:13:38 -0500 (Mon, 01 Jul 2024) $

// 16AI32SSC: Sample Application: source file

#include "main.h"



//*****************************************************************************
static void _wait_til_done(const tx_data_t* tx)
{
	// Wait until the processing thread indicates that it is done.

	for (;;)
	{
		if (tx->done)
			break;

		gsc_time_sleep_ms(1);
	}
}



//*****************************************************************************
static void _wait_til_rx_reading(const args_t* args)
{
	const rx_data_t*	rx	= args->rx;
	const tx_data_t*	tx	= args->tx;

	// Wait until the Rx thread begins reading data.

	for (;;)
	{
		if ((rx->started == 0) || (rx->reading) || (tx->stop))
			break;

		gsc_time_sleep_ms(1);
	}
}



//*****************************************************************************
static void _wait_til_started(const tx_data_t* tx)
{
	// Wait until the processing thread indicates that it has started.

	for (;;)
	{
		if ((tx->started) || (tx->stop))
			break;

		gsc_time_sleep_ms(1);
	}
}



//*****************************************************************************
static void _wait_for_start(const tx_data_t* tx)
{
	// Wait until the controlling thread tells us to start working.

	for (;;)
	{
		if ((tx->start) || (tx->stop))
			break;

		gsc_time_sleep_ms(1);
	}
}



//*****************************************************************************
static void _get_file_pointer(const args_t* args, const char* extension)
{
	tx_data_t*	tx	= args->tx;

	if (tx->file == NULL)
	{
		strcpy(tx->name, "datast.");
		strcat(tx->name, extension);

		tx->file	= fopen(tx->name, "w+b");

		if (tx->file == NULL)
		{
			tx->errs	= 1;
			sprintf(tx->err_buf, "File createion failure: %s", tx->name);
		}
	}
}



//*****************************************************************************
static void _write_file_bin(const args_t* args, gsc_buf_man_t* bm)
{
	int			ret;
	tx_data_t*	tx	= args->tx;

	for (;;)	// A convenience loop.
	{
		_get_file_pointer(args, "bin");

		if (tx->errs)
			break;

		ret	= fwrite((char*) bm->buffer + bm->offset, bm->count, 1, tx->file);

		if (ret < 0)
		{
			bm->count	= 0;
			tx->errs	= 1;
			sprintf(tx->err_buf, "Tx binary write failure: %d", (int) ret);
		}
		else
		{
			tx->total_bytes	+= bm->count;
			bm->count		= 0;
			bm->offset		= 0;
		}

		break;
	}
}



//*****************************************************************************
static void _write_file_text(const args_t* args, gsc_buf_man_t* bm)
{
	char		buf[32];
	u32			chan		= 0;
	int			i;
	int			len;
	char*		prefix;
	u8*			ptr;
	int			remainder;
	int			ret			= 0;
	int			samples;
	u32*		src;
	tx_data_t*	tx			= args->tx;
	u32			value;

	for (;;)	// A convenience loop.
	{
		_get_file_pointer(args, "txt");

		if (tx->errs)
			break;

		samples		= bm->count / 4;
		remainder	= bm->count % 4;
		src			= (void*) ((char*) bm->buffer + bm->offset);

		for (i = 0; i < samples; i++, src++)
		{
			value	= src[0];
			chan	= (value & 0x80000000) ? 1 : chan + 1;

			if (args->tx_chan_tag == CHAN_TAG_EXCLUDE)
				value	&= 0xFFFF;
			else if (args->tx_chan_tag == CHAN_TAG_ONLY)
				value	&= 0x80000000;

			if (i == 0)
				prefix	= "";
			else
				prefix	= chan ? "  " : "\r\n";

			if (args->tx_decimal)
				sprintf(buf, "%s% 9ld", prefix, (long) value);
			else
				sprintf(buf, "%s%08lX", prefix, (long) value);

			len	= strlen(buf);
			ret	= fwrite(buf, len, 1, tx->file);

			if (ret < 0)
			{
				bm->count	= 0;
				tx->errs	= 1;
				sprintf(tx->err_buf, "Tx text write failure: %d", (int) ret);
				break;
			}
		}

		if (tx->errs)
			break;

		if (remainder)
		{
			buf[0]	= 0;
			ptr		= (u8*) src;

			if (samples)
				strcat(buf, "  ");

			sprintf(buf + strlen(buf), "%02lX", (long) (ptr[0] & 0xFF));

			if (remainder >= 2)
			{
				strcat(buf, "  ");
				sprintf(buf + strlen(buf), "%02lX", (long) (ptr[1] & 0xFF));
			}

			if (remainder >= 3)
			{
				strcat(buf, "  ");
				sprintf(buf + strlen(buf), "%02lX", (long) (ptr[2] & 0xFF));
			}

			strcat(buf, "\r\n");
			len	= strlen(buf);
			ret	= fwrite(buf, len, 1, tx->file);

			if (ret < 0)
			{
				bm->count	= 0;
				tx->errs	= 1;
				sprintf(tx->err_buf, "Tx text write failure: %d", (int) ret);
			}
		}

		if (ret >= 0)
		{
			tx->total_bytes	+= bm->count;
			bm->count		= 0;
			bm->offset		= 0;
		}

		break;
	}
}



//*****************************************************************************
static void _tx_process(const args_t* args)
{
	gsc_buf_man_t	bm;
	int				eof;
	int				errs	= 0;
	long			ms;
	long long		ns;
	rx_data_t*		rx		= args->rx;
	tx_data_t*		tx		= args->tx;
	long			us;

	memset(&bm, 0, sizeof(gsc_buf_man_t));

	_wait_til_rx_reading(args);
	os_time_get_ns(&tx->time_start);
	tx->total_ms	= 0;

	for (; (rx->started) && (errs == 0);)
	{
		// Request an data buffer so we can process it.
		errs	= gsc_buf_man_request_data(&bm);

		if (tx->stop)
		{
			gsc_buf_man_release_buffer(&bm);
			break;
		}

		// Perform validation.

		if (errs)
		{
			tx->errs	= 1;
			strcpy(tx->err_buf, "Tx data request: failed");
			gsc_buf_man_release_buffer(&bm);
			break;
		}

		if ((bm.buffer == NULL) && (bm.eof == 0))
		{
			tx->errs	= 1;
			strcpy(tx->err_buf, "Tx data request: NULL buffer");
			gsc_buf_man_release_buffer(&bm);
			break;
		}

		if ((bm.size == 0) && (bm.eof == 0))
		{
			tx->errs	= 1;
			strcpy(tx->err_buf, "Tx data request: zero sized buffer");
			gsc_buf_man_release_buffer(&bm);
			break;
		}

		switch (args->tx_option)
		{
			default:

				tx->errs	= 1;
				bm.count	= 0;
				errs		= ERROR_INTERNAL_INT(args->tx_option);
				break;

			case TX_OPTION_BIT_BUCKET:

				tx->total_bytes	+= bm.count;
				bm.count		= 0;
				break;

			case TX_OPTION_WRITE_FILE_BIN:

				_write_file_bin(args, &bm);
				break;

			case TX_OPTION_WRITE_FILE_TEXT:

				_write_file_text(args, &bm);
				break;
		}

		eof		= bm.eof;
		bm.eof	= 0;
		errs	= gsc_buf_man_release_buffer(&bm);

		if ((tx->errs == 0) && (errs))
		{
			tx->errs	= 1;
			strcpy(tx->err_buf, "Tx buffer release failure");
		}

		if ((eof) || (tx->errs) || (tx->stop))
			break;
	}

	os_time_get_ns(&tx->time_stop);
	ns	= ((long long) tx->time_stop.tv_sec * 1000000000)
		+ ((long long) tx->time_stop.tv_nsec)
		- ((long long) tx->time_start.tv_sec * 1000000000)
		- ((long long) tx->time_start.tv_nsec);
	us	= (ns + 500) / 1000;
	ms	= (us + 500) / 1000;
	tx->total_ms	= ms;
}



//*****************************************************************************
static int _tx_thread(void* arg)
{
	const args_t*	args	= arg;
	tx_data_t*		tx		= args->tx;

	// Tell the controlling code that this thread has started.
	tx->started	= 1;

	// Wait till we're told to start.
	_wait_for_start(tx);

	// Perform the expected activity.
	_tx_process(args);

	// Tell the controlling code that we're done.
	tx->done	= 1;
	return(0);
}



//*****************************************************************************
int tx_start(const args_t* args)
{
	int				errs	= 0;
	const char*		ptr		= NULL;
	tx_data_t*		tx		= args->tx;

	if (args->range)
	{
		printf(", Tx Startup");
		fflush(stdout);
	}
	else
	{
		gsc_label("Tx Startup");
	}

	switch (args->tx_option)
	{
		default:

			errs	= ERROR_INTERNAL_INT(args->tx_option);
			break;

		case TX_OPTION_BIT_BUCKET:

			ptr		= "discarding data";
			break;

		case TX_OPTION_WRITE_FILE_BIN:

			ptr	= "writing to binary file";
			break;

		case TX_OPTION_WRITE_FILE_TEXT:

			ptr	= "writing to text file";
			break;
	}

	if (errs == 0)
	{
		errs	= os_thread_create(&tx->thread, "Tx Thread", _tx_thread, (void*) args);

		if (errs)
			ptr	= "creating Tx thread";
	}

	if (errs == 0)
		_wait_til_started(tx);

	if (args->range)
	{
		if (errs)
			printf(" FAIL <---");
	}
	else
	{
		printf("%s  (%s)\n", errs ? "FAIL <---" : "PASS", ptr);
	}

	return(errs);
}



//*****************************************************************************
int tx_stop(const args_t* args)
{
	int			errs	= 0;
	int			samples;
	tx_data_t*	tx		= args->tx;

	// STOP ===============================================
	perform_text(args, 1, "Tx Stop", ", Tx Stop");
	tx->stop	= 1;

	if (tx->started)
		_wait_til_done(tx);

	if (tx->file)
	{
		fclose(tx->file);
		tx->file	= NULL;
	}

	os_thread_destroy(&tx->thread);
	perform_text(args, 0, "Done\n", NULL);

	gsc_label_level_inc();

	// STATUS =============================================
	perform_text(args, 1, "Status", NULL);

	if (tx->errs == 0)
	{
		errs	= 0;
		perform_text(args, 0, "PASS\n", NULL);
	}
	else
	{
		errs	= 1;

		if (args->range == 0)
		{
			printf("FAIL <---");

			if (tx->err_buf[0])
				printf(" (%s)", tx->err_buf);

			printf("\n");
		}
	}

	// THROUGHPUT =========================================

	if (args->rx_data == RX_DATA_WAIT_LL)
		samples	= tx->total_bytes / 4;
	else if (args->data_pack)
		samples	= tx->total_bytes / 2;
	else
		samples	= tx->total_bytes / 4;

	if (tx->total_ms)
		tx->overall_rate	= (double) (1000LL * samples) / tx->total_ms;

	if (args->range == 0)
	{
		gsc_label("Throughput");
		gsc_label_long_comma(samples);
		printf(" samples, ");
		gsc_label_float_comma((double) tx->total_ms / 1000, 0, 3);
		printf(" Secs");

		if (tx->total_ms)
		{
			printf(", ");
			gsc_label_float_comma(tx->overall_rate, 0, 3);
			printf(" S/S");
		}

		printf("\n");
	}

	gsc_label_level_dec();
	return(errs);
}


