// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/samples/snapshot/rx.c $
// $Rev: 54746 $
// $Date: 2024-07-01 09:32:04 -0500 (Mon, 01 Jul 2024) $

// 16AI32SSC: Sample Application: source file

#include "main.h"



// variables ******************************************************************

static	os_time_ns_t	_start_time;
static	int				_start_time_set	= 0;



//*****************************************************************************
static void _wait_til_done(const rx_data_t* rx)
{
	// Wait until the processing thread indicates that it is done.

	for (;;)
	{
		if (rx->done)
			break;

		gsc_time_sleep_ms(1);
	}
}



//*****************************************************************************
static void _wait_til_started(const rx_data_t* rx)
{
	// Wait until the processing thread indicates that it has started.

	for (;;)
	{
		if ((rx->started) || (rx->stop))
			break;

		gsc_time_sleep_ms(1);
	}
}



//*****************************************************************************
static void _wait_for_start(const rx_data_t* rx)
{
	// Wait until the controlling thread tells us to start working.

	for (;;)
	{
		if ((rx->start) || (rx->stop))
			break;

		gsc_time_sleep_ms(1);
	}
}



//*****************************************************************************
// Clear the input buffer, then read in data til good data appears.
static int _flush_input_buffer(const args_t* args, rx_data_t* rx, gsc_buf_man_t* bm)
{
	u32	buffer[128 * 1024];
	int	errs;
	int	i;
	int	ret;
	int	tmp;

	for (;;)	// A convenience loop.
	{
		errs	= ai32ssc_rx_io_timeout(args->fd, -1, 0, 5, NULL);

		if (errs)
		{
			bm->eof		= 1;
			rx->errs	= 1;
			sprintf(rx->err_buf, "%d. Adjusting Rx I/O Timeout: failed", __LINE__);
			break;
		}

		// DISABLE ========================================
		errs	= ai32ssc_adc_enable(args->fd, -1, 0, AI32SSC_ADC_ENABLE_NO, NULL);

		if (errs)
		{
			bm->eof		= 1;
			rx->errs	= 1;
			sprintf(rx->err_buf, "%d. Input Buffer disable: failed", __LINE__);
			break;
		}

		// CLEAR ==========================================
		errs	= ai32ssc_ain_buf_clear(args->fd, -1, 0, 0);

		if (errs)
		{
			bm->eof		= 1;
			rx->errs	= 1;
			sprintf(rx->err_buf, "%d. Input Buffer clear: failed", __LINE__);
			break;
		}

		// ENABLE =========================================
		errs	= ai32ssc_adc_enable(args->fd, -1, 0, AI32SSC_ADC_ENABLE_YES, NULL);

		if (errs)
		{
			bm->eof		= 1;
			rx->errs	= 1;
			sprintf(rx->err_buf, "%d. Input Buffer enable: failed", __LINE__);
			break;
		}

		for (i = 0;; i++)
		{
			// READ ===========================================
			ret	= ai32ssc_read(args->fd, buffer, sizeof(buffer));

			if (ret < 0)
			{
				errs		= 1;
				bm->eof		= 1;
				rx->errs	= 1;
				sprintf(rx->err_buf, "%d. Rx read failure: %d", __LINE__, (int) ret);
				break;
			}

			// DISPLAY ========================================

			if (ret < 8)	// 2 samples at 4 bytes per sample
			{
				errs		= 1;
				bm->eof		= 1;
				rx->errs	= 1;
				sprintf(rx->err_buf, "%d. Insufficient data read: %d bytes", __LINE__, (int) ret);
				break;
			}

			// TEST ===========================================

			if ((buffer[0] & 0xFFFF) == (buffer[1] & 0xFFFF))
			{
				if (i >= 10)
				{
					errs		= 1;
					bm->eof		= 1;
					rx->errs	= 1;
					sprintf(rx->err_buf, "%d. Input Buffer flush: failed", __LINE__);
					break;
				}

				continue;
			}

			break;
		}

		tmp		= ai32ssc_rx_io_timeout(args->fd, -1, 0, 0, NULL);

		if (tmp)
		{
			errs		= 1;
			bm->eof		= 1;
			rx->errs	= 1;
			sprintf(rx->err_buf, "%d. Restoring Rx I/O Timeout: failed", __LINE__);
			break;
		}


		break;
	}

	return(errs);
}



//*****************************************************************************
static void _check_eof(const args_t* args, rx_data_t* rx, gsc_buf_man_t* bm)
{
	os_time_ns_t	now;

	os_time_get_ns(&now);
	bm->eof	= 1;

	if (now.tv_sec < rx->time_limit.tv_sec)
	{
		bm->eof	= 0;
	}
	else if (	(now.tv_sec  == rx->time_limit.tv_sec) &&
				(now.tv_nsec <  rx->time_limit.tv_nsec))
	{
		bm->eof	= 0;
	}

	if ((args->rx_mb) && (rx->total_bytes < rx->byte_limit))
		bm->eof	= 0;
}



//*****************************************************************************
static int _rx_sync(const args_t* args, rx_data_t* rx, gsc_buf_man_t* bm)
{
	int	errs;
	s32	get;
	int	i;

	for (;;)	// A convenience loop.
	{
		// CLEAR ==========================================
		errs	= ai32ssc_ain_buf_clear(args->fd, -1, 0, 0);

		if (errs)
		{
			bm->eof		= 1;
			rx->errs	= 1;
			sprintf(rx->err_buf, "%d. Input Buffer clear: failed", __LINE__);
			break;
		}

		// WAIT FOR DATA TO APPEAR ========================

		for (i = 0;; i++)
		{
			errs	= ai32ssc_ain_buf_level(args->fd, -1, 0, &get);

			if (errs)
			{
				bm->eof		= 1;
				rx->errs	= 1;
				sprintf(rx->err_buf, "%d. Input Buffer fill level: failed", __LINE__);
				break;
			}

			if (i > 1000000)
			{
				errs		= 1;
				bm->eof		= 1;
				rx->errs	= 1;
				sprintf(rx->err_buf, "%d. Input Buffer fill level: data took too long", __LINE__);
				break;
			}

			if (get)
				break;
		}

		if (errs)
			break;

		// CLEAR ==========================================
		errs	= ai32ssc_ain_buf_clear(args->fd, -1, 0, 0);

		if (errs)
		{
			bm->eof		= 1;
			rx->errs	= 1;
			sprintf(rx->err_buf, "%d. Input Buffer clear: failed", __LINE__);
			break;
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _zero_data_read(const args_t* args, rx_data_t* rx, gsc_buf_man_t* bm)
{
	if (args) { }	// Silence lint

	memset(bm->buffer, 0, bm->size);
	bm->offset		= 0;
	bm->count		= bm->size - (bm->size % 4);
	rx->total_bytes	+= bm->count;
	return(0);
}



//*****************************************************************************
static int _func_select(
	const args_t*	args,
	rx_data_t*		rx,
	gsc_buf_man_t*	bm,
	int				(**data_sync)(const args_t* args, rx_data_t* rx, gsc_buf_man_t* bm),
	int				(**data_read)(const args_t* args, rx_data_t* rx, gsc_buf_man_t* bm))
{
	int	errs	= 0;

	switch (args->rx_option)
	{
		default:

			bm->eof		= 1;
			rx->errs	= 1;
			errs		= ERROR_INTERNAL_INT(args->rx_option);
			break;

		case RX_OPTION_ZERO_DATA:

			data_sync[0]	= NULL;
			data_read[0]	= _zero_data_read;
			break;

		case RX_OPTION_READ_DEV:

			switch (args->rx_data)
			{
				default:

					bm->eof		= 1;
					rx->errs	= 1;
					errs		= ERROR_INTERNAL_INT(args->rx_data);
					break;

				case RX_DATA_POLL_FIFO:

					data_sync[0]	= _rx_sync;
					data_read[0]	= rx_poll_fifo;
					break;

				case RX_DATA_POLL_READ:

					data_sync[0]	= _rx_sync;
					data_read[0]	= rx_poll_read;
					break;

				case RX_DATA_WAIT_LL:

					data_sync[0]	= _rx_sync;
					data_read[0]	= rx_wait_ll;
					break;

				case RX_DATA_WAIT_READ:

					data_sync[0]	= _rx_sync;
					data_read[0]	= rx_wait_read;
					break;
			}

			break;
	}

	return(errs);
}



//*****************************************************************************
static int _get_rx_buffer(rx_data_t* rx, gsc_buf_man_t* bm)
{
	int	errs;

	for (;;)	// A convenience loop.
	{
		// Request an empty buffer so we can fill it.
		errs	= gsc_buf_man_request_empty(bm);

		if (errs)
		{
			bm->eof		= 1;
			rx->errs	= 1;
			sprintf(rx->err_buf, "%d. Rx empty request: failed", __LINE__);
			gsc_buf_man_release_buffer(bm);
			break;
		}

		if (rx->stop)
		{
			bm->eof	= 1;
			gsc_buf_man_release_buffer(bm);
			break;
		}

		if (bm->buffer == NULL)
		{
			errs		= 1;
			bm->eof		= 1;
			rx->errs	= 1;
			sprintf(rx->err_buf, "%d. Rx empty request: NULL buffer", __LINE__);
			gsc_buf_man_release_buffer(bm);
			break;
		}

		if (bm->size == 0)
		{
			errs		= 1;
			bm->eof		= 1;
			rx->errs	= 1;
			sprintf(rx->err_buf, "%d. Rx empty request: zero sized buffer", __LINE__);
			gsc_buf_man_release_buffer(bm);
			break;
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static void _rx_process(const args_t* args)
{
	gsc_buf_man_t	bm;
	int				(*data_sync)(const args_t* args, rx_data_t* rx, gsc_buf_man_t* bm);
	int				(*data_read)(const args_t* args, rx_data_t* rx, gsc_buf_man_t* bm);
	int				eof;
	int				err;
	int				errs		= 0;
	long			ms;
	long long		ns;
	int				ret;
	rx_data_t*		rx			= args->rx;
	long			us;

	memset(&bm, 0, sizeof(gsc_buf_man_t));
	rx->byte_limit		= 1000000LL * args->rx_mb;
	rx->read_calls		= 0;
	rx->wait_done		= 0;
	rx->wait_timeout	= 0;
	errs				+= _flush_input_buffer(args, rx, &bm);
	errs				+= _func_select(args, rx, &bm, &data_sync, &data_read);

	if (data_sync)
		errs	+= (data_sync)(args, rx, &bm);

	os_time_get_ns(&rx->time_start);
	rx->time_limit.tv_sec	= rx->time_start.tv_sec + args->rx_seconds;
	rx->time_limit.tv_nsec	= rx->time_start.tv_nsec;
	rx->total_ms			= 0;

	for (; errs == 0;)
	{
 		rx->reading		= 1;	// Let Tx thread know we've started reading data.
		errs			= _get_rx_buffer(rx, &bm);

		if (errs == 0)
		{
			ret	= (data_read)(args, rx, &bm);

			if (ret < 0)
			{
				bm.eof		= 1;
				rx->errs	= 1;
				rx->stop	= 1;
				sprintf(rx->err_buf, "%d. Rx read failure: %d", __LINE__, ret);
				break;
			}

			bm.count		= ret;
			rx->total_bytes	+= ret;
			_check_eof(args, rx, &bm);
		}

		eof	= bm.eof;
		err	= gsc_buf_man_release_buffer(&bm);

		if ((err) && (rx->errs == 0))
		{
			errs		= 1;
			rx->errs	= 1;
			rx->stop	= 1;
			sprintf(rx->err_buf, "%d. Rx buffer release failure", __LINE__);
		}

		if ((errs) || (eof) || (rx->errs) || (rx->stop))
			break;
	}

	os_time_get_ns(&rx->time_stop);
	ns	= ((long long) rx->time_stop.tv_sec * 1000000000)
		+ ((long long) rx->time_stop.tv_nsec)
		- ((long long) rx->time_start.tv_sec * 1000000000)
		- ((long long) rx->time_start.tv_nsec);
	us	= (ns + 500) / 1000;
	ms	= (us + 500) / 1000;
	rx->total_ms	= ms;
}



//*****************************************************************************
static int _rx_thread(void* arg)
{
	const args_t*	args	= arg;
	rx_data_t*		rx		= args->rx;

	// Tell the controlling code that this thread has started.
	rx->started	= 1;

	// Wait till we're told to start.
	_wait_for_start(rx);

	// Perform the expected activity.
	_rx_process(args);

	// Tell the controlling code that we're done.
	rx->done	= 1;
	return(0);
}



//*****************************************************************************
int rx_start(const args_t* args)
{
	char		buf[1024]	= "";
	int			errs		= 0;
	rx_data_t*	rx			= args->rx;

	perform_text(args, 1, "Rx Startup", ", Rx Startup");

	if (_start_time_set == 0)
	{
		os_time_get_ns(&_start_time);
		_start_time_set	= 1;
	}

	rx->start_time	= _start_time;

	switch (args->rx_option)
	{
		default:

			errs	= ERROR_INTERNAL_RX_INT(rx, args->rx_option);
			break;

		case RX_OPTION_ZERO_DATA:

			strcpy(buf, "providing NULL data");
			break;

		case RX_OPTION_READ_DEV:

			strcpy(buf, "reading from device");
			break;
	}

	if (args->rx_seconds)
	{
		strcat(buf, ", >= ");
		gsc_label_long_comma_buf(args->rx_seconds, buf + strlen(buf));
		strcat(buf, " Second");

		if (args->rx_seconds != 1)
			strcat(buf, "s");
	}

	if (args->rx_mb)
	{
		strcat(buf, ", >= ");
		gsc_label_long_comma_buf(args->rx_mb, buf + strlen(buf));
		strcat(buf, " MB");
	}

	if (errs == 0)
	{
		errs	= os_thread_create(&rx->thread, "Rx Thread", _rx_thread, (void*) args);

		if (errs)
			strcpy(buf, "creating Rx thread");
	}

	if (errs == 0)
		_wait_til_started(rx);

	if (args->range == 0)
		printf("%s  (%s)\n", errs ? "FAIL <---" : "PASS", buf);
	else if (errs)
		printf(" FAIL <---");

	return(errs);
}



//*****************************************************************************
int rx_stop(const args_t* args)
{
	int			errs	= 0;
	rx_data_t*	rx		= args->rx;
	int			samples;

	// STOP ===============================================
	perform_text(args, 1, "Rx Stop", ", Rx Stop");

	rx->stop	= 1;
	ai32ssc_rx_io_abort(args->fd, -1, 0, NULL);

	if (rx->started)
		_wait_til_done(rx);

	os_thread_destroy(&rx->thread);

	perform_text(args, 0, "Done\n", NULL);

	gsc_label_level_inc();

	// STATUS =============================================
	perform_text(args, 1, "Status", NULL);

	if (rx->errs == 0)
	{
		errs	= 0;
		perform_text(args, 0, "PASS\n", NULL);
	}
	else
	{
		errs	= 1;

		if (args->range == 0)
		{
			printf("FAIL <---");

			if (rx->err_buf[0])
				printf(" (%s)", rx->err_buf);

			printf("\n");
		}
	}

	// THROUGHPUT =========================================

	if (args->rx_data == RX_DATA_WAIT_LL)
		samples	= rx->total_bytes / 4;
	else if (args->data_pack)
		samples	= rx->total_bytes / 2;
	else
		samples	= rx->total_bytes / 4;

	if (rx->total_ms)
		rx->overall_rate	= (double) (1000LL * samples) / rx->total_ms;

	if (args->range == 0)
	{
		gsc_label("Throughput");
		gsc_label_long_comma(samples);
		printf(" samples, ");
		gsc_label_float_comma((double) rx->total_ms / 1000, 0, 3);
		printf(" Secs");

		if (rx->total_ms)
		{
			printf(", ");
			gsc_label_float_comma(rx->overall_rate, 0, 3);
			printf(" S/S");
		}

		printf("\n");
	}

	// Stats ==============================================

	if (args->range == 0)
	{
		gsc_label("ai32ssc_read()");
		gsc_label_long_comma(rx->read_calls);
		printf(" calls\n");

		gsc_label("Wait Event Done");
		gsc_label_long_comma(rx->wait_done);
		printf("\n");

		gsc_label("Wait Event Timeout");
		gsc_label_long_comma(rx->wait_timeout);
		printf("\n");
	}

	gsc_label_level_dec();
	return(errs);
}


