// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/samples/snapshot/perform.c $
// $Rev: 54732 $
// $Date: 2024-07-01 08:13:38 -0500 (Mon, 01 Jul 2024) $

// 16AI32SSC: Sample Application: source file

#include "main.h"



//*****************************************************************************
static int _buffer_config(const args_t* args, int* allocated)
{
	int	errs;

	perform_text(args, 1, "Buffer Configuration", ", Allocate");

	errs	= gsc_buf_man_init();

	if (errs == 0)
		errs	= gsc_buf_man_setup(BUFFER_QTY, BUFFER_SIZE);

	if (errs)
	{
		perform_text(args, 0, "FAIL <---\n", " FAIL <---");
	}
	else if (args->range == 0)
	{
		printf("PASS  (%d buffers, ", (int) BUFFER_QTY);
		gsc_label_long_comma(BUFFER_SIZE);
		printf(" bytes each)\n");
	}

	allocated[0]	= errs ? 0 : 1;
	return(errs);
}



//*****************************************************************************
static void _buffer_free(const args_t* args)
{
	perform_text(args, 1, "Buffer Free", ", Free");
	gsc_buf_man_free_all();
	perform_text(args, 0, "Done\n", NULL);
}



//*****************************************************************************
static void _buffer_stats(void)
{
	gsc_buf_man_stats();
}



//*****************************************************************************
static void _buffer_stop(void)
{
	gsc_buf_man_stop();
}



//*****************************************************************************
static int _config_device(const args_t* args, s32 fsamp)
{
	s32			chan_qty	= 32;
	int			errs		= 0;
	rx_data_t*	rx			= args->rx;
	tx_data_t*	tx			= args->tx;
	int			verbose;

	if (args->range)
	{
		verbose	= 0;
		printf("Config");
		fflush(stdout);
	}
	else
	{
		verbose	= 1;
		gsc_label("Configuration");
		printf("\n");
		gsc_label_level_inc();
	}

	errs	+= ai32ssc_config_ai		(args->fd, -1, verbose, fsamp);
	errs	+= ai32ssc_rx_io_mode		(args->fd, -1, verbose, args->io_mode, NULL);
	errs	+= ai32ssc_query			(args->fd, -1, verbose, AI32SSC_QUERY_CHANNEL_QTY, &chan_qty);
	errs	+= ai32ssc_ain_buf_overflow	(args->fd, -1, verbose, AI32SSC_AIN_BUF_OVERFLOW_IGNORE, NULL);
	errs	+= ai32ssc_ain_buf_underflow(args->fd, -1, verbose, AI32SSC_AIN_BUF_UNDERFLOW_IGNORE, NULL);
	errs	+= ai32ssc_data_packing		(args->fd, -1, verbose, args->data_pack, NULL);
	errs	+= ai32ssc_rx_io_timeout	(args->fd, -1, verbose, 0, NULL);	// MUST BE ZERO FOR NO SLEEPING!

	if (args->chan_qty)
	{
		if (chan_qty > args->chan_qty)
			chan_qty	= args->chan_qty;

		errs	+= ai32ssc_chan_active	(args->fd, -1, verbose, AI32SSC_CHAN_ACTIVE_RANGE, NULL);
		errs	+= ai32ssc_chan_first	(args->fd, -1, verbose, 0, NULL);
		errs	+= ai32ssc_chan_last	(args->fd, -1, verbose, chan_qty - 1, NULL);
	}

	gsc_label_level_dec();

	rx->chan_qty	= chan_qty;
	tx->chan_qty	= chan_qty;

	if ((args->range) && (errs))
		printf(" FAIL <---");

	return(errs);
}



//*****************************************************************************
static int _save_str(rx_data_t* rx, FILE* file, const char* src)
{
	int	errs	= 0;
	int	len;
	int	ret;

	len		= (int) strlen(src);
	ret		= fwrite(src, len, 1, file);

	if (ret <= 0)
	{
		errs		= 1;
		rx->errs	= 1;
		rx->stop	= 1;
		sprintf(rx->err_buf, "%d. file write error: %d", __LINE__, ret);
	}

	return(errs);
}



//*****************************************************************************
static int _save_results(const args_t* args, rx_data_t* rx, s32 fsamp, double rate)
{
	static int	opened	= 0;

	char	buf[80];
	char	name[128];
	int		errs		= 0;
	FILE*	file;

	sprintf(name, "rate_%d_", (int) args->rx->chan_qty);

	switch (args->rx_data)
	{
		default:	errs	= ERROR_INTERNAL_RX_INT(rx, args->rx_data);	break;
		case RX_DATA_POLL_FIFO:	strcat(name, "poll_fifo");				break;
		case RX_DATA_POLL_READ:	strcat(name, "poll_read");				break;
		case RX_DATA_WAIT_LL:	strcat(name, "wait_ll");				break;
		case RX_DATA_WAIT_READ:	strcat(name, "wait_read");				break;
	}

	if (args->data_pack)
	{
		switch (args->rx_data)
		{
			default:	errs	= ERROR_INTERNAL_RX_INT(rx, args->rx_data);	break;
			case RX_DATA_POLL_FIFO:	strcat(name, "_dp");					break;
			case RX_DATA_POLL_READ:	strcat(name, "_dp");					break;
			case RX_DATA_WAIT_LL:											break;
			case RX_DATA_WAIT_READ:	strcat(name, "_dp");					break;
		}
	}

	if (errs == 0)
	{
		strcat(name, ".txt");
		file	= fopen(name, "a");

		if (file == NULL)
		{
			errs		= 1;
			rx->errs	= 1;
			rx->stop	= 1;
			sprintf(rx->err_buf, "%d. file open error", __LINE__);
		}
		else
		{
			if (opened == 0)
			{
				opened	= 1;
				errs	+= _save_str(rx, file, "# Fsamp   Results\n");
				errs	+= _save_str(rx, file, "# ======  ==========\n");
			}

			sprintf(buf, "  %6d  %10.3f\n", (int) fsamp, (float) rate);
			errs	+= _save_str(rx, file, buf);
			fclose(file);
		}
	}

	return(errs);
}



//*****************************************************************************
static int _range_results_save(const args_t* args, int errs)
{
	double		rate;
	rx_data_t*	rx	= args->rx;

	rate	= rx->overall_rate / rx->chan_qty;

	if (errs == 0)
	{
		printf(", Save");
		errs	= _save_results(args, rx, rx->fsamp, rate);
	}

	if (errs)
	{
		printf(": FAIL <---");

		if (rx->err_buf[0])
			printf(" (%s)", rx->err_buf);
	}
	else
	{
		printf(": PASS (");
		gsc_label_float_comma(rate, 0, 3);
		printf(")");
	}

	printf("\n");
	return(errs);
}



//*****************************************************************************
static int _work_start(const args_t* args, int errs)
{
	char		buf[80]	= "";
	rx_data_t*	rx		= args->rx;
	tx_data_t*	tx		= args->tx;

	if (args->range)
	{
		printf(", Working");
		fflush(stdout);
	}
	else
	{
		strcpy(buf, "Working");

		if (args->force)
		{
			errs	= 0;
			strcat(buf, " (Forced)");
		}
	}

	if (args->range == 0)
		gsc_label(buf);

	if (errs)
	{
		errs	= 0;

		if (args->range)
		{
			printf(" FAIL <---");
		}
		else
		{
			printf("ABORTED DUE TO SETUP ERRORS <---\n");
		}
	}
	else
	{
		if (args->range == 0)
			printf("\n");

		gsc_label_level_inc();

		// Begin working.

		if (args->range == 0)
			gsc_label("Starting");

		tx->start	= 1;
		rx->start	= 1;

		if (args->range == 0)
			printf("Done\n");

		if (args->range)
		{
			printf(", Wait");
			fflush(stdout);
		}
		else
		{
			gsc_label("Waiting");
		}

		for (;;)
		{
			if ((rx->errs) || (tx->errs))
			{
				tx->stop	= 1;
				rx->stop	= 1;
				_buffer_stop();
			}

			if ((rx->done) && (tx->done))
				break;

			gsc_time_sleep_ms(10);
		}

		_buffer_stop();

		if (args->range == 0)
			printf("Done\n");

		gsc_label_level_dec();
	}

	return(errs);
}



//*****************************************************************************
static int _test_series(const args_t* args, s32 fsamp)
{
	int			allocated	= 0;
	char		buf[80];
	int			errs;
	rx_data_t*	rx			= args->rx;
	tx_data_t*	tx			= args->tx;

	memset(rx, 0, sizeof(rx[0]));
	memset(tx, 0, sizeof(tx[0]));

	rx->fsamp	= fsamp;

	if (args->range)
	{
		gsc_label_long_comma_buf(fsamp, buf);
		strcat(buf, " S/S");
		gsc_label(buf);
	}

	if ((args->range) && (args->delay_s))
	{
		printf("Delay");
		fflush(stdout);
		os_sleep_ms(args->delay_s * 1000);
		printf(", ");
	}

	errs	= _config_device(args, fsamp);

	if (errs == 0)
		errs	= _buffer_config(args, &allocated);

	if (errs == 0)
		errs	= rx_start(args);

	if ((errs == 0) && (rx->started))
		errs	= tx_start(args);

	if (errs == 0)
		errs	= _work_start(args, errs);

	if (rx->started)
		errs	+= rx_stop(args);

	if (tx->started)
		errs	+= tx_stop(args);

	if (allocated)
		_buffer_free(args);

	if (args->range)
	{
		errs	+= _range_results_save(args, errs);
	}
	else
	{
		errs	+= ai32ssc_ain_buf_overflow(args->fd, -1, 1, -1, NULL);
		errs	+= ai32ssc_ain_buf_underflow(args->fd, -1, 1, -1, NULL);
	}

	if ((errs == 0) && (args->stats) && (args->range == 0))
		_buffer_stats();

	return(errs);
}



//*****************************************************************************
int perform_tests(const args_t* args)
{
	int	errs	= 0;
	s32	fsamp;
	s32	ll;		// Low Latency
	int	tmp;

	for (;;)	// A convenience loop.
	{
		if (args->rx_data == RX_DATA_WAIT_LL)
		{
			errs	+= ai32ssc_query(args->fd, -1, 1, AI32SSC_QUERY_LOW_LATENCY, &ll);

			if (ll == 0)
			{
				errs++;
				gsc_label(NULL);
				printf("FAIL <---  (Low Latency selected, but not available.)\n");
				break;
			}
		}

		if (args->range == 0)
		{
			errs	= _test_series(args, args->fsamp);
		}
		else
		{
			for (fsamp = args->range_min; fsamp <= args->range_max; fsamp += args->range_inc)
			{
				tmp		= _test_series(args, fsamp);
				errs	+= tmp;
			}
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
void perform_text(const args_t* args, int label, const char* single, const char* range)
{
	if (args->range)
	{
		if (range)
		{
			printf("%s", range);
			fflush(stdout);
		}
	}
	else
	{
		if (single)
		{
			if (label)
				gsc_label(single);
			else
				printf("%s", single);
		}
	}
}


