// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/samples/snapshot/main.c $
// $Rev: 54732 $
// $Date: 2024-07-01 08:13:38 -0500 (Mon, 01 Jul 2024) $

// 16AI32SSC: Sample Application: source file

#include "main.h"



//*****************************************************************************
static int _parse_args(int argc, char** argv, args_t* args)
{
	const gsc_arg_item_t	list[]	=
	{
		//	type				var						arg			values							desc
		{	GSC_ARG_S32_FLAG,	&args->io_mode,			"-bmdma",	{ GSC_IO_MODE_BMDMA },			"Transfer data using Block Mode DMA."						},
		{	GSC_ARG_S32_RANGE,	&args->chan_qty,		"-chans",	{ 1, 32 },						"Use # number of channels (default: all)."					},
		{	GSC_ARG_S32_FLAG,	&args->io_mode,			"-dmdma",	{ GSC_IO_MODE_DMDMA },			"Transfer data using Demand Mode DMA."						},
		{	GSC_ARG_S32_FLAG,	&args->force,			"-f",		{ 1 },							"Force operation even if errors occur."						},
		{	GSC_ARG_S32_RANGE,	&args->fsamp,			"-fsamp",	{ 1, 200000 },					"Sample input data at # S/S (default: 8,000)."				},
		{	GSC_ARG_S32_RANGE,	&args->delay_s,			"-d",		{ 1, 60 },						"Delay for # seconds before each -r iterations."			},
		{	GSC_ARG_S32_FLAG,	&args->data_pack,		"-dp",		{ 1 },							"Enable Data Packing?"										},
		{	GSC_ARG_S32_RANGE,	&args->range_inc,		"-inc",		{ 1, 200000 },					"Incriment -r range test by # S/S."							},
		{	GSC_ARG_S32_RANGE,	&args->range_max,		"-max",		{ 1, 200000 },					"End -r range test at # S/S."								},
		{	GSC_ARG_S32_RANGE,	&args->range_min,		"-min",		{ 1, 200000 },					"Start -r range test at # S/S."								},
		{	GSC_ARG_S32_FLAG,	&args->io_mode,			"-pio",		{ GSC_IO_MODE_PIO },			"Transfer data using PIO (default)."						},
		{	GSC_ARG_S32_FLAG,	&args->range,			"-r",		{ 1 },							"Run fsamp range test from -min to -max by -inc."			},
		{	GSC_ARG_S32_FLAG,	&args->rx_data,			"-rxpf",	{ RX_DATA_POLL_FIFO },			"Poll for data via the FIFO Status then read the data."		},
		{	GSC_ARG_S32_FLAG,	&args->rx_data,			"-rxprx",	{ RX_DATA_POLL_READ },			"Poll for data via the read service."						},
		{	GSC_ARG_S32_FLAG,	&args->rx_data,			"-rxwll",	{ RX_DATA_WAIT_LL },			"Use the Wait Event service then Low Latency retrieval."	},
		{	GSC_ARG_S32_FLAG,	&args->rx_data,			"-rxwrx",	{ RX_DATA_WAIT_READ },			"Use the Wait Event service then read the data."			},
		{	GSC_ARG_S32_FLAG,	&args->rx_option,		"-rx0",		{ RX_OPTION_ZERO_DATA },		"Provide null data rather than read from the device."		},
		{	GSC_ARG_S32_FLAG,	&args->rx_option,		"-rxd",		{ RX_OPTION_READ_DEV },			"Read data from the device (default: 5)."					},
		{	GSC_ARG_S32_RANGE,	&args->rx_mb,			"-rxmb",	{ 1, 30000 },					"Receive # megabytes of data."								},
		{	GSC_ARG_S32_RANGE,	&args->rx_seconds,		"-rxs",		{ 1, 3600 },					"Provide Rx data for # seconds (default: 5)."				},
		{	GSC_ARG_S32_FLAG,	&args->stats,			"-s",		{ 1 },							"Show transfer statistics upon completion."					},
		{	GSC_ARG_S32_FLAG,	&args->tx_option,		"-tx0",		{ TX_OPTION_BIT_BUCKET },		"Do nothing with the received data (default)."				},
		{	GSC_ARG_S32_FLAG,	&args->tx_option,		"-txb",		{ TX_OPTION_WRITE_FILE_BIN },	"Write the receive data to a binary file (datast.bin)."		},
		{	GSC_ARG_S32_FLAG,	&args->tx_chan_tag,		"-txcte",	{ CHAN_TAG_EXCLUDE },			"Exclude the channel tag from .txt file content."			},
		{	GSC_ARG_S32_FLAG,	&args->tx_chan_tag,		"-txcto",	{ CHAN_TAG_ONLY },				"Save only the channel tag in .txt file content."			},
		{	GSC_ARG_S32_FLAG,	&args->tx_decimal,		"-txd",		{ 1 },							"Save .txt file content in decimal format."					},
		{	GSC_ARG_S32_FLAG,	&args->tx_decimal,		"-txh",		{ 0 },							"Save .txt file content as hexadecimal format (default)."	},
		{	GSC_ARG_S32_FLAG,	&args->tx_option,		"-txt",		{ TX_OPTION_WRITE_FILE_TEXT },	"Write the receive data to a text file (datast.txt)."		},
		{	GSC_ARG_DEV_INDEX,	&args->index,			"index",	{ 1 },							"The zero based index of the device to access."				}
	};

	const gsc_arg_set_t	set	=
	{
		/* name	*/	"snapshot",
		/* desc	*/	"Read all active channels as soon as the data is available.",
		/* qty	*/	SIZEOF_ARRAY(list),
		/* list	*/	list
	};

	int	errs;
	s32	tmp;

	memset(args, 0, sizeof(args_t));
	args->fsamp			= 8000;
	args->io_mode		= GSC_IO_MODE_PIO;
	args->range_inc		= -1;
	args->range_max		= -1;
	args->range_min		= -1;
	args->rx_seconds	= 5;
	args->rx_option		= RX_OPTION_READ_DEV;
	args->tx_option		= TX_OPTION_BIT_BUCKET;

	errs	= gsc_args_parse(argc, argv, &set);

	if ((errs == 0) && (args->rx_data == 0))
	{
		errs++;
		gsc_label(NULL);
		printf("FAIL <---  (Data retrieval method not specified: -rxwll or -rxwr)\n");
	}

	if ((errs == 0) && (args->range))
	{
		if (args->range_inc < 0)
		{
			errs++;
			gsc_label(NULL);
			printf("FAIL <---  (-inc not specified.)\n");
		}

		if (args->range_max < 0)
		{
			errs++;
			gsc_label(NULL);
			printf("FAIL <---  (-max not specified.)\n");
		}

		if (args->range_min < 0)
		{
			errs++;
			gsc_label(NULL);
			printf("FAIL <---  (-min not specified.)\n");
		}

		if ((errs == 0) && (args->range_max < args->range_min))
		{
			gsc_label("ADJUSTMENT");
			printf("Swapping -min and -max <---\n");
			tmp				= args->range_max;
			args->range_max	= args->range_min;
			args->range_min	= tmp;
		}
	}

	return(errs);
}



//*****************************************************************************
static void _show_access_index(const args_t* args)
{
	gsc_label("Accessing Device");
	printf("%d\n", args->index);
}



//*****************************************************************************
static void _show_time_stamp(void)
{
	const char*	psz;
	struct tm*	stm;
	time_t		tt;

	time(&tt);
	stm	= localtime(&tt);
	psz	= (char*) asctime(stm);
	gsc_label("Time Stamp");
	printf("%s", psz);
}



/******************************************************************************
*
*	Function:	main
*
*	Purpose:
*
*		Control the overall flow of the application.
*
*	Arguments:
*
*		argc			The number of command line arguments.
*
*		argv			The list of command line arguments.
*
*	Returned:
*
*		EXIT_SUCCESS	No errors were encounterred.
*		EXIT_FAILURE	One or more problems were encounterred.
*
******************************************************************************/

int main(int argc, char** argv)
{
	args_t		args;
	int			errs;
	long		hours;
	long		mins;
	time_t		now;
	int			ret;
	rx_data_t	rx;
	long		secs;
	time_t		test;
	tx_data_t	tx;

	for (;;)	// A convenience loop.
	{
		gsc_label_init(30);
		test	= time(NULL);
		errs	= _parse_args(argc, argv, &args);

		memset(&rx, 0, sizeof(rx));
		memset(&tx, 0, sizeof(tx));
		args.rx		= &rx;
		args.tx		= &tx;

		// Introduction

		os_id_host();
		_show_time_stamp();
		_show_access_index(&args);
		errs	+= ai32ssc_init_util(1);
		errs	+= os_id_driver(ai32ssc_open, ai32ssc_read, ai32ssc_close);
		errs	+= ai32ssc_count_boards(1, &args.qty);

		if (args.qty <= 0)
			errs++;

		// Open access to device

		if (errs == 0)
			errs	= ai32ssc_open_util(args.index, 0, -1, 1, &args.fd);

		if (errs == 0)
		{
			// Device identification

			errs	= ai32ssc_id_device(args.fd, -1, 1);

			// Perform testing

			errs	+= perform_tests(&args);

			// Close device access

			errs	+= ai32ssc_close_util(args.fd, -1, 1);
		}

		// End of processing.

		now	= time(NULL);

		if (errs)
		{
			printf(	"\nRESULTS: FAIL <---  (%d error%s)",
					errs,
					(errs == 1) ? "" : "s");
		}
		else
		{
			printf("\nRESULTS: PASS");
		}

		secs	= (long) (now - test);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(" (duration %ld:%ld:%02ld)\n", hours, mins, secs);
		break;
	}

	ret	= ((args.qty <= 0) || (errs)) ? EXIT_FAILURE : EXIT_SUCCESS;
	return(ret);
}


