// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/samples/sbtest/wait_event.c $
// $Rev: 53697 $
// $Date: 2023-09-07 13:52:59 -0500 (Thu, 07 Sep 2023) $

// 16AI32SSC: Sample Application: source file

#include "main.h"



// macros *********************************************************************

#define	_1_SEC		(1000L)
#define	_15_SEC		(15L *  1000L)
#define	_30_SEC		(30L * 1000L)



// macros *********************************************************************

static	int			rx_stat;
static	sem_t		sem_enter;
static	sem_t		sem_exit;
static	gsc_wait_t	wait_dst;
static	int			wait_fd;



//*****************************************************************************
static int _service_test(int fd)
{
	// There are no settings bits to test here.
	return(0);
}



//*****************************************************************************
static int	_wait_main_dma(int fd)
{
	static const service_data_t	list[]	=
	{
		{					// Select DMA mode.
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_MODE,
			/* arg		*/	GSC_IO_MODE_BMDMA,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_OVERFLOW,
			/* arg		*/	AI32SSC_IO_OVERFLOW_IGNORE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_UNDERFLOW,
			/* arg		*/	AI32SSC_IO_UNDERFLOW_IGNORE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		// Put data in the FIFO.

		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_AIN_BUF_THR_LVL,
			/* arg		*/	100,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RAG_ENABLE,
			/* arg		*/	AI32SSC_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_CLK_SRC,
			/* arg		*/	AI32SSC_ADC_CLK_SRC_RAG,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_BURST_SYNC,
			/* arg		*/	AI32SSC_BURST_SYNC_DISABLE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_ENABLE,
			/* arg		*/	AI32SSC_ADC_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_SLEEP_MS,
			/* cmd		*/	0,
			/* arg		*/	50,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	u32	buf[128];
	int	errs	= 0;
	s32	sts		= 0;

	errs	+= ai32ssc_initialize(fd, -1, 0);
	errs	+= service_ioctl_set_reg_list(fd, list);

	// Now perform a DMA read to trigger the wait.

	if (errs == 0)
	{
		sts		= ai32ssc_read(fd, buf, sizeof(buf));

		if (sts != sizeof(buf))
		{
			errs++;
			printf(	"FAIL <---  (%d. ai32ssc_read: got %ld, expected %ld)\n",
					__LINE__,
					(long) sts,
					(long) SIZEOF_ARRAY(buf));
		}
	}

	errs	+= ai32ssc_initialize(fd, -1, 0);
	return(errs);
}



//*****************************************************************************
static int	_wait_main_gsc(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_IRQ0_SEL,
			/* arg		*/	AI32SSC_IRQ0_INIT_DONE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_INITIALIZE,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;
	int	ret;

	errs	= service_ioctl_set_reg_list(fd, list);
	ret		= errs;
	return(ret);
}



//*****************************************************************************
static int	_wait_gsc_init_done(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_IRQ0_SEL,
			/* arg		*/	AI32SSC_IRQ0_INIT_DONE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_INITIALIZE,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;
	int	ret;

	errs	= service_ioctl_set_reg_list(fd, list);
	ret		= errs;
	return(ret);
}



//*****************************************************************************
static int	_wait_gsc_autocal_done(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_IRQ0_SEL,
			/* arg		*/	AI32SSC_IRQ0_AUTOCAL_DONE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_AUTOCAL,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;
	int	ret;

	errs	= service_ioctl_set_reg_list(fd, list);
	ret		= errs;
	return(ret);
}



//*****************************************************************************
static int	_wait_gsc_sync_start(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_IRQ0_SEL,
			/* arg		*/	AI32SSC_IRQ0_SYNC_START,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RAG_ENABLE,
			/* arg		*/	AI32SSC_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_CLK_SRC,
			/* arg		*/	AI32SSC_ADC_CLK_SRC_RAG,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_BURST_SYNC,
			/* arg		*/	AI32SSC_BURST_SYNC_RBG,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_ENABLE,
			/* arg		*/	AI32SSC_ADC_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_SLEEP_MS,
			/* cmd		*/	0,
			/* arg		*/	50,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_INPUT_SYNC,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;
	int	ret;

	errs	= service_ioctl_set_reg_list(fd, list);
	ret		= errs;
	return(ret);
}



//*****************************************************************************
static int	_wait_gsc_sync_done(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_IRQ0_SEL,
			/* arg		*/	AI32SSC_IRQ0_SYNC_DONE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RAG_ENABLE,
			/* arg		*/	AI32SSC_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_CLK_SRC,
			/* arg		*/	AI32SSC_ADC_CLK_SRC_RAG,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_BURST_SYNC,
			/* arg		*/	AI32SSC_BURST_SYNC_RBG,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_ENABLE,
			/* arg		*/	AI32SSC_ADC_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_SLEEP_MS,
			/* cmd		*/	0,
			/* arg		*/	50,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_INPUT_SYNC,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;
	int	ret;

	errs	= service_ioctl_set_reg_list(fd, list);
	ret		= errs;
	return(ret);
}



//*****************************************************************************
static int	_wait_gsc_burst_start(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_IRQ0_SEL,
			/* arg		*/	AI32SSC_IRQ0_BURST_START,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_CLK_SRC,
			/* arg		*/	AI32SSC_ADC_CLK_SRC_RAG,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_BURST_SYNC,
			/* arg		*/	AI32SSC_BURST_SYNC_BCR,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RAG_ENABLE,
			/* arg		*/	AI32SSC_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_ENABLE,
			/* arg		*/	AI32SSC_ADC_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_INPUT_SYNC,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;
	int	ret;

	errs	= service_ioctl_set_reg_list(fd, list);
	ret		= errs;
	return(ret);
}



//*****************************************************************************
static int	_wait_gsc_burst_done(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_IRQ0_SEL,
			/* arg		*/	AI32SSC_IRQ0_BURST_DONE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_CLK_SRC,
			/* arg		*/	AI32SSC_ADC_CLK_SRC_RAG,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_BURST_SYNC,
			/* arg		*/	AI32SSC_BURST_SYNC_BCR,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RAG_ENABLE,
			/* arg		*/	AI32SSC_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_ENABLE,
			/* arg		*/	AI32SSC_ADC_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_INPUT_SYNC,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;
	int	ret;

	errs	= service_ioctl_set_reg_list(fd, list);
	ret		= errs;
	return(ret);
}



//*****************************************************************************
static int	_wait_gsc_in_buf_thr_l2h(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_IRQ1_SEL,
			/* arg		*/	AI32SSC_IRQ1_IN_BUF_THR_L2H,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_AIN_BUF_THR_LVL,
			/* arg		*/	100,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RAG_ENABLE,
			/* arg		*/	AI32SSC_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_CLK_SRC,
			/* arg		*/	AI32SSC_ADC_CLK_SRC_RAG,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_BURST_SYNC,
			/* arg		*/	AI32SSC_BURST_SYNC_DISABLE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_ENABLE,
			/* arg		*/	AI32SSC_ADC_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_SLEEP_MS,
			/* cmd		*/	0,
			/* arg		*/	50,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;
	int	ret;

	errs	= service_ioctl_set_reg_list(fd, list);
	ret		= errs;
	return(ret);
}



//*****************************************************************************
static int	_wait_gsc_in_buf_thr_h2l(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_AIN_BUF_THR_LVL,
			/* arg		*/	100,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_IRQ1_SEL,
			/* arg		*/	AI32SSC_IRQ1_IN_BUF_THR_H2L,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RAG_ENABLE,
			/* arg		*/	AI32SSC_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_CLK_SRC,
			/* arg		*/	AI32SSC_ADC_CLK_SRC_RAG,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_BURST_SYNC,
			/* arg		*/	AI32SSC_BURST_SYNC_DISABLE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_ENABLE,
			/* arg		*/	AI32SSC_ADC_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_SLEEP_MS,
			/* cmd		*/	0,
			/* arg		*/	50,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_AIN_BUF_CLEAR,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;

	errs	= service_ioctl_set_reg_list(fd, list);
	return(errs);
}



//*****************************************************************************
static int	_wait_gsc_in_buf_ovr_undr(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_IRQ1_SEL,
			/* arg		*/	AI32SSC_IRQ1_IN_BUF_OVR_UNDR,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_REG_READ,
			/* cmd		*/	0,
			/* arg		*/	0,
			/* reg		*/	AI32SSC_GSC_IBDR,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;
	int	ret;

	errs	= service_ioctl_set_reg_list(fd, list);
	ret		= errs;
	return(ret);
}



//*****************************************************************************
static int	_wait_io_rx_done(int fd)
{
	static const service_data_t	list[]	=
	{
		{					// Select PIO mode.
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_MODE,
			/* arg		*/	GSC_IO_MODE_PIO,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_OVERFLOW,
			/* arg		*/	AI32SSC_IO_OVERFLOW_IGNORE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_UNDERFLOW,
			/* arg		*/	AI32SSC_IO_UNDERFLOW_IGNORE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		// Fill the FIFO.

		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_ENABLE,
			/* arg		*/	AI32SSC_ADC_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RAG_ENABLE,
			/* arg		*/	AI32SSC_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_ADC_CLK_SRC,
			/* arg		*/	AI32SSC_ADC_CLK_SRC_RAG,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_BURST_SYNC,
			/* arg		*/	AI32SSC_BURST_SYNC_DISABLE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RAG_ENABLE,
			/* arg		*/	AI32SSC_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{				// Wait for data.
			/* service	*/	SERVICE_SLEEP_MS,
			/* cmd		*/	0,
			/* arg		*/	50,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	u32	buf[16];
	int	errs	= 0;
	int	ret;
	s32	sts		= 0;

	errs	+= ai32ssc_initialize(fd, -1, 0);

	// Put some data in the Rx FIFO.
	errs	+= service_ioctl_set_reg_list(fd, list);

	// Now perform a read to trigger the wait.

	if (errs == 0)
	{
		sts		= ai32ssc_read(fd, buf, sizeof(buf));

		if (sts < 0)
		{
			errs++;
		}
		else if (sts != sizeof(buf))
		{
			errs++;
			printf(	"FAIL <---  (%d. ai32ssc_read: got %ld, expected %ld)\n",
					__LINE__,
					(long) sts,
					(long) sizeof(buf));
		}
	}

	errs	+= ai32ssc_initialize(fd, -1, 0);
	ret		= errs;
	return(ret);
}



//*****************************************************************************
static int	_wait_io_rx_error(int fd)
{
	int	errs	= 0;
	int	ret;

	// Now perform a read to trigger the wait.
	ret	= ai32ssc_read(fd, NULL, 3);

	if (ret >= 0)
	{
		errs++;
		printf("FAIL <---  (%d. expected < 0, got %d)\n", __LINE__, ret);
	}

	ret	= errs;
	return(ret);
}



//*****************************************************************************
static int	_wait_io_rx_timeout(int fd)
{
	static const service_data_t	list[]	=
	{
		{					// Select PIO mode.
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_MODE,
			/* arg		*/	GSC_IO_MODE_PIO,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_OVERFLOW,
			/* arg		*/	AI32SSC_IO_OVERFLOW_IGNORE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_UNDERFLOW,
			/* arg		*/	AI32SSC_IO_UNDERFLOW_IGNORE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_TIMEOUT,
			/* arg		*/	1,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	u32	buf[16];
	int	errs	= 0;
	int	ret;

	errs	+= ai32ssc_initialize(fd, -1, 0);

	// Verify that the status is an overflow.
	errs	+= service_ioctl_set_reg_list(fd, list);

	// Now perform a read to trigger the wait.

	if (errs == 0)
	{
		ret	= ai32ssc_read(fd, buf, sizeof(buf));

		if (ret != 0)
		{
			errs++;
			printf(	"FAIL <---  (%d. expected %d, got %d)\n",
					__LINE__,
					0,
					ret);
		}
	}

	errs	+= ai32ssc_initialize(fd, -1, 0);
	ret		= errs;
	return(ret);
}



//*****************************************************************************
static int _read_thread(void* arg)
{
	u32		buf[1024];
	int		fd	= (int) (long) arg;
	long	status;

	status	= ai32ssc_read(fd, buf, sizeof(buf));

	if (status == 0)
	{
		rx_stat	= 1;
	}
	else
	{
		rx_stat	= 2;
		printf("FAIL <---  (%d. ai32ssc_read returned %ld, expected 0)\n",
				__LINE__,
				(long) status);
	}

	return(0);
}



//*****************************************************************************
static int	_wait_io_rx_abort(int fd)
{
	static const service_data_t	list[]	=
	{
		{					// Select PIO mode.
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_MODE,
			/* arg		*/	GSC_IO_MODE_PIO,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_OVERFLOW,
			/* arg		*/	AI32SSC_IO_OVERFLOW_IGNORE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_UNDERFLOW,
			/* arg		*/	AI32SSC_IO_UNDERFLOW_IGNORE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AI32SSC_IOCTL_RX_IO_TIMEOUT,
			/* arg		*/	10,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int			errs	= 0;
	int			i;
	int			ret;
	s32			set;
	os_thread_t	thread;

	errs	+= ai32ssc_initialize(fd, -1, 0);

	// Verify that the status is an overflow.
	errs	+= service_ioctl_set_reg_list(fd, list);

	// Now start the read thread so we can do the abort.
	memset(&thread, 0, sizeof(thread));

	if (errs == 0)
	{
		errs	+= os_thread_create(	&thread,
										"Rx",
										_read_thread,
										(void*) (long) fd);
	}

	for (i = 0; (errs == 0) && (i < 100); i++)
	{
		set		= 0;
		ret		= ai32ssc_ioctl(fd, AI32SSC_IOCTL_RX_IO_ABORT, &set);
		errs	+= ret ? 1 : 0;

		if (errs)
			break;

		if (set)
			break;

		os_sleep_ms(100);
	}

	// Wait for the thread to report its status.

	for (i = 0; i < 100; i++)
	{
		if (errs)
			break;

		if (rx_stat == 0)
		{
			os_sleep_ms(100);
			continue;
		}

		if (rx_stat == 1)
			break;

		errs++;
		break;
	}

	errs	+= os_thread_destroy(&thread);
	errs	+= ai32ssc_initialize(fd, -1, 0);
	ret		= errs;
	return(ret);
}



//*****************************************************************************
static int	_wait_timeout(int fd)
{
	int	ret;

	os_sleep_ms(2000);
	ret	= 0;
	return(ret);
}



//*****************************************************************************
static int	_wait_cancel(int fd)
{
	int			errs	= 0;
	int			ret;
	gsc_wait_t	wait;

	wait.flags			= 0;
	wait.main			= 0xFFFFFFFF;
	wait.gsc			= 0xFFFFFFFF;
	wait.alt			= 0xFFFFFFFF;
	wait.io				= 0xFFFFFFFF;
	wait.timeout_ms		= 0;
	wait.count			= 0;
	ret		= ai32ssc_ioctl(wait_fd, AI32SSC_IOCTL_WAIT_CANCEL, &wait);
	errs	+= ret ? 1 : 0;
	ret		= errs;
	return(ret);
}



// macros *********************************************************************

static const struct
{
	const char*	name;
	int			(*function)(int fd);
	gsc_wait_t	wait;
} wait_src[]	=
{
	// name						function					flags			main								...			timeout_ms
	{ "DMA0/DMA1",				_wait_main_dma,				{ 0,			GSC_WAIT_MAIN_DMA0 |
																			GSC_WAIT_MAIN_DMA1,					0, 0, 0,	_15_SEC	}	},
	{ "GSC",					_wait_main_gsc,				{ 0,			GSC_WAIT_MAIN_GSC,					0, 0, 0,	_15_SEC	}	},

	// name						function					flags ...		gsc									...			timeout_ms
	{ "Initialization Done",	_wait_gsc_init_done,		{ 0, 0,			AI32SSC_WAIT_GSC_INIT_DONE,			0, 0,		_15_SEC	}	},
	{ "Autocal Done",			_wait_gsc_autocal_done,		{ 0, 0,			AI32SSC_WAIT_GSC_AUTOCAL_DONE,		0, 0,		_15_SEC	}	},
	{ "Sync Start",				_wait_gsc_sync_start,		{ 0, 0,			AI32SSC_WAIT_GSC_SYNC_START,		0, 0,		_15_SEC	}	},
	{ "Sync Done",				_wait_gsc_sync_done,		{ 0, 0,			AI32SSC_WAIT_GSC_SYNC_DONE,			0, 0,		_15_SEC	}	},
	{ "Burst Start",			_wait_gsc_burst_start,		{ 0, 0,			AI32SSC_WAIT_GSC_BURST_START,		0, 0,		_15_SEC	}	},
	{ "Burst Done",				_wait_gsc_burst_done,		{ 0, 0,			AI32SSC_WAIT_GSC_BURST_DONE,		0, 0,		_15_SEC	}	},
	{ "Buf Threshold Low->Hi",	_wait_gsc_in_buf_thr_l2h,	{ 0, 0,			AI32SSC_WAIT_GSC_IN_BUF_THR_L2H,	0, 0,		_15_SEC	}	},
	{ "Buf Threshold Hi->Low",	_wait_gsc_in_buf_thr_h2l,	{ 0, 0,			AI32SSC_WAIT_GSC_IN_BUF_THR_H2L,	0, 0,		_15_SEC	}	},
	{ "Input Buffer Error",		_wait_gsc_in_buf_ovr_undr,	{ 0, 0,			AI32SSC_WAIT_GSC_IN_BUF_OVR_UNDR,	0, 0,		_15_SEC	}	},

	// name						function					flags ...		io												timeout_ms
	{ "Rx Done",				_wait_io_rx_done,			{ 0, 0, 0, 0,	AI32SSC_WAIT_IO_RX_DONE,						_15_SEC	}	},
	{ "Rx Error",				_wait_io_rx_error,			{ 0, 0, 0, 0,	AI32SSC_WAIT_IO_RX_ERROR,						_15_SEC	}	},
	{ "Rx Timeout",				_wait_io_rx_timeout,		{ 0, 0, 0, 0,	AI32SSC_WAIT_IO_RX_TIMEOUT,						_15_SEC	}	},
	{ "Rx Abort",				_wait_io_rx_abort,			{ 0, 0, 0, 0,	AI32SSC_WAIT_IO_RX_ABORT,						_15_SEC	}	},

	// name						function					flags ...		io												timeout_ms
	{ "Wait Timeout",			_wait_timeout,				{ 0, 0, 0, 0,	AI32SSC_WAIT_IO_RX_DONE,						_1_SEC	}	},
	{ "Wait Cancel",			_wait_cancel,				{ 0, 0, 0, 0,	AI32SSC_WAIT_IO_RX_DONE,						_30_SEC	}	},

	// terminate list
	{ NULL, NULL, { 1 } }
};



//*****************************************************************************
static int _wait_thread(void* arg)
{
	int	status;

	sem_post(&sem_enter);
	status	= ai32ssc_ioctl(wait_fd, AI32SSC_IOCTL_WAIT_EVENT, &wait_dst);

	if (status)
		wait_dst.flags	= 0xFFFFFFFF;

	sem_post(&sem_exit);
	return(0);
}



//*****************************************************************************
static int _process_thread(int index)
{
	char		buf[64];
	int			errs	= 0;
	int			i;
	int			ret;
	int			status;
	os_thread_t	thread;
	gsc_wait_t	wait;

	for (;;)	// A convenience loop.
	{
		sem_init(&sem_enter, 0, 0);
		sem_init(&sem_exit, 0, 0);
		memset(&thread, 0, sizeof(thread));
		sprintf(buf, "wait event %d", index);
		wait_dst	= wait_src[index].wait;
		errs		+= os_thread_create(&thread, buf, _wait_thread, NULL);

		// Wait for the thread to become waiting.
		sem_wait(&sem_enter);

		if (errs == 0)
		{
			wait	= wait_src[index].wait;

			for (i = 0; i < 100; i++)
			{
				status		= ai32ssc_ioctl(wait_fd, AI32SSC_IOCTL_WAIT_STATUS, &wait);

				if (status)
				{
					printf(	"FAIL <---  (%d. status request %d: errno %d)\n",
							__LINE__,
							index,
							errno);
					errs++;
					break;
				}

				if (wait.count == 0)
				{
					os_sleep_ms(100);
					continue;
				}

				if (wait.count == 1)
					break;

				errs++;
				printf(	"FAIL <---  (%d. invalid wait count %ld)\n",
						__LINE__,
						(long) wait.count);
				break;
			}
		}

		// Initiate the respective event.
		errs	+= wait_src[index].function(wait_fd);

		if (errs)
			break;		// There was an error.

		// Verify that the wait ended as expected.

		if (wait_src[index].wait.flags == 0xFFFFFFFF)
			break;		// There was an error.

		// Wait for the resumed thread to run.
		sem_wait(&sem_exit);

		for (i = 0;; i++)
		{
			if (wait_dst.flags)
				break;

			if (i >= 100)
			{
				printf(	"FAIL <---  (%d. thread failed to resume)\n",
						__LINE__);
				errs++;
				break;
			}

			os_sleep_ms(100);
		}

		if (errs)
			break;		// There was an error.

		if (wait_src[index].wait.timeout_ms == _1_SEC)
		{
			if (wait_dst.flags != GSC_WAIT_FLAG_TIMEOUT)
			{
				printf(	"FAIL <---  (%d. flag: expect 0x%lX, got 0x%lX)\n",
						__LINE__,
						(long) GSC_WAIT_FLAG_TIMEOUT,
						(long) wait_dst.flags);
				errs++;
			}

			break;
		}

		if (wait_src[index].wait.timeout_ms == _15_SEC)
		{
			if (wait_dst.flags != GSC_WAIT_FLAG_DONE)
			{
				printf(	"FAIL <---  (%d. flag: expect 0x%lX, got 0x%lX)\n",
						__LINE__,
						(long) GSC_WAIT_FLAG_DONE,
						(long) wait_dst.flags);
				errs++;
			}

			break;
		}

		if (wait_src[index].wait.timeout_ms == _30_SEC)
		{
			if (wait_dst.flags != GSC_WAIT_FLAG_CANCEL)
			{
				printf(	"FAIL <---  (%d. flag: expect 0x%lX, got 0x%lX)\n",
						__LINE__,
						(long) GSC_WAIT_FLAG_TIMEOUT,
						(long) wait_dst.flags);
				errs++;
			}

			break;
		}

		printf("FAIL <---  (%d. INTERNAL ERROR)\n", __LINE__);
		errs++;
		break;
	}

	// Force termination, just in case of an error.
	wait.flags			= 0;
	wait.main			= 0xFFFFFFFF;
	wait.gsc			= 0xFFFFFFFF;
	wait.alt			= 0xFFFFFFFF;
	wait.io				= 0xFFFFFFFF;
	wait.timeout_ms		= 0;
	wait.count			= 0;
	ret		= ai32ssc_ioctl(wait_fd, AI32SSC_IOCTL_WAIT_CANCEL, &wait);
	errs	+= ret ? 1 : 0;
	errs	+= os_thread_destroy(&thread);

	if (errs == 0)
		printf("PASS\n");

	return(errs);
}



//*****************************************************************************
static int _function_test(int fd)
{
	int	errs	= 0;
	int	i;

	printf("\n");
	gsc_label_level_inc();

	// Create threads awaiting each of the source crtiteria
	// and verify that it resumes when the criteria is met.

	for (i = 0; wait_src[i].wait.flags == 0; i++)
	{
		gsc_label(wait_src[i].name);
		wait_fd	= fd;
		memset(&wait_dst, 0, sizeof(wait_dst));
		errs	+= _process_thread(i);
	}

	gsc_label_level_dec();
	return(errs);
}



/******************************************************************************
*
*	Function:	wait_event_test
*
*	Purpose:
*
*		Perform a test of the IOCTL service AI32SSC_IOCTL_WAIT_EVENT.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int wait_event_test(int fd)
{
	int	errs	= 0;

	gsc_label("AI32SSC_IOCTL_WAIT_EVENT");
	errs	+= _service_test(fd);
	errs	+= _function_test(fd);

	return(errs);
}


