// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/samples/savedata/perform.c $
// $Rev: 54763 $
// $Date: 2024-07-01 17:05:28 -0500 (Mon, 01 Jul 2024) $

// 16AI32SSC: Sample Application: source file

#include "main.h"



// macros *********************************************************************

#define	_1M					(1024L * 1024L)
#define	TAG_DECODE(v)		GSC_FIELD_DECODE((v),31,31)
#define	TAG_MASK(v)			((v) & 0x80000000)
#define	VALUE_MASK(v)		((v) & 0x0FFFF)



// variables ******************************************************************

static	u32	_buffer[_1M];



//*****************************************************************************
static int _channels(const args_t* args, s32* chans)
{
	int	errs;

	gsc_label("Input Channels");
	errs	= ai32ssc_query(args->fd, -1, 0, AI32SSC_QUERY_CHANNEL_QTY, chans);

	if (errs == 0)
	{
		printf("%ld Channels\n", (long) chans[0]);
	}
	else
	{
		printf("FAIL <---\n");
		chans[0]	= 32;
		gsc_label(NULL);
		printf("We'll assume %d channels are present.\n", (int) chans[0]);
	}

	return(errs);
}



//*****************************************************************************
static int _read_data(const args_t* args)
{
	char	buf[80];
	int		errs	= 0;
	int		get		= sizeof(_buffer);
	int		got;
	int		repeat;

	for (repeat = 0; (errs == 0) && (repeat < args->repeat);)
	{
		repeat++;
		strcpy(buf, "Reading");

		if (args->repeat > 1)
		{
			strcat(buf, " #");
			gsc_label_long_comma_buf(repeat, buf + strlen(buf));
		}

		gsc_label(buf);

		got	= ai32ssc_read(args->fd, _buffer, get);

		if (got < 0)
		{
			errs	= 1;
			printf("FAIL <---  (read error: %d)\n", got);
			break;
		}

		if (got != get)
		{
			errs	= 1;
			printf(	"FAIL <---  (got %ld bytes, requested %ld)\n",
					(long) got,
					(long) get);
			break;
		}

		printf("PASS  (");
		gsc_label_long_comma(get / 4);
		printf(" samples)\n");
	}

	return(errs);
}



//*****************************************************************************
static int _save_eol(const char* name, FILE* file)
{
	int	errs	= 0;
	int	i;

	i	= fprintf(file, "\r\n");

	if (i != 2)
	{
		printf("FAIL <---  (fprintf() failure to %s)\n", name);
		errs	= 1;
	}

	return(errs);
}



//*****************************************************************************
static int _save_value(const args_t* args, const char* name, FILE* file, u32 value)
{
	char	buf1[40];
	char	buf2[80];
	int		errs	= 0;
	int		i;
	int		len;

	if (args->chan_tag == CHAN_TAG_EXCLUDE)
		value	= VALUE_MASK(value);
	else if (args->chan_tag == CHAN_TAG_ONLY)
		value	= TAG_MASK(value);

	if (args->decimal)
		sprintf(buf1, "%ld", (long) value);
	else
		sprintf(buf1, "%08lX", (long) value);

	sprintf(buf2, "  %8s", buf1);
	len	= strlen(buf2);
	i	= fprintf(file, "%8s", buf2);

	if (i != len)
	{
		printf("FAIL <---  (fprintf() failure to %s)\n", name);
		errs	= 1;
	}

	return(errs);
}



//*****************************************************************************
static int _validate_data(int chans, u32 last, long index, u32 data)
{
	static	int	chan_tag	= 0;
	int			errs		= 0;
	int			expect;
	static	int	last_tag	= 0;

	if (data & 0x7FFF0000)
	{
		errs	= 1;
		printf(" FAIL <---  (Upper bits are invalid: [%ld] = 0x%08lX)\n", index, (long) data);
	}

	chan_tag	= TAG_DECODE(data) ? 0 : chan_tag + 1;

	if (chan_tag >= chans)
	{
		errs	= 1;
		printf(" FAIL <---  (Invalid Channel Tag: [%ld] = 0x%08lX)\n", index, (long) data);
	}

	if (last == 0xFFFFFFFF)
	{
		if (chan_tag != 0)
		{
			errs	= 1;
			printf(" FAIL <---  (First tag not zero: [%ld] = 0x%08lX)\n", index, (long) data);
		}
	}
	else
	{
		last_tag	= TAG_DECODE(last) ? 0 : last_tag + 1;
		expect		= (last_tag + 1) % chans;

		if (chan_tag != expect)
		{
			errs	= 1;
			printf(	" FAIL <---  (Lost Synchronozation: [%ld] = 0x%08lX, [%ld] = 0x%08lX)\n",
					index - 1,
					(long) last,
					index,
					(long) data);
		}
	}

	return(errs);
}



//*****************************************************************************
static int _save_data(const args_t* args, int chans, int errs)
{
	char		buf[64];
	FILE*		file;
	long		index;
	u32			last	= 0xFFFFFFFF;
	const char*	name	= "data.txt";
	long		samples	= sizeof(_buffer) / 4;
	int			tag_got;

	strcpy(buf, "Saving");

	if (args->force_save)
	{
		errs	= 0;
		strcat(buf, " (Forced)");
	}

	gsc_label(buf);

	for (;;)
	{
		if ((errs) && (args->force_save == 0))
		{
			printf("SKIPPED  (errors)\n");
			errs	= 0;
			break;
		}

		file	= fopen(name, "w+b");

		if (file == NULL)
		{
			printf("FAIL <---  (unable to create %s)\n", name);
			errs	= 1;
			break;
		}

		for (index = 0; index < samples; index++)
		{
			tag_got	= TAG_DECODE(_buffer[index]) ? 0 : 1;

			if ((index > 0) && (tag_got == 0))
				errs	+= _save_eol(name, file);

			errs	+= _save_value(args, name, file, _buffer[index]);
			errs	+= _validate_data(chans, last, index, _buffer[index]);
			last	= _buffer[index];
		}

		fclose(file);

		if (errs == 0)
			printf("PASS  (%s)\n", name);

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _config(const args_t* args)
{
	int	errs	= 0;

	gsc_label("Configure");
	printf("\n");
	gsc_label_level_inc();

	errs	+= ai32ssc_config_ai		(args->fd, -1, 1, args->fsamp);
	errs	+= ai32ssc_ain_mode			(args->fd, -1, 1, args->ai_mode, NULL);
	errs	+= ai32ssc_rx_io_mode		(args->fd, -1, 1, args->io_mode, NULL);
	errs	+= ai32ssc_rx_io_overflow	(args->fd, -1, 1, AI32SSC_IO_OVERFLOW_IGNORE, NULL);
	errs	+= ai32ssc_rx_io_underflow	(args->fd, -1, 1, AI32SSC_IO_UNDERFLOW_IGNORE, NULL);
	errs	+= ai32ssc_ain_buf_clear	(args->fd, -1, 1, 1);
	errs	+= ai32ssc_ain_buf_overflow	(args->fd, -1, 1, AI32SSC_AIN_BUF_OVERFLOW_CLEAR, NULL);
	errs	+= ai32ssc_ain_buf_underflow(args->fd, -1, 1, AI32SSC_AIN_BUF_UNDERFLOW_CLEAR, NULL);
	errs	+= ai32ssc_rx_io_timeout	(args->fd, -1, 1, 300, NULL);

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
int perform_tests(const args_t* args)
{
	s32	chans	= 32;
	int	errs	= 0;

	gsc_label("Capture & Save");
	printf("\n");
	gsc_label_level_inc();

	errs	+= _config					(args);
	errs	+= _channels				(args, &chans);
	errs	+= ai32ssc_ain_buf_clear	(args->fd, -1, 1, 1);
	errs	+= _read_data				(args);
	errs	+= ai32ssc_ain_buf_overflow	(args->fd, -1, 1, -1, NULL);
	errs	+= ai32ssc_ain_buf_underflow(args->fd, -1, 1, -1, NULL);
	errs	+= _save_data				(args, chans, errs);

	gsc_label_level_dec();
	return(errs);
}



