// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/driver/main.h $
// $Rev: 54772 $
// $Date: 2024-07-01 18:03:02 -0500 (Mon, 01 Jul 2024) $

// 16AI32SSC: Device Driver: header file

#ifndef	__MAIN_H__
#define	__MAIN_H__

#include "16ai32ssc.h"

#define	DEV_BAR_SHOW						0
#define	DEV_METRICS_SHOW					0
#define	DEV_PCI_ID_SHOW						0
#define	DEV_SUPPORTS_IRQ
#define	DEV_SUPPORTS_READ
#define	DEV_SUPPORTS_WAIT
#define	GSC_ALT_DEV_GET(a)					(a)
#define	GSC_ALT_STRUCT_GET(i,d)				(d)
#define	GSC_ALT_STRUCT_T					dev_data_t
#define	GSC_DEVS_PER_BOARD					1
typedef struct _dev_data_t					dev_data_t;

#include "gsc_main.h"



// macros *********************************************************************

#define	DEV_MODEL							"16AI32SSC"	// Upper case form of the below.
#define	DEV_NAME							"16ai32ssc"	// MUST AGREE WITH AI32SSC_BASE_NAME

#define	DEV_VERSION							"2.8"		// FOR DEVICE SPECIFIC CODE ONLY!
// 2.8	Updated the PIO and DMA threshold values based on testing.
//		Updated for Fedora 38.
//		Removed LL Control Register, LL Hold and LL Release as they are not in firmware.
//		Corrected the operation of the LL Read IOCTL service.
// 2.7	Updated to support the 6.x series kernel.
//		Renamed all Auto_Cal content to Autocal.
//		Added PIO and BMDMA Threshold IOCTL services for performance testing.
//		Updated the Initialize IOCTL service for consistency.
//		Updated the Autocalibration IOCTL service for consistency.
//		Updated the Input Buffer Clear IOCTL service for consistency.
// 2.6	Updated comments for consistency.
//		Updated display output for consistency.
//		Update to some error processing.
//		Updated to support segregation of PLX specific sources.
// 2.5	Updated the Initialize IOCTL service to include waiting on the IRQ.
//		Reorganized some structure content for consistency and clarity.
//		Updated I/O initialization for consistency and clarity.
//		White space cleanup.
//		Updated some comments.
// 2.4	Added support for the 5.x kernel series.
//		Removed compiler warnings for Fedora 31.
//		Updated initial file content for consistency.
//		Updated for porting to Windows.
//		Standardized various IOCTL system log messages.
//		Standardized processing of the initialization IOCTL service.
//		Standardized processing of the autocalibration IOCTL service.
// 2.3	Modified code so open succeeds even if initialization fails.
//		Added Low Latency support.
//		Updated per changes to common code.
// 2.2	Bug fix: was ignoring BAR setup return value.
//		Updated per changes to the OS specific PCI services.
//		Made updates for DEV_SUPPORTS_XXX macro changes.
//		Updated initial file content for consistency.
//		Updated AI32SSC_QUERY_COUNT processing.
//		Added NOTPARALLEL to the makefile.
//		Modified how the last query service value is defined.
// 2.1	The BAR code has been updated to include the data under a single structure.
//		The register definitions have been updated.
//		I/O services now pass around an os_mem_t structure instead of a buffer pointer.
//		White space cleanup.
//		Changed the arbitrary wait event callback argument to a void* type.
//		Modified to allow multiple apps to simultaneously access a single device.
//		Correct access to BAR0 and BAR1 (is RO, was RW).
//		Added an infinite I/O timeout option.
//		Corrected the IOCTL code for the register write and mod services.
// 2.0	Updated to use the newer common driver sources.
//		Removed GNU notice from non-Linux specific files.
//		Removed Linux specific content from non-Linux specific source files.
//		White space cleanup.
//		Now using a spinlock rather than enabling and disabling interrupts.
//		Updated gsc_irq_open() and gsc_irq_close().
//		Updated gsc_dma_open() and gsc_dma_close().
//		Enhanced output reported when the DEV_PCI_ID_SHOW option is enabled.
// 1.14	Reduced #include list in driver interface header.
// 1.13	Corrected a system time tick count roll over bug in ioctl.c.
// 1.12	Updated for the 3.x kernel.
//		Added the AI32SSC_IOCTL_RBG_SYNC_OUTPUT IOCTL service.
//		Added the AI32SSC_QUERY_RBG_SYNC_OUTPUT QUERY option.
//		Added the AI32SSC_QUERY_LOW_LATENCY QUERY option.
//		Updated supported master clock frequescies. Added 40000000 and 48000000.
// 1.11	BAR0 and BAR2 are now the only BAR regions used.
//		Include all common source, though not all are used.
// 1.10	Eliminated the global dev_check_id() routine.
// 1.9	Added ISR support for pre-005 firmware.
//		Fixed the IRQ0 and IRQ1 Select IOCTL services for pre-005 firmware.
// 1.8	Corrected a compile bug for Fedora 7.
// 1.7	Corrected expected initialize and autocalibrate periods.
//		Corrected an Input Buffer Threshold Status service bug.
// 1.6	Added the AI32SSC_IOCTL_SCAN_MARKER_VAL IOCTL service.
//		Implemented support for the common PIO I/O routines.
//		Changed use of DEV_SUPPORTS_PROC_ID_STR macro.
//		Changed use of DEV_SUPPORTS_READ macro.
//		Changed use of DEV_SUPPORTS_WRITE macro.
//		Changed use of DEV_IO_AUTO_START macro.
// 1.5	Made the Autocalibration IOCTL code more robust.
//		Made the Initialization IOCTL code more robust.
//		Fixed AI32SSC_IOCTL_RX_IO_ABORT service bug.
//		Fixed AI32SSC_IOCTL_TT_CHAN_MASK definition.
// 1.4	Overhauled IOCTL based local interrupt implementation.
//		Updated per changes to the common source code.
//		: Added DEV_IO_AUTO_START macro - auto-start not support.
//		: Added DEV_SUPPORTS_PROC_ID_STR macro - not supported.
//		: Implemented IOCTL based I/O Abort support.
//		: Implemented IOCTL based event/interrupt wait support.
//		: Updated to use the common gsc_open() and gsc_close() services.
//		: Implemented explicit local interrupt support.
// 1.3	Corrected a bug in the AI32SSC_IOCTL_AIN_BUF_CLEAR service.
//		Remived unused macros.
// 1.2	Updated the read services to use common argument data types.
//		Fixed a typo/bug in dma_read_available.
//		Updated the driver to use the common open and close service calls.
//		Fixed a bug in the Scan Marker IOCTL service.
// 1.1	Added the dev_check_id() function in device.c.
// 1.0	Initial release of the new driver supporting the basic 16AI32SSC.

// I/O services
#define	DEV_IO_STREAM_QTY					(DEV_IO_RX_STREAM_QTY + DEV_IO_TX_STREAM_QTY)
#define	DEV_IO_RX_STREAM_QTY				1
#define	DEV_IO_TX_STREAM_QTY				0

#define	GSC_READ_PIO_WORK_32_BIT

// WAIT services
#define	DEV_WAIT_GSC_ALL					AI32SSC_WAIT_GSC_ALL
#define	DEV_WAIT_ALT_ALL					AI32SSC_WAIT_ALT_ALL
#define	DEV_WAIT_IO_ALL						AI32SSC_WAIT_IO_ALL



// data types *****************************************************************

struct _dev_io_t
{
	// Initialized by open and IOCTL services (initialize and/or service specific).

	s32					bmdma_threshold;	// Lower DMA transfer size limit.
	s32					io_mode;			// PIO, DMA, DMDMA
	s32					overflow_check;		// Check overflow when reading?
	s32					pio_threshold;		// Use PIO if samples <= this.
	s32					timeout_s;			// I/O timeout in seconds.
	s32					underflow_check;	// Check underflow when reading?

	// Initialized by I/O service at time of use.

	int					abort;
	gsc_dma_ch_t*		dma_channel;		// Use this channel for DMA.
	int					non_blocking;		// Is this non-blocking I/O?

	// Initialized by device specific I/O create code.

	int					bytes_per_sample;	// Sample size in bytes.
	u32					io_reg_offset;		// Offset of device's I/O FIFO.
	VADDR_T				io_reg_vaddr;		// Address of device's I/O FIFO.

	void				(*dev_io_sw_init)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	void				(*dev_io_close)			(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	void				(*dev_io_open)			(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	int					(*dev_io_startup)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	long				(*dev_pio_available)	(GSC_ALT_STRUCT_T* alt, dev_io_t* io, size_t count);
	long				(*dev_bmdma_available)	(GSC_ALT_STRUCT_T* alt, dev_io_t* io, size_t count);
	long				(*dev_dmdma_available)	(GSC_ALT_STRUCT_T* alt, dev_io_t* io, size_t count);
	long				(*dev_pio_xfer)			(GSC_ALT_STRUCT_T* alt, dev_io_t* io, const os_mem_t* mem, size_t count, os_time_tick_t st_end);
	long				(*dev_bmdma_xfer)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io, const os_mem_t* mem, size_t count, os_time_tick_t st_end);
	long				(*dev_dmdma_xfer)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io, const os_mem_t* mem, size_t count, os_time_tick_t st_end);

	struct
	{
		u32				abort;			// ...WAIT_IO_XXX_ABORT
		u32				done;			// ...WAIT_IO_XXX_DONE
		u32				error;			// ...WAIT_IO_XXX_ERROR
		u32				timeout;		// ...WAIT_IO_XXX_TIMEOUT
	} wait;

	// Initialize by GSC commone create code.

	os_mem_t			mem;			// I/O buffer.
	os_sem_t			sem;			// Only one Tx or Rx at a time.
};

struct _dev_data_t
{
	os_pci_t			pci;			// The kernel PCI device descriptor.
	os_data_t			os;				// OS specific data.
	os_spinlock_t		spinlock;		// Control ISR access.
	os_sem_t			sem;			// Control thread access.
	gsc_dev_type_t		board_type;		// Corresponds to basic board type.
	const char*			model;			// Base model number in upper case.
	int					board_index;	// Index of the board being accessed.
	int					users;			// Number of currently active open() requests.
	int					share;			// Were we opened in shared mode?

	gsc_bar_t			bar;			// device register mappings
	gsc_dma_t			dma;			// For DMA based I/O.
	gsc_irq_t			irq;			// For interrut support.
	gsc_wait_node_t*	wait_list;

	struct
	{					// This is for streaming I/O
		dev_io_t		rx;				// Analog Input read
		dev_io_t*		io_streams[DEV_IO_STREAM_QTY];
	} io;

	struct
	{
		VADDR_T			plx_intcsr_32;	// Interrupt Control/Status Register
		VADDR_T			plx_dmaarb_32;	// DMA Arbitration Register
		VADDR_T			plx_dmathr_32;	// DMA Threshold Register

		VADDR_T			gsc_acar_32;	// Active Channel Assignment Register
		VADDR_T			gsc_asiocr_32;	// Auxiliary Sync I/O Control Register
		VADDR_T			gsc_bcfgr_32;	// Board Configuration Register
		VADDR_T			gsc_bctlr_32;	// Board Control Register
		VADDR_T			gsc_bsizr_32;	// Burst Size Register
		VADDR_T			gsc_bufsr_32;	// Buffer Size Register
		VADDR_T			gsc_ibcr_32;	// Input Buffer Control Register
		VADDR_T			gsc_ibdr_32;	// Input Buffer Data Register
		VADDR_T			gsc_icr_32;		// Interrupt Control Register
		VADDR_T			gsc_llcr_32;	// Low Latency Control Register
		VADDR_T			gsc_ll_data_32;	// Low Latency Data Registers Base
		VADDR_T			gsc_ragr_32;	// Rate-A Generator Register
		VADDR_T			gsc_rbgr_32;	// Rate-B Generator Register
		VADDR_T			gsc_smlwr_32;	// Scan Marker Lower Word Register
		VADDR_T			gsc_smuwr_32;	// Scan Marker Upper Word Register
		VADDR_T			gsc_sscr_32;	// Scan & Sync Control Register

		// These are the Time Tag registers
		VADDR_T			gsc_ttacmr_32;	// Time Tag Active Channel Mask Register
		VADDR_T			gsc_ttcr_32;	// Time Tag Cinfiguration Register
		VADDR_T			gsc_ttrdr_32;	// Time Tag Rate Divider Register
	} vaddr;

	struct
	{
		u32				gsc_bcfgr_32;		// Board Configuration Register

		s32				autocal_ms;			// Maximum ms for autocalibrate

		s32				channels_max;		// Maximum channels supported by model.
		s32				channel_qty;		// The number of channels on the device.

		s32				fifo_size;			// Size of FIFO - not the fill level.
		s32				fsamp_max;			// The maximum Fsamp rate per channel.
		s32				fsamp_min;			// The minimum Fsamp rate per channel.
		s32				fw_ver;				// The firmware version.

		s32				initialize_ms;		// Maximum ms for initialize

		s32				low_latency;		// Is Low Latency supported?

		s32				master_clock;		// Master clock frequency

		s32				rate_gen_fgen_max;	// Rate Generator maximum output rate.
		s32				rate_gen_fgen_min;	// Rate Generator minimum output rate.
		s32				rate_gen_nrate_mask;// Mask of valid Nrate bits.
		s32				rate_gen_nrate_max;	// Minimum Nrate value.
		s32				rate_gen_nrate_min;	// Maximum Nrate value.
		s32				rate_gen_qty;		// The number of rate generators on device.
		s32				rbg_sync_output;	// Is Rate-B SYNC Output supported?

		s32				time_tag_support;	// Is Time Tag support present?

	} cache;
};



// prototypes *****************************************************************

void		dev_io_close(dev_data_t* dev);
int			dev_io_create(dev_data_t* dev);
void		dev_io_destroy(dev_data_t* dev);
int			dev_io_open(dev_data_t* dev);
dev_io_t*	dev_io_read_select(dev_data_t* dev, size_t count);
int			dev_irq_create(dev_data_t* dev);
void		dev_irq_destroy(dev_data_t* dev);
int			dev_read_create(dev_data_t* dev, dev_io_t* io);

int			initialize_ioctl(dev_data_t* dev, void* arg);



#endif
