// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AI32SSC/driver/device.c $
// $Rev: 54770 $
// $Date: 2024-07-01 17:22:23 -0500 (Mon, 01 Jul 2024) $

// 16AI32SSC: Device Driver: source file

#include "main.h"



// variables ******************************************************************

const gsc_dev_id_t	dev_id_list[]	=
{
	// model		Vendor	Device	SubVen	SubDev	type
	{ "16AI32SSC",	0x10B5, 0x9056, 0x10B5, 0x3101,	GSC_DEV_TYPE_16AI32SSC	},	// initial criteria
	{ NULL }
};



//*****************************************************************************
static void _channels_compute(dev_data_t* dev)
{
	switch (dev->cache.gsc_bcfgr_32 & 0x30000)
	{
		case 0x00000:	dev->cache.channel_qty	= 32;	break;
		case 0x10000:	dev->cache.channel_qty	= 16;	break;
		case 0x20000:
		case 0x30000:
		default:		dev->cache.channel_qty	= -1;	break;
	}

	dev->cache.channels_max	= 32;
}



//*****************************************************************************
static void _master_clock_compute(dev_data_t* dev)
{
	switch (dev->cache.gsc_bcfgr_32 & 0xC0000)
	{
		case 0x00000:	dev->cache.master_clock	= 50000000L;	break;
		case 0x40000:	dev->cache.master_clock	= 40000000L;	break;
		case 0x80000:	dev->cache.master_clock	= 48000000L;	break;
		case 0xC0000:
		default:		dev->cache.master_clock	= -1;			break;
	}
}



//*****************************************************************************
static void _custom_feature_compute(dev_data_t* dev)
{
	switch (dev->cache.gsc_bcfgr_32 & 0x300000)
	{
		default:
		case 0x300000:
		case 0x000000:

			dev->cache.low_latency		= 0;
			dev->cache.time_tag_support	= 0;
			break;

		case 0x100000:

			dev->cache.low_latency		= 0;
			dev->cache.time_tag_support	= 1;
			break;

		case 0x200000:

			dev->cache.low_latency		= 1;
			dev->cache.time_tag_support	= 0;
			break;
	}
}



//*****************************************************************************
static void _rbg_sync_output_compute(dev_data_t* dev)
{
	if (dev->cache.fw_ver >= 0x101)
		dev->cache.rbg_sync_output	= 1;
	else
		dev->cache.rbg_sync_output	= 0;
}



//*****************************************************************************
static int _type_compute(dev_data_t* dev)
{
	VADDR_T	_0x40_va;
	u32		_0x40_val;
	VADDR_T	_0x80_va;
	u32		_0x80_val;
	u32		bctlr;
	VADDR_T	bctlr_va;
	int		ret;

	_0x40_va	= GSC_VADDR(dev, 0x40);
	_0x80_va	= GSC_VADDR(dev, 0x80);
	bctlr_va	= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_BCTLR));

	_0x40_val	= os_reg_mem_rx_u32(dev, _0x40_va);
	_0x80_val	= os_reg_mem_rx_u32(dev, _0x80_va);
	bctlr		= os_reg_mem_rx_u32(dev, bctlr_va);

	if ((bctlr != _0x40_val) && (bctlr != _0x80_val))
	{
		ret	= 0;		// This IS one of our devices.
	}
	else
	{
		ret	= -ENODEV;	// This is NOT one of our devices.
#if DEV_PCI_ID_SHOW
		printk("This device is not supported by this driver.\n");
#endif
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	dev_device_create
*
*	Purpose:
*
*		Do everything needed to setup and use the given device.
*
*	Arguments:
*
*		dev		The structure to initialize.
*
*	Returned:
*
*		0		All is well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int dev_device_create(dev_data_t* dev)
{
	static const gsc_bar_maps_t	bar_map	=
	{
		{
			// mem	io	rw
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 0: PLX registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 1: PLX registers, I/O mapped
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RW },	// BAR 2: GSC registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 3: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 4: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO }	// BAR 5: unused
		}
	};

	u32	dma;
	int	ret;

	for (;;)	// A convenience loop.
	{
		// Verify some macro contents.
		ret	= gsc_macro_test_base_name(AI32SSC_BASE_NAME);
		if (ret)	break;

		ret	= gsc_macro_test_model();
		if (ret)	break;

		// PCI setup.
		ret	= os_pci_dev_enable(&dev->pci);
		if (ret)	break;

		ret	= os_pci_master_set(&dev->pci);
		if (ret)	break;

		// Control ISR access to the device and data structure.
		ret	= os_spinlock_create(&dev->spinlock);
		if (ret)	break;

		// Control access to the device and data structure.
		ret	= os_sem_create(&dev->sem);
		if (ret)	break;

		// Access the BAR regions.
		ret	= gsc_bar_create(dev, &dev->bar, &bar_map);
		if (ret)	break;

		// Verify that this is a supported device.
		ret	= _type_compute(dev);
		if (ret)	break;

		// Firmware access.
		dev->vaddr.gsc_acar_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_ACAR));
		dev->vaddr.gsc_asiocr_32	= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_ASIOCR));
		dev->vaddr.gsc_bcfgr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_BCFGR));
		dev->vaddr.gsc_bctlr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_BCTLR));
		dev->vaddr.gsc_bsizr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_BSIZR));
		dev->vaddr.gsc_bufsr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_BUFSR));
		dev->vaddr.gsc_ibcr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_IBCR));
		dev->vaddr.gsc_ibdr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_IBDR));
		dev->vaddr.gsc_icr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_ICR));
		dev->vaddr.gsc_ragr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_RAGR));
		dev->vaddr.gsc_rbgr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_RBGR));
		dev->vaddr.gsc_smlwr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_SMLWR));
		dev->vaddr.gsc_smuwr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_SMUWR));
		dev->vaddr.gsc_sscr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_SSCR));
		dev->vaddr.gsc_ttcr_32		= GSC_VADDR(dev, 0x50);	// Time Tag Cofiguration Register
		dev->vaddr.gsc_ttacmr_32	= GSC_VADDR(dev, 0x54);	// Time Tag Active Channel Mask Register
		dev->vaddr.gsc_ttrdr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AI32SSC_GSC_TTRDR));

		dev->vaddr.gsc_ll_data_32	= GSC_VADDR(dev, 0x100);

		// Data cache initialization.
		dev->cache.gsc_bcfgr_32			= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bcfgr_32);

		dev->cache.autocal_ms			= 1000;
		dev->cache.fifo_size			= _256K;
		dev->cache.fsamp_max			= 200000L;
		dev->cache.fsamp_min			= 1L;
		dev->cache.fw_ver				= dev->cache.gsc_bcfgr_32 & 0xFFF;
		dev->cache.initialize_ms		= 3;
		dev->cache.rate_gen_fgen_max	= 200000L;
		dev->cache.rate_gen_fgen_min	= 1;
		dev->cache.rate_gen_nrate_mask	= 0xFFFF;
		dev->cache.rate_gen_nrate_max	= 0xFFFF;
		dev->cache.rate_gen_nrate_min	= 250;
		dev->cache.rate_gen_qty			= 2;

		_channels_compute(dev);
		_master_clock_compute(dev);
		_custom_feature_compute(dev);
		_rbg_sync_output_compute(dev);

		// Initialize additional resources.
		ret	= dev_irq_create(dev);
		if (ret)	break;

		ret	= dev_io_create(dev);
		if (ret)	break;

		dma	= GSC_DMA_SEL_STATIC
			| GSC_DMA_CAP_BMDMA_READ
			| GSC_DMA_CAP_DMDMA_READ;
		ret	= gsc_dma_create(dev, dma, dma);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	dev_device_destroy
*
*	Purpose:
*
*		Do everything needed to release the referenced device.
*
*	Arguments:
*
*		dev		The partial data for the device of interest.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void dev_device_destroy(dev_data_t* dev)
{
	if (dev)
	{
		gsc_dma_destroy(dev);
		dev_io_destroy(dev);
		dev_irq_destroy(dev);
		gsc_bar_destroy(&dev->bar);
		os_sem_destroy(&dev->sem);
		os_spinlock_destroy(&dev->spinlock);
		os_pci_master_clear(&dev->pci);
		os_pci_dev_disable(&dev->pci);
	}
}


