// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/12AISS8AO4/utils/util_reg.c $
// $Rev: 51886 $
// $Date: 2022-10-20 09:53:54 -0500 (Thu, 20 Oct 2022) $

// 12AISS8AO4: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	_GSC_REG(a)			"GSC " #a, AISS8AO4_GSC_##a, 0, 0
#define	WIDTH				24
#define	SIZEOF_ARRAY(a)		(sizeof((a)) / sizeof((a)[0]))



//*****************************************************************************
static int _bctlr_detail(int fd, int supported, u32 value, int width)
{
	static const char*	aim[]	=
	{
		"Differential",
		"Single-Ended",
		"Zero Test",
		"Vref Test",
		"Output Channel 0 Loopback",
		"Output Channel 1 Loopback",
		"Output Channel 2 Loopback",
		"Output Channel 3 Loopback"
	};

	static const char*	range_i[]	=
	{
		"+-0.1 Volts",
		"+-1 Volts",
		"+-10 Volts",
		"RESERVED"
	};

	static const char*	range_o[]	=
	{
		"+-2.5 Volts",
		"+-5 Volts",
		"+-10 Volts",
		"RESERVED"
	};

	static const char*	async[]		= { "No/Asynchronous",	"Yes/Synchronous"	};
	static const char*	disabled[]	= { "Disabled",			"Enabled"			};
	static const char*	fail[]		= { "No (fail)",		"Yes (pass)"		};
	static const char*	idle[]		= { "Idle",				"Active"			};
	static const char*	immed[]		= { "Immediate",		"Simultaneous"		};
	static const char*	no[]		= { "No",				"Yes"				};
	static const char*	target[]	= { "Target",			"Initiator"			};
	static const char*	twos[]		= { "Twos Compliment",	"Offset Binary"		};

	width	+= 10;

	gsc_reg_field_show(width, WIDTH, value, 31, 31, 1, idle,		"Initialize"			);
	gsc_reg_field_show(width, WIDTH, value, 30, 30, 1, NULL,		"Reserved"				);
	gsc_reg_field_show(width, WIDTH, value, 29, 29, 1, fail,		"Auto-Cal Pass"			);
	gsc_reg_field_show(width, WIDTH, value, 28, 28, 1, idle,		"Auto-Cal"				);
	gsc_reg_field_show(width, WIDTH, value, 27, 27, 1, disabled,	"Rate-B Generator"		);
	gsc_reg_field_show(width, WIDTH, value, 26, 26, 1, disabled,	"Rate-A Generator"		);
	gsc_reg_field_show(width, WIDTH, value, 25, 25, 1, twos,		"Data Format"			);
	gsc_reg_field_show(width, WIDTH, value, 24, 24, 1, target,		"Initiator"				);
	gsc_reg_field_show(width, WIDTH, value, 23, 21, 1, NULL,		"Reserved"				);
	gsc_reg_field_show(width, WIDTH, value, 20, 20, 1, idle,		"Output SW Clock"		);
	gsc_reg_field_show(width, WIDTH, value, 19, 19, 1, async,		"Sync Outputs"			);
	gsc_reg_field_show(width, WIDTH, value, 18, 18, 1, immed,		"Simul. Outputs"		);
	gsc_reg_field_show(width, WIDTH, value, 17, 16, 1, range_o,		"Output Range"			);
	gsc_reg_field_show(width, WIDTH, value, 15, 15, 1, no,			"Input Overflow"		);
	gsc_reg_field_show(width, WIDTH, value, 14, 14, 1, idle,		"AI Threshold Flag"		);
	gsc_reg_field_show(width, WIDTH, value, 13, 13, 1, idle,		"Clear Input Buffer"	);
	gsc_reg_field_show(width, WIDTH, value, 12, 12, 1, disabled,	"Enable Input Buffer"	);
	gsc_reg_field_show(width, WIDTH, value, 11, 11, 1, idle,		"Input SW Trigger"		);
	gsc_reg_field_show(width, WIDTH, value, 10, 10, 1, idle,		"Input Burst Busy"		);
	gsc_reg_field_show(width, WIDTH, value,  9,  9, 1, disabled,	"Enable Input Burst"	);
	gsc_reg_field_show(width, WIDTH, value,  8,  8, 1, idle,		"Input SW Clock"		);
	gsc_reg_field_show(width, WIDTH, value,  7,  6, 1, range_i,		"Input Range B"			);
	gsc_reg_field_show(width, WIDTH, value,  5,  4, 1, range_i,		"Input Range A"			);
	gsc_reg_field_show(width, WIDTH, value,  3,  3, 1, NULL,		"Reserved"				);
	gsc_reg_field_show(width, WIDTH, value,  2,  0, 1, aim,			"Input Mode"			);

	return(0);
}



//*****************************************************************************
static int _diopr_detail(int fd, int supported, u32 value, int width)
{
	static const char*	dir[]	= { "Input",	"Output"	};

	width	+= 10;

	gsc_reg_field_show(width, WIDTH, value, 31, 25, 1, NULL,	"Reserved"				);
	gsc_reg_field_show(width, WIDTH, value, 24, 24, 1, dir,		"Byte 1 Dir"			);
	gsc_reg_field_show(width, WIDTH, value, 23, 16, 1, NULL,	"Byte 1"				);
	gsc_reg_field_show(width, WIDTH, value, 15,  9, 1, NULL,	"Reserved"				);
	gsc_reg_field_show(width, WIDTH, value,  8,  8, 1, dir,		"Byte 0 Dir"			);
	gsc_reg_field_show(width, WIDTH, value,  7,  0, 1, NULL,	"Byte 0"				);

	return(0);
}



//*****************************************************************************
static int _aocxr_detail(int fd, int supported, u32 value, int width)
{
	width	+= 10;

	gsc_reg_field_show(width, WIDTH, value, 31, 12, 1, NULL,	"Reserved"	);
	gsc_reg_field_show(width, WIDTH, value, 11,  0, 1, NULL,	"Data"		);

	return(0);
}



//*****************************************************************************
static int _aibr_detail(int fd, int supported, u32 value, int width)
{
	static const char*	no[]	= { "No",	"Yes"	};

	width	+= 10;

	gsc_reg_field_show(width, WIDTH, value, 31, 21, 1, NULL,	"Reserved"		);
	gsc_reg_field_show(width, WIDTH, value, 20, 20, 1, no,		"End of Burst"	);
	gsc_reg_field_show(width, WIDTH, value, 19, 16, 1, NULL,	"Channel Tag"	);
	gsc_reg_field_show(width, WIDTH, value, 15, 12, 1, NULL,	"Padding"		);
	gsc_reg_field_show(width, WIDTH, value, 11,  0, 1, NULL,	"Data"			);

	return(0);
}



//*****************************************************************************
static int _rgxr_detail(int fd, int supported, u32 value, int width)
{
	long	v;

	width	+= 10;

	gsc_reg_field_show(width, WIDTH, value, 31, 16, 1, NULL,	"Reserved"	);
	gsc_reg_field_show(width, WIDTH, value, 15,  0, 0, NULL,	"Ndiv"		);
	v	= value & 0xFFFF;
	printf("0x%lX  (Fref / %ld)\n", v, v);

	return(0);
}



//*****************************************************************************
static int _aicr_detail(int fd, int supported, u32 value, int width)
{
	static const char*	disabled[]	= { "Disabled",	"Enabled"	};

	width	+= 10;

	gsc_reg_field_show(width, WIDTH, value, 31, 31, 1, disabled,	"Input Channel 7 Enable"	);
	gsc_reg_field_show(width, WIDTH, value, 30, 30, 1, disabled,	"Input Channel 6 Enable"	);
	gsc_reg_field_show(width, WIDTH, value, 29, 29, 1, disabled,	"Input Channel 5 Enable"	);
	gsc_reg_field_show(width, WIDTH, value, 28, 28, 1, disabled,	"Input Channel 4 Enable"	);
	gsc_reg_field_show(width, WIDTH, value, 27, 27, 1, disabled,	"Input Channel 3 Enable"	);
	gsc_reg_field_show(width, WIDTH, value, 26, 26, 1, disabled,	"Input Channel 2 Enable"	);
	gsc_reg_field_show(width, WIDTH, value, 25, 25, 1, disabled,	"Input Channel 1 Enable"	);
	gsc_reg_field_show(width, WIDTH, value, 24, 24, 1, disabled,	"Input Channel 0 Enable"	);
	gsc_reg_field_show(width, WIDTH, value, 23, 16, 1, NULL,		"Reserved"					);
	gsc_reg_field_show(width, WIDTH, value, 15,  0, 1, NULL,		"Burst Block Size"			);

	return(0);
}



//*****************************************************************************
static int _ibsr_detail(int fd, int supported, u32 value, int width)
{
	width	+= 10;

	gsc_reg_field_show(width, WIDTH, value, 31, 17, 1, NULL,	"Reserved"			);
	gsc_reg_field_show(width, WIDTH, value, 16,  0, 1, NULL,	"Buffer Fill Level"	);

	return(0);
}



//*****************************************************************************
static int _ibtr_detail(int fd, int supported, u32 value, int width)
{
	static const char*	clear[]	= { "Clear",	"Set"	};

	width	+= 10;

	gsc_reg_field_show(width, WIDTH, value, 31, 17, 1, NULL,	"Reserved"			);
	gsc_reg_field_show(width, WIDTH, value, 16, 16, 1, clear,	"Threshold Flag"	);
	gsc_reg_field_show(width, WIDTH, value, 15,  0, 1, NULL,	"Threshold Level"	);

	return(0);
}



//*****************************************************************************
static int _icsr_detail(int fd, int supported, u32 value, int width)
{
	static const char*	clear[]	= { "Clear",	"Set"		};
	static const char*	idle[]	= { "Idle",		"Active"	};

	width	+= 10;

	gsc_reg_field_show(width, WIDTH, value, 31, 25, 1, NULL,	"Reserved"					);
	gsc_reg_field_show(width, WIDTH, value, 24, 24, 1, idle,	"DIO 0 Low->Hi"				);
	gsc_reg_field_show(width, WIDTH, value, 23, 23, 1, idle,	"AO Clock"					);
	gsc_reg_field_show(width, WIDTH, value, 22, 22, 1, idle,	"AI Clock"					);
	gsc_reg_field_show(width, WIDTH, value, 21, 21, 1, idle,	"AI Burst Done"				);
	gsc_reg_field_show(width, WIDTH, value, 20, 20, 1, idle,	"AI Burst Start"			);
	gsc_reg_field_show(width, WIDTH, value, 19, 19, 1, idle,	"AI Overflow"				);
	gsc_reg_field_show(width, WIDTH, value, 18, 18, 1, idle,	"AI Thresh Flag Low->Hi"	);
	gsc_reg_field_show(width, WIDTH, value, 17, 17, 1, idle,	"AI Thresh Flag Hi->Low"	);
	gsc_reg_field_show(width, WIDTH, value, 16, 16, 1, idle,	"Auto-Cal Complete"			);

	gsc_reg_field_show(width, WIDTH, value, 15,  9, 1, NULL,	"Reserved"					);
	gsc_reg_field_show(width, WIDTH, value,  8,  8, 1, clear,	"DIO 0 Low->Hi"				);
	gsc_reg_field_show(width, WIDTH, value,  7,  7, 1, clear,	"AO Clock"					);
	gsc_reg_field_show(width, WIDTH, value,  6,  6, 1, clear,	"AI Clock"					);
	gsc_reg_field_show(width, WIDTH, value,  5,  5, 1, clear,	"AI Burst Done"				);
	gsc_reg_field_show(width, WIDTH, value,  4,  4, 1, clear,	"AI Burst Start"			);
	gsc_reg_field_show(width, WIDTH, value,  3,  3, 1, clear,	"AI Overflow"				);
	gsc_reg_field_show(width, WIDTH, value,  2,  2, 1, clear,	"AI Thresh Flag Low->Hi"	);
	gsc_reg_field_show(width, WIDTH, value,  1,  1, 1, clear,	"AI Thresh Flag Hi->Low"	);
	gsc_reg_field_show(width, WIDTH, value,  0,  0, 1, clear,	"Auto-Cal Complete"			);

	return(0);
}



//*****************************************************************************
static int _bcfgr_detail(int fd, int supported, u32 value, int width)
{
	static const char*	_64k[]	= { "64K",		"16K"	};
	static const char*	_0[]	= { "0",		"4"		};
	static const char*	_8[]	= { "8",		"4"		};

	long	v;

	width	+= 10;

	gsc_reg_field_show(width, WIDTH, value, 31, 19, 1, NULL,	"Reserved"			);
	gsc_reg_field_show(width, WIDTH, value, 18, 18, 1, _0,		"Output Channels"	);
	gsc_reg_field_show(width, WIDTH, value, 17, 17, 1, _64k,	"Buffer Size"		);
	gsc_reg_field_show(width, WIDTH, value, 16, 16, 1, _8,		"Input Channels"	);
	gsc_reg_field_show(width, WIDTH, value, 15, 12, 1, NULL,	"Reserved"			);
	gsc_reg_field_show(width, WIDTH, value, 11,  0, 0, NULL,	"Firmware Revision"	);
	v	= value & 0xFFF;
	printf("%03lX\n", v);
	return(0);
}



// variables ******************************************************************

static gsc_reg_def_t	_gsc[]	=
{
	{ _GSC_REG(BCTLR),		0,	_bctlr_detail,	"Board Control Register"				},
	{ _GSC_REG(DIOPR),		0,	_diopr_detail,	"Digital I/O Port Register"				},
	{ _GSC_REG(AOC0R),		0,	_aocxr_detail,	"Anlog Output Channel 0 Register"		},
	{ _GSC_REG(AOC1R),		0,	_aocxr_detail,	"Anlog Output Channel 1 Register"		},
	{ _GSC_REG(AOC2R),		0,	_aocxr_detail,	"Anlog Output Channel 2 Register"		},
	{ _GSC_REG(AOC3R),		0,	_aocxr_detail,	"Anlog Output Channel 3 Register"		},
	{ _GSC_REG(AIBR),		0,	_aibr_detail,	"Analog Input Buffer Register"			},
	{ _GSC_REG(RGAR),		0,	_rgxr_detail,	"Rate Generator A Register"				},
	{ _GSC_REG(RGBR),		0,	_rgxr_detail,	"Rate Generator B Register"				},
	{ _GSC_REG(AICR),		0,	_aicr_detail,	"Analog Input Configuration Register"	},
	{ _GSC_REG(IBSR),		0,	_ibsr_detail,	"Input Buffer Size Register"			},
	{ _GSC_REG(IBTR),		0,	_ibtr_detail,	"Input Buffer Threshold Register"		},
	{ _GSC_REG(ICSR),		0,	_icsr_detail,	"Interrupt Control/Status Register"		},
	{ _GSC_REG(BCFGR),		0,	_bcfgr_detail,	"Board Configuration Register"			},
	{ _GSC_REG(AVR),		0,	NULL,			"Auto Cal Values Register"				},
	{ _GSC_REG(ARWR),		0,	NULL,			"Auto Cal Read/Write Register"			},

	{ NULL, 0, 0, 0, 0,		NULL	}
};



//*****************************************************************************
static const gsc_reg_def_t* _find_reg(u32 reg, const gsc_reg_def_t* list)
{
	const gsc_reg_def_t*	def		= NULL;
	int						i;

	for (i = 0; list[i].name; i++)
	{
		if (reg == list[i].reg)
		{
			def	= &list[i];
			break;
		}
	}

	return(def);
}



/******************************************************************************
*
*	Function:	aiss8ao4_reg_get_def_id
*
*	Purpose:
*
*		Retrieve the register definition structure given the register id.
*
*	Arguments:
*
*		reg		The id of the register to access.
*
*	Returned:
*
*		NULL	The register id wasn't found.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* aiss8ao4_reg_get_def_id(int reg)
{
	const gsc_reg_def_t*	def;

	def	= _find_reg(reg, _gsc);
	return(def);
}



/******************************************************************************
*
*	Function:	aiss8ao4_reg_get_def_index
*
*	Purpose:
*
*		Retrieve the register definition structure based on an index.
*
*	Arguments:
*
*		index	The index of the register to access.
*
*	Returned:
*
*		NULL	The index doesn't correspond to a known register.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* aiss8ao4_reg_get_def_index(int index)
{
	const gsc_reg_def_t*	def;

	if (index < 0)
		def	= NULL;
	else if (index >= (SIZEOF_ARRAY(_gsc) - 1))
		def	= NULL;
	else
		def	= &_gsc[index];

	return(def);
}



/******************************************************************************
*
*	Function:	aiss8ao4_reg_get_desc
*
*	Purpose:
*
*		Retrieve the description of the specified register.
*
*	Arguments:
*
*		reg		The register whose description is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* aiss8ao4_reg_get_desc(int reg)
{
	const gsc_reg_def_t*	def;
	const char*				desc;

	def	= _find_reg(reg, _gsc);

	if (def)
		desc	= def->desc;
	else
		desc	= "UNKNOWN";

	return(desc);
}



/******************************************************************************
*
*	Function:	aiss8ao4_reg_get_name
*
*	Purpose:
*
*		Retrieve the name of the specified register.
*
*	Arguments:
*
*		reg		The register whose name is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* aiss8ao4_reg_get_name(int reg)
{
	const gsc_reg_def_t*	def;
	const char*				name;

	def	= _find_reg(reg, _gsc);

	if (def)
		name	= def->name;
	else
		name	= "UNKNOWN";

	return(name);
}



/******************************************************************************
*
*	Function:	aiss8ao4_reg_list
*
*	Purpose:
*
*		List the GSC registers and their values.
*
*	Arguments:
*
*		fd		The handle to access the device.
*
*		detail	List the register details?
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int aiss8ao4_reg_list(int fd, int detail)
{
	int	errs;

	errs	= gsc_reg_list(fd, _gsc, detail, aiss8ao4_reg_read);
	return(errs);
}



