// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/12AISS8AO4/utils/util_query.c $
// $Rev: 50261 $
// $Date: 2022-02-10 19:14:50 -0600 (Thu, 10 Feb 2022) $

// 12AISS8AO4: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	aiss8ao4_query
*
*	Purpose:
*
*		Provide a visual wrapper for the AISS8AO4_IOCTL_QUERY service.
*
*	Arguments:
*
*		fd		Use this handle to access the device.
*
*		index	The index of the device to access. Ignore if < 0.
*				This is for display purposes only.
*
*		verbose	Work verbosely?
*
*		set		This is the query option to access.
*
*		get		The results are reported here. This may be NULL.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int aiss8ao4_query(int fd, int index, int verbose, s32 set, s32* get)
{
	char		buf[128];
	int			errs	= 0;
	const char*	ptr;
	s32			query	= set;
	int			ret;
	s32			tmp;

	if (verbose)
	{
		switch (query)
		{
			default:

				errs++;
				sprintf(buf, "Query Error (#%ld)", (long) query);
				ptr	= buf;
				break;

			case AISS8AO4_QUERY_AUTO_CAL_MS:	ptr	= "Auto-Cal Duration";		break;
			case AISS8AO4_QUERY_CHANNEL_AI_MAX:	ptr	= "Max A/I Channels";		break;
			case AISS8AO4_QUERY_CHANNEL_AI_QTY:	ptr	= "Installed A/I Channels";	break;
			case AISS8AO4_QUERY_CHANNEL_AO_MAX:	ptr	= "Max A/O Channels";		break;
			case AISS8AO4_QUERY_CHANNEL_AO_QTY:	ptr	= "Installed A/O Channels";	break;
			case AISS8AO4_QUERY_COUNT:			ptr	= "Query Options";			break;
			case AISS8AO4_QUERY_DEVICE_TYPE:	ptr	= "Device Type";			break;
			case AISS8AO4_QUERY_FGEN_MAX:		ptr	= "Fgen Max";				break;
			case AISS8AO4_QUERY_FGEN_MIN:		ptr	= "Fgen Min";				break;
			case AISS8AO4_QUERY_FIFO_SIZE_RX:	ptr	= "Rx FIFO Size";			break;
			case AISS8AO4_QUERY_FSAMP_MAX:		ptr	= "Fsamp Max";				break;
			case AISS8AO4_QUERY_FSAMP_MIN:		ptr	= "Fsamp Min";				break;
			case AISS8AO4_QUERY_INIT_MS:		ptr	= "Init. Duration";			break;
			case AISS8AO4_QUERY_MASTER_CLOCK:	ptr	= "Master Clock";			break;
			case AISS8AO4_QUERY_NDIV_MASK:		ptr	= "Nrate Mask";				break;
			case AISS8AO4_QUERY_NDIV_MAX:		ptr	= "Nrate Max";				break;
			case AISS8AO4_QUERY_NDIV_MIN:		ptr	= "Nrate Min";				break;
			case AISS8AO4_QUERY_RATE_GEN_QTY:	ptr	= "Rate Gen Qty";			break;
		}

		gsc_label_index(ptr, index);
	}

	ret		= aiss8ao4_ioctl(fd, AISS8AO4_IOCTL_QUERY, &set);
	errs	+= ret ? 1 : 0;

	if (verbose)
	{
		switch (query)
		{
			default:

				errs++;
				sprintf(buf, "%ld", (long) set);
				break;

			case AISS8AO4_QUERY_CHANNEL_AI_MAX:
			case AISS8AO4_QUERY_CHANNEL_AI_QTY:
			case AISS8AO4_QUERY_CHANNEL_AO_MAX:
			case AISS8AO4_QUERY_CHANNEL_AO_QTY:
			case AISS8AO4_QUERY_COUNT:
			case AISS8AO4_QUERY_NDIV_MAX:
			case AISS8AO4_QUERY_NDIV_MIN:
			case AISS8AO4_QUERY_RATE_GEN_QTY:

				gsc_label_long_comma_buf(set, buf);
				break;

			case AISS8AO4_QUERY_NDIV_MASK:

				sprintf(buf, "0x%lX", (long) set);
				break;

			case AISS8AO4_QUERY_FGEN_MAX:
			case AISS8AO4_QUERY_FGEN_MIN:
			case AISS8AO4_QUERY_MASTER_CLOCK:

				gsc_label_long_comma_buf(set, buf);
				strcat(buf, " Hz");
				break;

			case AISS8AO4_QUERY_AUTO_CAL_MS:
			case AISS8AO4_QUERY_INIT_MS:

				gsc_label_long_comma_buf(set, buf);
				strcat(buf, " ms");
				break;

			case AISS8AO4_QUERY_FSAMP_MAX:
			case AISS8AO4_QUERY_FSAMP_MIN:

				gsc_label_long_comma_buf(set, buf);
				strcat(buf, " S/S");
				break;

			case AISS8AO4_QUERY_FIFO_SIZE_RX:

				ptr	= "";
				tmp	= set;

				if ((tmp) && ((tmp % 1024) == 0))
				{
					ptr	= "K";
					tmp	/= 1024;
				}

				gsc_label_long_comma_buf(tmp, buf);
				strcat(buf, ptr);
				strcat(buf, " Words");
				break;

			case AISS8AO4_QUERY_DEVICE_TYPE:

				if (set == GSC_DEV_TYPE_12AISS8AO4)
				{
					strcpy(buf, "12AISS8AO4");
				}
				else
				{
					errs++;
					sprintf(buf, "INVALID: %ld", (long) set);
				}

				break;
		}

		if (errs)
			printf("FAIL <---  (%s)\n", buf);
		else
			printf("%s\n", buf);
	}

	if (get)
		get[0]	= set;

	return(errs);
}


