// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/12AISS8AO4/utils/util_fsamp_ai.c $
// $Rev: 51886 $
// $Date: 2022-10-20 09:53:54 -0500 (Thu, 20 Oct 2022) $

// 12AISS8AO4: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	ABS(v)					(((v) < 0) ? (-(v)) : (v))
#define	LIMIT_MAX(v,max)		(((v) > (max)) ? (max) : (v))
#define	LIMIT_MIN(v,min)		(((v) < (min)) ? (min) : (v))
#define	LIMIT_RANGE(v,min,max)	(LIMIT_MIN((LIMIT_MAX((v),(max))),(min)))



//*****************************************************************************
int aiss8ao4_fsamp_ai_compute(int fd, int index, int verbose, s32 fsamp, s32* ndiv, double* rate)
{
	double	delta;
	double	delta_best	= FLT_MAX;
	int		errs		= 0;
	double	fsamp_best	= 0;
	double	fsamp_got;
	s32		fsamp_max;	// Queried value.
	s32		fsamp_min;	// Queried value.
	double	fsamp_want;
	s32		master;
	s32		nd;
	s32		nd_best		= 0;
	s32		nd_max;
	s32		nd_min;
	s32		nd_use;
	s32		ndiv_max;	// Queried value.
	s32		ndiv_min;	// Queried value.

	if (verbose)
	{
		gsc_label_index("AI Sample Rate", index);
		gsc_label_long_comma(fsamp);
		printf(" S/S");
	}

	errs	+= aiss8ao4_query(fd, -1, 0, AISS8AO4_QUERY_FSAMP_MAX, &fsamp_max);
	errs	+= aiss8ao4_query(fd, -1, 0, AISS8AO4_QUERY_FSAMP_MIN, &fsamp_min);
	errs	+= aiss8ao4_query(fd, -1, 0, AISS8AO4_QUERY_MASTER_CLOCK, &master);
	errs	+= aiss8ao4_query(fd, -1, 0, AISS8AO4_QUERY_NDIV_MAX, &ndiv_max);
	errs	+= aiss8ao4_query(fd, -1, 0, AISS8AO4_QUERY_NDIV_MIN, &ndiv_min);

	fsamp_want	= (double) LIMIT_RANGE(fsamp, fsamp_min, fsamp_max);
	nd			= master / fsamp_want;
	nd_min		= nd - 1;
	nd_max		= nd + 1;
	nd_min		= (nd_min < ndiv_min) ? ndiv_min : nd_min;
	nd_max		= (nd_max > ndiv_max) ? ndiv_max : nd_max;

	for (nd = nd_min; nd <= nd_max; nd++)
	{
		if (nd < nd_min)
			nd_use	= nd_min;
		else if (nd > nd_max)
			nd_use	= nd_max;
		else
			nd_use	= nd;

		fsamp_got	= ((double) master) / nd_use;
		delta		= fsamp_got - fsamp_want;

		if ((ABS(delta)) < (ABS(delta_best)))
		{
			delta_best	= delta;
			fsamp_best	= fsamp_got;
			nd_best		= nd_use;
		}
	}

	ndiv[0]	= nd_best;

	if (rate)
		rate[0]	= fsamp_best;

	if (verbose)
	{
		printf("%s  (", errs ? " FAIL <---" : "");
		printf("Master ");
		gsc_label_long_comma((long) master);
		printf(", Ndiv ");
		gsc_label_long_comma((long) ndiv[0]);
		printf(", Fsamp %.3f", fsamp_best);
		printf(")\n");
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	aiss8ao4_fsamp_ai_get
*
*	Purpose:
*
*		Determine the input sample rate.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		verbose	Work verbosely>
*
*		get		Store the sample rate here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int aiss8ao4_fsamp_ai_get(int fd, int verbose, s32* get)
{
	s32	ndiv		= -1;
	s32	enable		= -1;
	int	errs		= 0;
	s32	master;
	s32	set			= 0;

	errs	+= aiss8ao4_gen_a_enable(fd, -1, 0, -1, &enable);
	errs	+= aiss8ao4_gen_a_ndiv(fd, -1, 0, -1, &ndiv);
	errs	+= aiss8ao4_query(fd, -1, 0, AISS8AO4_QUERY_MASTER_CLOCK, &master);

	if (errs)
	{
		set	= 0;

		if (verbose)
			printf("FAIL <---  (%d. service errors)\n", __LINE__);
	}
	else if (enable == AISS8AO4_GEN_ENABLE_NO)
	{
		set	= 0;

		if (verbose)
			printf("0 S/S  (Rate Gen. A is disabled.)\n");
	}
	else if (verbose)
	{
		set	= master / ndiv;
		gsc_label_long_comma(set);
		printf(" S/S  (Source = Rate Gen A, Ref = ");
		gsc_label_long_comma(master);
		printf(", Ndiv = ");
		gsc_label_long_comma(ndiv);
		printf(")\n");
	}

	if (get)
		get[0]	= set;

	return(errs);
}



/******************************************************************************
*
*	Function:	aiss8ao4_fsamp_ai_report
*
*	Purpose:
*
*		Determine and report the input sample rate for the specified channel.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		chan	The index of the channel of interest.
*
*		get		Store the sample rate here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int aiss8ao4_fsamp_ai_report(int fd, int index, int verbose, int chan, s32* get)
{
	s32		bit			= 0x1L << chan;
	char	buf[64];
	int		errs		= 0;
	s32		select;
	s32		set			= 0;

	if (verbose)
	{
		sprintf(buf, "AI Channel %d", chan);
		gsc_label_index(buf, index);
	}

	errs	+= aiss8ao4_ai_chan_sel(fd, -1, 0, -1, &select);

	if (select & bit)
	{
		// The channel is enabled.
		errs	+= aiss8ao4_fsamp_ai_get(fd, verbose, &set);
	}
	else
	{
		// The channel is disabled.
		set	= 0;

		if (verbose)
			printf("0 S/S  (Channel Disabled)\n");
	}

	if (get)
		get[0]	= set;

	return(errs);
}



/******************************************************************************
*
*	Function:	aiss8ao4_fsamp_ai_report_all
*
*	Purpose:
*
*		Determine and report the input sample rate for all channels.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int aiss8ao4_fsamp_ai_report_all(int fd, int index, int verbose, s32* fsamp)
{
	s32	chans;
	int	chan;
	int	errs	= 0;
	s32	sps;
	s32	total	= 0;

	if (verbose)
		gsc_label_index("AI Sample Rates", index);

	errs	+= aiss8ao4_query(fd, -1, 0, AISS8AO4_QUERY_CHANNEL_AI_QTY, &chans);

	if (errs)
	{
	}
	else
	{
		if (verbose)
		{
			printf("\n");
			gsc_label_level_inc();
		}

		for (chan = 0; chan < chans; chan++)
		{
			sps		= 0;
			errs	+= aiss8ao4_fsamp_ai_report(fd, -1, verbose, chan, &sps);
			total	+= sps;
		}

		if (verbose)
		{
			gsc_label("Overall AI Rate");
			gsc_label_long_comma(total);
			printf(" S/S\n");

			gsc_label_level_dec();
		}
	}

	if (fsamp)
		fsamp[0]	= total;

	return(errs);
}


