// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/12AISS8AO4/samples/stream/tx.c $
// $Rev: 51902 $
// $Date: 2022-10-20 10:18:56 -0500 (Thu, 20 Oct 2022) $

// 12AISS8AO4: Sample Application: source file

#include "main.h"



// macros *********************************************************************

#define	MAX_RECORDS	10240



// variables ******************************************************************

static	int	_records	= 0;

static struct
{
	long	bm_count;
	long	chan_0_tags;
	long	end_of_burst;
	long	err_values;
	long	non_zero_data;
} _record[MAX_RECORDS];



//*****************************************************************************
static void _wait_til_done(const tx_data_t* tx)
{
	// Wait until the processing thread indicates that it is done.

	for (;;)
	{
		if (tx->done)
			break;

		gsc_time_sleep_ms(1);
	}
}



//*****************************************************************************
static void _wait_til_started(const tx_data_t* tx)
{
	// Wait until the processing thread indicates that it has started.

	for (;;)
	{
		if ((tx->started) || (tx->stop))
			break;

		gsc_time_sleep_ms(1);
	}
}



//*****************************************************************************
static void _wait_for_start(const tx_data_t* tx)
{
	// Wait until the controlling thread tells us to start working.

	for (;;)
	{
		if ((tx->start) || (tx->stop))
			break;

		gsc_time_sleep_ms(1);
	}
}



//*****************************************************************************
static void _write_file_bin(tx_data_t* tx, gsc_buf_man_t* bm)
{
	int	ret;

	ret	= fwrite((char*) bm->buffer + bm->offset, bm->count, 1, tx->file);

	if (ret < 0)
	{
		bm->count	= 0;
		tx->errs	= 1;
		sprintf(tx->err_buf, "Tx binary write failure: %d", (int) ret);
	}
	else
	{
		tx->total_bytes	+= bm->count;
		bm->count		= 0;
		bm->offset		= 0;
	}
}



//*****************************************************************************
static void _write_file_text(tx_data_t* tx, gsc_buf_man_t* bm)
{
	char	buf[32];
	u32		chan;
	int		i;
	int		len;
	char*	prefix;
	u8*		ptr;
	int		remainder;
	int		ret			= 0;
	int		samples;
	u32*	src;
	char*	suffix;
	u32		value;

	samples		= bm->count / 4;
	remainder	= bm->count % 4;
	src			= (void*) ((char*) bm->buffer + bm->offset);

	for (i = 0; i < samples; i++, src++)
	{
		value	= src[0];
		chan	= (value & 0xF0000) >> 16;
		prefix	= chan ? "  " : "";
		suffix	= (chan == (tx->chan_qty - 1)) ? "\r\n" : "";
		sprintf(buf, "%s%08lX%s", prefix, (long) value, suffix);
		len	= strlen(buf);
		ret	= fwrite(buf, len, 1, tx->file);

		if (ret < 0)
		{
			bm->count	= 0;
			tx->errs	= 1;
			sprintf(tx->err_buf, "Tx text write failure: %d", (int) ret);
			break;
		}
	}

	if (remainder)
	{
		buf[0]	= 0;
		ptr		= (u8*) src;

		if (samples)
			strcat(buf, "  ");

		sprintf(buf + strlen(buf), "%02lX", (long) (ptr[0] & 0xFF));

		if (remainder >= 2)
		{
			strcat(buf, "  ");
			sprintf(buf + strlen(buf), "%02lX", (long) (ptr[1] & 0xFF));
		}

		if (remainder >= 3)
		{
			strcat(buf, "  ");
			sprintf(buf + strlen(buf), "%02lX", (long) (ptr[2] & 0xFF));
		}

		strcat(buf, "\r\n");
		len	= strlen(buf);
		ret	= fwrite(buf, len, 1, tx->file);

		if (ret < 0)
		{
			bm->count	= 0;
			tx->errs	= 1;
			sprintf(tx->err_buf, "Tx text write failure: %d", (int) ret);
		}
	}

	if (ret >= 0)
	{
		tx->total_bytes	+= bm->count;
		bm->count		= 0;
		bm->offset		= 0;
	}
}



//*****************************************************************************
static void _validate_data(const gsc_buf_man_t* bm)
{
	long	chan_0_tags		= 0;
	long	end_of_burst	= 0;
	long	err_values		= 0;
	long	l;
	long	non_zero_data	= 0;
	u32*	ptr				= bm->buffer;
	long	samples			= bm->count / 4;

	for (l = 0; l < samples; l++)
	{
		// D0-D15: data (16-bits)
		// Check for non-zero data values.

		if (ptr[l] & 0xFFFF)
			non_zero_data++;

		// D16-D19: channel tag (4-bits)
		// Count the channel 0 tags.

		if ((ptr[l] & 0xF0000) == 0)
			chan_0_tags++;

		// D20: End of Burst (1-bit)
		// Count the EOB flags.

		if (ptr[l] & 0x100000)
			end_of_burst++;

		// D21-D31: reserved (11-bits)
		// Check for invalid values.

		if (ptr[l] & 0xFFE00000)
			err_values++;
	}

	// Record the results.

	_record[_records].bm_count		= bm->count;
	_record[_records].err_values	= err_values;
	_record[_records].chan_0_tags	= chan_0_tags;
	_record[_records].end_of_burst	= end_of_burst;
	_record[_records].non_zero_data	= non_zero_data;

	_records++;
}



//*****************************************************************************
static void _tx_process(tx_data_t* tx)
{
	gsc_buf_man_t	bm;
	int				eof;
	int				errs	= 0;

	memset(&bm, 0, sizeof(gsc_buf_man_t));

	for (;;)
	{
		// Request an data buffer so we can process it.
		errs	= gsc_buf_man_request_data(&bm);

		if (tx->stop)
		{
			gsc_buf_man_release_buffer(&bm);
			break;
		}

		// Perform validation.

		if (errs)
		{
			tx->errs	= 1;
			strcpy(tx->err_buf, "Tx data request: failed");
			gsc_buf_man_release_buffer(&bm);
			break;
		}

		if (bm.buffer == NULL)
		{
			tx->errs	= 1;
			strcpy(tx->err_buf, "Tx data request: NULL buffer");
			gsc_buf_man_release_buffer(&bm);
			break;
		}

		if (bm.size == 0)
		{
			tx->errs	= 1;
			strcpy(tx->err_buf, "Tx data request: zero sized buffer");
			gsc_buf_man_release_buffer(&bm);
			break;
		}

		if ((tx->validate) && (_records < MAX_RECORDS))
			_validate_data(&bm);

		switch (tx->option)
		{
			default:

				tx->errs	= 1;
				bm.count	= 0;
				sprintf(tx->err_buf, "Tx option invalid: %d", (int) tx->option);
				break;

			case TX_OPTION_BIT_BUCKET:

				tx->total_bytes	+= bm.count;
				bm.count		= 0;
				break;

			case TX_OPTION_WRITE_FILE_BIN:

				_write_file_bin(tx, &bm);
				break;

			case TX_OPTION_WRITE_FILE_TEXT:

				_write_file_text(tx, &bm);
				break;
		}

		eof		= bm.eof;
		bm.eof	= 0;
		errs	= gsc_buf_man_release_buffer(&bm);

		if ((tx->errs == 0) && (errs))
		{
			tx->errs	= 1;
			strcpy(tx->err_buf, "Tx buffer release failure");
		}

		if ((eof) || (tx->errs) || (tx->stop))
			break;
	}
}



//*****************************************************************************
static int _tx_thread(void* arg)
{
	tx_data_t*	tx	= arg;

	// Tell the controlling code that this thread has started.
	tx->started	= 1;

	// Wait till we're told to start.
	_wait_for_start(tx);

	// Perform the expected activity.
	tx->total_ms	= gsc_time_delta_ms();
	_tx_process(tx);
	tx->total_ms	= gsc_time_delta_ms() - tx->total_ms;

	// Tell the controlling code that we're done.
	tx->done	= 1;
	return(0);
}



//*****************************************************************************
int tx_start(tx_data_t* tx)
{
	int			errs	= 0;
	const char*	ptr		= NULL;

	gsc_label("Tx Startup");

	switch (tx->option)
	{
		default:

			errs	= 1;
			ptr		= "invalid option";
			break;

		case TX_OPTION_BIT_BUCKET:

			ptr		= "discarding data";
			break;

		case TX_OPTION_WRITE_FILE_BIN:

			strcpy(tx->name, "datast.bin");
			ptr			= "writing to binary file";
			tx->file	= fopen(tx->name, "w+b");

			if (tx->file == NULL)
			{
				errs	= 1;
				printf("FAIL <---  (unable to create file: %s)\n", tx->name);
			}

			break;

		case TX_OPTION_WRITE_FILE_TEXT:

			strcpy(tx->name, "datast.txt");
			ptr			= "writing to text file";
			tx->file	= fopen(tx->name, "w+b");

			if (tx->file == NULL)
			{
				errs	= 1;
				printf("FAIL <---  (unable to create file: %s)\n", tx->name);
			}

			break;
	}

	if (errs == 0)
	{
		errs	= aiss8ao4_query(tx->fd, -1, 0, AISS8AO4_QUERY_CHANNEL_AI_QTY, &tx->chan_qty);

		if (errs)
			ptr	= "Channel count query";
	}

	if (errs == 0)
	{
		errs	= os_thread_create(&tx->thread, "Tx Thread", _tx_thread, tx);

		if (errs)
			ptr	= "creating Tx thread";
	}

	if (errs)
	{
		printf("FAIL <---  (%s)\n", ptr);
	}
	else
	{
		_wait_til_started(tx);
		printf("PASS  (%s)\n", ptr);
	}

	return(errs);
}



//*****************************************************************************
int tx_stop(tx_data_t* tx)
{
	int	errs	= 0;

	// STOP ===============================================
	gsc_label("Tx Stop");
	tx->stop	= 1;

	if (tx->started)
		_wait_til_done(tx);

	if (tx->file)
	{
		fclose(tx->file);
		tx->file	= NULL;
	}

	os_thread_destroy(&tx->thread);
	printf("Done\n");

	gsc_label_level_inc();

	// STATUS =============================================
	gsc_label("Status");

	if (tx->errs == 0)
	{
		errs	= 0;
		printf("PASS\n");
	}
	else
	{
		errs	= 1;
		printf("FAIL <---");

		if (tx->err_buf[0])
			printf(" (%s)", tx->err_buf);

		printf("\n");
	}

	// THROUGHPUT =========================================
	gsc_label("Throughput");
	gsc_label_long_comma(tx->total_bytes / 4);
	printf(" samples, ");
	gsc_label_long_comma(tx->total_ms / 1000);
	printf(".%03ld Secs", (long) tx->total_ms % 1000);

	if (tx->total_ms)
	{
		printf(", ");
		gsc_label_long_comma((tx->total_bytes * 10000 / 4 + 5) / tx->total_ms / 10);
		printf(" S/S");
	}

	printf("\n");

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
void tx_validation_results(void)
{
	long long	bm_count		= 0;
	long long	chan_0_tags		= 0;
	long long	end_of_burst	= 0;
	long long	err_values		= 0;
	int			i;
	long long	non_zero_data	= 0;
	float		ratio;
	long long	samples			= 0;

	printf("Tx Validation Results: %d records\n", _records);

	printf("  Record  Bytes       Samples     Non-Zero    Chan 0 Tag  End Burst   Error Vals  NZD/C0T\n");
	printf("  ======  ==========  ==========  ==========  ==========  ==========  ==========  =======\n");

	for (i = 0; i < _records; i++)
	{
		printf("  %6d", i);
		printf("%12ld", (long) _record[i].bm_count);
		printf("%12ld", (long) _record[i].bm_count / 4);
		printf("%12ld", (long) _record[i].non_zero_data);
		printf("%12ld", (long) _record[i].chan_0_tags);
		printf("%12ld", (long) _record[i].end_of_burst);
		printf("%12ld", (long) _record[i].err_values);

		if (_record[i].chan_0_tags)
		{
			ratio	= (float) _record[i].non_zero_data / _record[i].chan_0_tags;
			printf("  %7.4f", ratio);
		}
		else
		{
			printf("  inf");
		}

		printf("\n");

		bm_count		+= _record[i].bm_count;
		samples			+= _record[i].bm_count / 4;
		err_values		+= _record[i].err_values;
		chan_0_tags		+= _record[i].chan_0_tags;
		end_of_burst	+= _record[i].end_of_burst;
		non_zero_data	+= _record[i].non_zero_data;
	}

	printf("  ======  ==========  ==========  ==========  ==========  ==========  ==========  =======\n");
	printf("  Totals");
	printf("%12lld", bm_count);
	printf("%12lld", samples);
	printf("%12lld", non_zero_data);
	printf("%12lld", chan_0_tags);
	printf("%12lld", end_of_burst);
	printf("%12lld", err_values);

	if (chan_0_tags)
	{
		ratio	= (float) non_zero_data / chan_0_tags;
		printf("  %7.4f", ratio);
	}
	else
	{
		printf("  inf");
	}

	printf("\n");
}


