// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/12AISS8AO4/samples/stream/rx.c $
// $Rev: 51902 $
// $Date: 2022-10-20 10:18:56 -0500 (Thu, 20 Oct 2022) $

// 12AISS8AO4: Sample Application: source file

#include "main.h"



//*****************************************************************************
static void _wait_til_done(const rx_data_t* rx)
{
	// Wait until the processing thread indicates that it is done.

	for (;;)
	{
		if (rx->done)
			break;

		gsc_time_sleep_ms(1);
	}
}



//*****************************************************************************
static void _wait_til_started(const rx_data_t* rx)
{
	// Wait until the processing thread indicates that it has started.

	for (;;)
	{
		if ((rx->started) || (rx->stop))
			break;

		gsc_time_sleep_ms(1);
	}
}



//*****************************************************************************
static void _wait_for_start(const rx_data_t* rx)
{
	// Wait until the controlling thread tells us to start working.

	for (;;)
	{
		if ((rx->start) || (rx->stop))
			break;

		gsc_time_sleep_ms(1);
	}
}



//*****************************************************************************
static int _clear_buffer(rx_data_t* rx)
{
	int	errs;

	for (;;)	// A convenience loop.
	{
		errs	= aiss8ao4_gen_a_enable(rx->fd, -1, 0, AISS8AO4_GEN_ENABLE_NO, NULL);

		if (errs)
		{
			strcpy(rx->err_buf, "Rate A Generator disable: failed");
			break;
		}

		errs	= aiss8ao4_ai_buf_clear(rx->fd, -1, 0);

		if (errs)
		{
			strcpy(rx->err_buf, "Input Buffer clear: failed");
			break;
		}

		errs	= aiss8ao4_ai_buf_overflow(rx->fd, -1, 0, AISS8AO4_BUF_OVERFLOW_CLEAR, NULL);

		if (errs)
		{
			strcpy(rx->err_buf, "Input Buffer overflow clear: failed");
			break;
		}

		errs	= aiss8ao4_gen_a_enable(rx->fd, -1, 0, AISS8AO4_GEN_ENABLE_YES, NULL);

		if (errs)
		{
			strcpy(rx->err_buf, "Rate A Generator enable: failed");
			break;
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static void _rx_process(rx_data_t* rx)
{
	gsc_buf_man_t	bm;
	int				eof;
	int				errs;
	int				ret;

	memset(&bm, 0, sizeof(gsc_buf_man_t));
	errs	= _clear_buffer(rx);

	if (errs)
	{
		bm.eof		= 1;
		rx->errs	= 1;
	}

	for (; errs == 0;)
	{
		// Request an empty buffer so we can fill it.
		errs	= gsc_buf_man_request_empty(&bm);

		if (rx->stop)
		{
			bm.eof	= 1;
			gsc_buf_man_release_buffer(&bm);
			break;
		}

		// Perform validation.

		if (errs)
		{
			bm.eof		= 1;
			rx->errs	= 1;
			strcpy(rx->err_buf, "Rx empty request: failed");
			gsc_buf_man_release_buffer(&bm);
			break;
		}

		if (bm.buffer == NULL)
		{
			bm.eof		= 1;
			rx->errs	= 1;
			strcpy(rx->err_buf, "Rx empty request: NULL buffer");
			gsc_buf_man_release_buffer(&bm);
			break;
		}

		if (bm.size == 0)
		{
			bm.eof		= 1;
			rx->errs	= 1;
			strcpy(rx->err_buf, "Rx empty request: zero sized buffer");
			gsc_buf_man_release_buffer(&bm);
			break;
		}

		switch (rx->option)
		{
			default:

				bm.eof		= 1;
				rx->errs	= 1;
				sprintf(rx->err_buf, "Rx option invalid: %d", (int) rx->option);
				break;

			case RX_OPTION_ZERO_DATA:

				memset(bm.buffer, 0, bm.size);
				bm.offset		= 0;
				bm.count		= bm.size - (bm.size % 4);
				rx->total_bytes	+= bm.count;

				if (rx->total_bytes >= (1000000LL * rx->rx_mb))
					bm.eof		= 1;
				else
					bm.eof		= 0;

				break;

			case RX_OPTION_READ_DEV:

				bm.offset	= 0;
				bm.count	= bm.size - (bm.size % 4);
				ret			= aiss8ao4_read(rx->fd, bm.buffer, bm.count);

				if (ret < 0)
				{
					bm.eof		= 1;
					bm.count	= 0;
					rx->errs	= 1;
					sprintf(rx->err_buf, "Rx read failure: %d", (int) ret);
					break;
				}

				bm.count		= ret;
				rx->total_bytes	+= ret;

				if (rx->total_bytes >= (1000000LL * rx->rx_mb))
				{
					bm.eof	= 1;

					// Disable input to the FIFO.
					errs	= aiss8ao4_gen_a_enable(rx->fd, -1, 0, AISS8AO4_GEN_ENABLE_NO, NULL);

					if (errs)
					{
						rx->errs	= 1;
						strcpy(rx->err_buf, "Rate-A Generator disable failure");
					}
				}
				else
				{
					bm.eof	= 0;
				}

				break;
		}

		eof		= bm.eof;
		errs	= gsc_buf_man_release_buffer(&bm);

		if ((rx->errs == 0) && (errs))
		{
			rx->errs	= 1;
			strcpy(rx->err_buf, "Rx buffer release failure");
		}

		if ((eof) || (rx->errs) || (rx->stop))
			break;
	}
}



//*****************************************************************************
static int _rx_thread(void* arg)
{
	rx_data_t*	rx	= arg;

	// Tell the controlling code that this thread has started.
	rx->started	= 1;

	// Wait till we're told to start.
	_wait_for_start(rx);

	// Perform the expected activity.
	rx->total_ms	= gsc_time_delta_ms();
	_rx_process(rx);
	rx->total_ms	= gsc_time_delta_ms() - rx->total_ms;

	// Tell the controlling code that we're done.
	rx->done	= 1;
	return(0);
}



//*****************************************************************************
int rx_start(rx_data_t* rx)
{
	int			errs	= 0;
	const char*	ptr		= NULL;

	gsc_label("Rx Startup");

	switch (rx->option)
	{
		default:

			errs	= 1;
			ptr		= "invalid option";
			break;

		case RX_OPTION_ZERO_DATA:

			ptr		= "providing NULL data";
			break;

		case RX_OPTION_READ_DEV:

			ptr		= "reading from device";
			break;
	}

	if (errs == 0)
	{
		errs	= os_thread_create(&rx->thread, "Rx Thread", _rx_thread, rx);

		if (errs)
			ptr	= "creating Rx thread";
	}

	if (errs)
	{
		printf("FAIL <---  (%s)\n", ptr);
	}
	else
	{
		_wait_til_started(rx);
		printf("PASS  (%s)\n", ptr);
	}

	return(errs);
}



//*****************************************************************************
int rx_stop(rx_data_t* rx)
{
	int	errs	= 0;

	// STOP ===============================================
	gsc_label("Rx Stop");
	rx->stop	= 1;
	aiss8ao4_rx_io_abort(rx->fd, -1, 0, NULL);

	if (rx->started)
		_wait_til_done(rx);

	os_thread_destroy(&rx->thread);
	printf("Done\n");

	gsc_label_level_inc();

	// STATUS =============================================
	gsc_label("Status");

	if (rx->errs == 0)
	{
		errs	= 0;
		printf("PASS\n");
	}
	else
	{
		errs	= 1;
		printf("FAIL <---");

		if (rx->err_buf[0])
			printf(" (%s)", rx->err_buf);

		printf("\n");
	}

	// THROUGHPUT =========================================
	gsc_label("Throughput");
	gsc_label_long_comma(rx->total_bytes / 4);
	printf(" samples, ");
	gsc_label_long_comma(rx->total_ms / 1000);
	printf(".%03ld Secs", (long) rx->total_ms % 1000);

	if (rx->total_ms)
	{
		printf(", ");
		gsc_label_long_comma((rx->total_bytes * 10000 / 4 + 5) / rx->total_ms / 10);
		printf(" S/S");
	}

	printf("\n");

	gsc_label_level_dec();
	return(errs);
}


