// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/12AISS8AO4/samples/sbtest/wait_event.c $
// $Rev: 51900 $
// $Date: 2022-10-20 10:02:06 -0500 (Thu, 20 Oct 2022) $

// 12AISS8AO4: Sample Application: source file

#include "main.h"



// macros *********************************************************************

#define	_1_SEC		(1000L)
#define	_15_SEC		(15L * 1000L)
#define	_30_SEC		(30L * 1000L)



// variables ******************************************************************
static	int			io_stat;
static	sem_t		sem_enter;
static	sem_t		sem_exit;
static	gsc_wait_t	wait_dst;
static	int			wait_fd;



//*****************************************************************************
static int	_wait_gsc_ai_burst_done(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_IRQ_ENABLE,
			/* arg		*/	AISS8AO4_IRQ_AI_BURST_DONE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_BURST_ENABLE,
			/* arg		*/	AISS8AO4_BURST_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_BURST_SIZE,
			/* arg		*/	0x1,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_GEN_A_ENABLE,
			/* arg		*/	AISS8AO4_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_SW_TRIGGER,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_SLEEP_MS,
			/* cmd		*/	0,
			/* arg		*/	10,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;

	errs	= service_ioctl_set_reg_list(fd, list);
	return(errs);
}



//*****************************************************************************
static int	_wait_gsc_ai_burst_start(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_IRQ_ENABLE,
			/* arg		*/	AISS8AO4_IRQ_AI_BURST_START,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_BURST_ENABLE,
			/* arg		*/	AISS8AO4_BURST_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_SW_TRIGGER,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_SLEEP_MS,
			/* cmd		*/	0,
			/* arg		*/	10,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;

	errs	= service_ioctl_set_reg_list(fd, list);
	return(errs);
}



//*****************************************************************************
static int	_wait_gsc_ai_clock(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_IRQ_ENABLE,
			/* arg		*/	AISS8AO4_IRQ_AI_CLOCK,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_SW_CLOCK,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;

	errs	= service_ioctl_set_reg_list(fd, list);
	return(errs);
}



//*****************************************************************************
static int	_wait_gsc_ai_overflow(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_IRQ_ENABLE,
			/* arg		*/	AISS8AO4_IRQ_AI_OVERFLOW,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_GEN_A_NDIV,
			/* arg		*/	16,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_GEN_A_ENABLE,
			/* arg		*/	AISS8AO4_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_ENABLE,
			/* arg		*/	AISS8AO4_AIO_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{				// Wait to see if we get any data.
			/* service	*/	SERVICE_SLEEP_MS,
			/* cmd		*/	0,
			/* arg		*/	50,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;

	errs	= service_ioctl_set_reg_list(fd, list);
	return(errs);
}



//*****************************************************************************
static int	_wait_gsc_ai_thresh_h2l(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_IRQ_ENABLE,
			/* arg		*/	AISS8AO4_IRQ_AI_THRESH_H2L,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_BUF_THR_LVL,
			/* arg		*/	1,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_GEN_A_ENABLE,
			/* arg		*/	AISS8AO4_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_ENABLE,
			/* arg		*/	AISS8AO4_AIO_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{				// Wait to see if we get any data.
			/* service	*/	SERVICE_SLEEP_MS,
			/* cmd		*/	0,
			/* arg		*/	10,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		// Clear the buffer.

		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_ENABLE,
			/* arg		*/	AISS8AO4_AIO_ENABLE_NO,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_BUF_CLEAR,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;

	errs	= service_ioctl_set_reg_list(fd, list);
	return(errs);
}



//*****************************************************************************
static int	_wait_gsc_ai_thresh_l2h(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_IRQ_ENABLE,
			/* arg		*/	AISS8AO4_IRQ_AI_THRESH_L2H,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_BUF_THR_LVL,
			/* arg		*/	1,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_ENABLE,
			/* arg		*/	AISS8AO4_AIO_ENABLE_NO,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_BUF_CLEAR,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_GEN_A_ENABLE,
			/* arg		*/	AISS8AO4_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_ENABLE,
			/* arg		*/	AISS8AO4_AIO_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{				// Wait to see if we get any data.
			/* service	*/	SERVICE_SLEEP_MS,
			/* cmd		*/	0,
			/* arg		*/	10,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;

	errs	= service_ioctl_set_reg_list(fd, list);
	return(errs);
}



//*****************************************************************************
static int	_wait_gsc_ao_clock(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_IRQ_ENABLE,
			/* arg		*/	AISS8AO4_IRQ_AO_CLOCK,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AO_TIMING,
			/* arg		*/	AISS8AO4_AO_TIMING_SIMUL,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_REG_MOD,
			/* cmd		*/	0,
			/* arg		*/	0,
			/* reg		*/	AISS8AO4_GSC_AOC0R,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_REG_MOD,
			/* cmd		*/	0,
			/* arg		*/	0,
			/* reg		*/	AISS8AO4_GSC_AOC1R,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_REG_MOD,
			/* cmd		*/	0,
			/* arg		*/	0,
			/* reg		*/	AISS8AO4_GSC_AOC2R,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_REG_MOD,
			/* cmd		*/	0,
			/* arg		*/	0,
			/* reg		*/	AISS8AO4_GSC_AOC3R,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AO_SW_CLOCK,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_SLEEP_MS,
			/* cmd		*/	0,
			/* arg		*/	10,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;
	s32	qty;

	errs	= aiss8ao4_query(fd, -1, 0, AISS8AO4_QUERY_CHANNEL_AO_QTY, &qty);

	if (errs)
	{
	}
	else if (qty <= 0)
	{
		errs	= -1;
		printf("SKIPPED  (No output channels)\n");
	}
	else
	{
		errs	= service_ioctl_set_reg_list(fd, list);
	}

	return(errs);
}



//*****************************************************************************
static int	_wait_gsc_auto_cal_done(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_IRQ_ENABLE,
			/* arg		*/	AISS8AO4_IRQ_AUTO_CAL_DONE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AUTO_CALIBRATE,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;

	errs	= service_ioctl_set_reg_list(fd, list);
	return(errs);
}



//*****************************************************************************
static int	_wait_gsc_dio_0_l2h(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_IRQ_ENABLE,
			/* arg		*/	AISS8AO4_IRQ_DIO_0_L2H,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_DIO_DIR_OUT,
			/* arg		*/	0x01,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_DIO_WRITE,
			/* arg		*/	0x00,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_DIO_WRITE,
			/* arg		*/	0x01,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_DIO_WRITE,
			/* arg		*/	0x00,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;

	errs	= service_ioctl_set_reg_list(fd, list);
	return(errs);
}



//*****************************************************************************
static int	_wait_io_rx_done(int fd)
{
	static const service_data_t	list[]	=
	{
		{					// Select PIO mode.
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_RX_IO_MODE,
			/* arg		*/	GSC_IO_MODE_PIO,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_RX_IO_OVERFLOW,
			/* arg		*/	AISS8AO4_IO_OVERFLOW_IGNORE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		// Fill the FIFO.

		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_GEN_A_NDIV,
			/* arg		*/	16,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_GEN_A_ENABLE,
			/* arg		*/	AISS8AO4_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_CHAN_SEL,
			/* arg		*/	0x3,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_ENABLE,
			/* arg		*/	AISS8AO4_AIO_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{				// Wait to see if we get any data.
			/* service	*/	SERVICE_SLEEP,
			/* cmd		*/	0,
			/* arg		*/	1,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	s32	arg;
	u32	buf[1024];
	int	errs;
	int	ret;

	// Verify that the status is an overflow.
	errs	= service_ioctl_set_reg_list(fd, list);

	if (errs == 0)
	{
		arg	= -1;
		ret	= aiss8ao4_ioctl(fd, AISS8AO4_IOCTL_AI_BUF_OVERFLOW, &arg);

		if (ret)
		{
			errs	= 1;
		}
		else if (arg != AISS8AO4_BUF_OVERFLOW_YES)
		{
			errs	= 1;
			printf(	"FAIL <---  (%d. overflow did not occur but was expected)\n",
					__LINE__);
		}
	}

	// Now perform a read to trigger the wait.

	if (errs == 0)
	{
		ret		= aiss8ao4_read(fd, buf, sizeof(buf));
		errs	= (ret == sizeof(buf)) ? 0 : 1;
	}

	return(errs);
}



//*****************************************************************************
static int	_wait_io_rx_error(int fd)
{
	int	errs	= 0;
	int	ret;

	ret	= aiss8ao4_read(fd, NULL, 3);

	if (ret >= 0)
	{
		errs	= 1;
		printf(	"FAIL <---  (%d. expected < 0, got %d)\n", __LINE__, ret);
	}

	return(errs);
}



//*****************************************************************************
static int	_wait_io_rx_timeout(int fd)
{
	static const service_data_t	list[]	=
	{
		{					// Select PIO mode.
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_RX_IO_MODE,
			/* arg		*/	GSC_IO_MODE_PIO,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_RX_IO_OVERFLOW,
			/* arg		*/	AISS8AO4_IO_OVERFLOW_IGNORE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_RX_IO_TIMEOUT,
			/* arg		*/	1,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	u32	buf[1024];
	int	errs;
	int	ret;

	// Verify that the status is an overflow.
	errs	= service_ioctl_set_reg_list(fd, list);

	// Now perform a read to trigger the wait.

	if (errs == 0)
	{
		ret	= aiss8ao4_read(fd, buf, sizeof(buf));

		if (ret != 0)
		{
			errs	= 1;
			printf(	"FAIL <---  (%d. expected 0, got %d)\n",
					__LINE__,
					ret);
		}
	}

	return(errs);
}



//*****************************************************************************
static int _read_thread(void* arg)
{
	u32		buf[1024];
	int		fd	= (int) (long) arg;
	long	ret;

	ret	= aiss8ao4_read(fd, buf, sizeof(buf));

	if (ret == 0)
	{
		io_stat	= 1;
	}
	else
	{
		io_stat	= 2;
		printf("FAIL <---  (%d. aiss8ao4_read returned %ld, expected 0)\n",
				__LINE__,
				(long) ret);
	}

	return(0);
}



//*****************************************************************************
static int	_wait_io_rx_abort(int fd)
{
	static const service_data_t	list[]	=
	{
		{					// Select PIO mode.
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_RX_IO_MODE,
			/* arg		*/	GSC_IO_MODE_PIO,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_RX_IO_TIMEOUT,
			/* arg		*/	10,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int			errs;
	int			i;
	int			ret;
	s32			set;
	os_thread_t	thread;

	// Verify that the status is an overflow.
	errs	= service_ioctl_set_reg_list(fd, list);

	// Now start the read thread so we can do the abort.
	memset(&thread, 0, sizeof(thread));

	if (errs == 0)
	{
		errs+= os_thread_create(	&thread,
									"Rx",
									_read_thread,
									(void*) (long) fd);
	}

	for (i = 0; (errs == 0) && (i < 100); i++)
	{
		set		= 0;
		ret		= aiss8ao4_ioctl(fd, AISS8AO4_IOCTL_RX_IO_ABORT, &set);
		errs	= ret ? 1 : 0;

		if (errs)
			break;

		if (set)
			break;

		os_sleep_ms(100);
	}

	// Wait for the thread to report its status.

	for (i = 0; i < 100; i++)
	{
		if (errs)
			break;

		if (io_stat == 0)
		{
			os_sleep_ms(100);
			continue;
		}

		if (io_stat == 1)
			break;

		errs	= 1;
		break;
	}

	errs	+= os_thread_destroy(&thread);
	return(errs);
}



//*****************************************************************************
static int	_wait_main_dma(int fd)
{
	static const service_data_t	list[]	=
	{
		{					// Select DMA mode.
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_RX_IO_MODE,
			/* arg		*/	GSC_IO_MODE_BMDMA,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_RX_IO_OVERFLOW,
			/* arg		*/	AISS8AO4_IO_OVERFLOW_IGNORE,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		// Fill the FIFO.

		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_GEN_A_NDIV,
			/* arg		*/	45,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_GEN_A_ENABLE,
			/* arg		*/	AISS8AO4_GEN_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_CHAN_SEL,
			/* arg		*/	0x3,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_ENABLE,
			/* arg		*/	AISS8AO4_AIO_ENABLE_YES,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{				// Wait so we get data.
			/* service	*/	SERVICE_SLEEP,
			/* cmd		*/	0,
			/* arg		*/	1,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	u32	buf[1024];
	int	errs;
	int	ret;
	s32	sts		= 0;

	// Verify that the status is an overflow.
	errs	= service_ioctl_set_reg_list(fd, list);

	if (errs == 0)
	{
		sts	= -1;
		ret	= aiss8ao4_ioctl(fd, AISS8AO4_IOCTL_AI_BUF_OVERFLOW, &sts);

		if (ret)
		{
			errs	= 1;
		}
		else if (sts != AISS8AO4_BUF_OVERFLOW_YES)
		{
			errs	= 1;
			printf(	"FAIL <---  (%d. overflow did not occur but was expected)\n",
					__LINE__);
		}
	}

	// Now perform a DMA read to trigger the wait.

	if (errs == 0)
	{
		ret		= aiss8ao4_read(fd, buf, sizeof(buf));
		errs	= (ret == sizeof(buf)) ? 0 : 1;
	}

	return(errs);
}



//*****************************************************************************
static int	_wait_main_gsc(int fd)
{
	static const service_data_t	list[]	=
	{
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_IRQ_ENABLE,
			/* arg		*/	AISS8AO4_IRQ_AI_CLOCK,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},
		{
			/* service	*/	SERVICE_NORMAL,
			/* cmd		*/	AISS8AO4_IOCTL_AI_SW_CLOCK,
			/* arg		*/	0,
			/* reg		*/	-1,
			/* mask		*/	0,
			/* value	*/	0
		},

		{ SERVICE_END_LIST }
	};

	int	errs;

	errs	= service_ioctl_set_reg_list(fd, list);
	return(errs);
}



//*****************************************************************************
static int	_wait_cancel(int fd)
{
	int			errs;
	int			ret;
	gsc_wait_t	wait;

	wait.flags			= 0;
	wait.main			= 0xFFFFFFFF;
	wait.gsc			= 0xFFFFFFFF;
	wait.alt			= 0xFFFFFFFF;
	wait.io				= 0xFFFFFFFF;
	wait.timeout_ms		= 0;
	wait.count			= 0;
	ret		= aiss8ao4_ioctl(wait_fd, AISS8AO4_IOCTL_WAIT_CANCEL, &wait);
	errs	= ret ? 1 : 0;
	return(errs);
}



//*****************************************************************************
static int	_wait_timeout(int fd)
{
	os_sleep_ms(2000);
	return(0);
}



//*****************************************************************************
static int _service_test(void)
{
	// There are no settings bits to test here.
	return(0);
}



//*****************************************************************************
static int _wait_thread(void* arg)
{
	int	ret;

	sem_post(&sem_enter);
	ret	= aiss8ao4_ioctl(wait_fd, AISS8AO4_IOCTL_WAIT_EVENT, &wait_dst);

	if (ret)
		wait_dst.flags	= 0xFFFFFFFF;

	sem_post(&sem_exit);
	return(0);
}


// variables ******************************************************************

static const struct
{
	const char*	name;
	int			(*function)(int fd);
	gsc_wait_t	wait;
} wait_src[]	=
{
	// name						function					flags			main								...			timeout_ms
	{ "DMA0/DMA1",				_wait_main_dma,				{ 0,			GSC_WAIT_MAIN_DMA0 |
																			GSC_WAIT_MAIN_DMA1,					0, 0, 0,	_15_SEC	}	},
	{ "GSC",					_wait_main_gsc,				{ 0,			GSC_WAIT_MAIN_GSC,					0, 0, 0,	_15_SEC	}	},

	// name						function					flags ...		gsc									...			timeout_ms
	{ "Auto-Cal Done",			_wait_gsc_auto_cal_done,	{ 0, 0,			AISS8AO4_WAIT_GSC_AUTO_CAL_DONE,	0, 0,		_15_SEC	}	},
	{ "AI Threshold Hi->Low",	_wait_gsc_ai_thresh_h2l,	{ 0, 0,			AISS8AO4_WAIT_GSC_AI_THRESH_H2L,	0, 0,		_15_SEC	}	},
	{ "AI Threshold Low->Hi",	_wait_gsc_ai_thresh_l2h,	{ 0, 0,			AISS8AO4_WAIT_GSC_AI_THRESH_L2H,	0, 0,		_15_SEC	}	},
	{ "AI Overflow",			_wait_gsc_ai_overflow,		{ 0, 0,			AISS8AO4_WAIT_GSC_AI_OVERFLOW,		0, 0,		_15_SEC	}	},
	{ "AI Burst Start",			_wait_gsc_ai_burst_start,	{ 0, 0,			AISS8AO4_WAIT_GSC_AI_BURST_START,	0, 0,		_15_SEC	}	},
	{ "AI Burst Done",			_wait_gsc_ai_burst_done,	{ 0, 0,			AISS8AO4_WAIT_GSC_AI_BURST_DONE,	0, 0,		_15_SEC	}	},
	{ "AI Clock",				_wait_gsc_ai_clock,			{ 0, 0,			AISS8AO4_WAIT_GSC_AI_CLOCK,			0, 0,		_15_SEC	}	},
	{ "AO Clock",				_wait_gsc_ao_clock,			{ 0, 0,			AISS8AO4_WAIT_GSC_AO_CLOCK,			0, 0,		_15_SEC	}	},
	{ "DIO 0 Low->Hi",			_wait_gsc_dio_0_l2h,		{ 0, 0,			AISS8AO4_WAIT_GSC_DIO_0_L2H,		0, 0,		_15_SEC	}	},

	// name						function					flags ...		io												timeout_ms
	{ "Rx Done",				_wait_io_rx_done,			{ 0, 0, 0, 0,	AISS8AO4_WAIT_IO_RX_DONE,						_15_SEC	}	},
	{ "Rx Error",				_wait_io_rx_error,			{ 0, 0, 0, 0,	AISS8AO4_WAIT_IO_RX_ERROR,						_15_SEC	}	},
	{ "Rx Timeout",				_wait_io_rx_timeout,		{ 0, 0, 0, 0,	AISS8AO4_WAIT_IO_RX_TIMEOUT,					_15_SEC	}	},
	{ "Rx Abort",				_wait_io_rx_abort,			{ 0, 0, 0, 0,	AISS8AO4_WAIT_IO_RX_ABORT,						_15_SEC	}	},

	// function					flags ...																					timeout_ms
	{ "Wait Timeout",			_wait_timeout,				{ 0, 0, 0, 0,	AISS8AO4_WAIT_IO_RX_DONE,						_1_SEC	}	},
	{ "Wait Cancel",			_wait_cancel,				{ 0, 0, 0, 0,	AISS8AO4_WAIT_IO_RX_DONE,						_30_SEC	}	},

	// terminate list
	{ NULL, NULL, { 1 } }
};



//*****************************************************************************
static int _process_thread(int index)
{
	char		buf[64];
	int			errs;
	int			i;
	int			ret;
	int			status;
	os_thread_t	thread;
	gsc_wait_t	wait;

	sem_init(&sem_enter, 0, 0);
	sem_init(&sem_exit, 0, 0);
	memset(&thread, 0, sizeof(thread));
	sprintf(buf, "wait event %d", index);
	wait_dst	= wait_src[index].wait;
	errs		= os_thread_create(&thread, buf, _wait_thread, NULL);

	// Wait for the thread to become waiting.
	sem_wait(&sem_enter);

	if (errs == 0)
	{
		wait	= wait_src[index].wait;

		for (i = 0; i < 100; i++)
		{
			ret	= aiss8ao4_ioctl(wait_fd, AISS8AO4_IOCTL_WAIT_STATUS, &wait);

			if (ret)
			{
				printf(	"FAIL <---  (%d. status request %d: error %d)\n",
						__LINE__,
						index,
						ret);
				errs	= 1;
				break;
			}

			if (wait.count == 1)
				break;

			os_sleep_ms(10);
		}

		if ((errs == 0) && (wait.count != 1))
		{
			printf(	"FAIL <---  (%d. thread didn't wait)\n", __LINE__);
			errs	= 1;
		}
	}

	// Initiate the respective event.
	status	= wait_src[index].function(wait_fd);

	if (status < 0)
	{
	}
	else
	{
		errs	+= status;

		// Verify that the wait ended as expected.

		if (wait_src[index].wait.flags == 0xFFFFFFFF)
		{
		}
		else
		{
			// Wait for the resumed thread to run.
			sem_wait(&sem_exit);

			for (i = 0;; i++)
			{
				if (wait_dst.flags)
					break;

				if (i >= 100)
				{
					printf(	"FAIL <---  (%d. thread failed to resume)\n",
							__LINE__);
					errs++;
					break;
				}

				os_sleep_ms(100);
			}

			if (errs)
			{
			}
			else if (wait_src[index].wait.timeout_ms == _1_SEC)
			{
				if (wait_dst.flags != GSC_WAIT_FLAG_TIMEOUT)
				{
					printf(	"FAIL <---  (%d. flag: expect 0x%lX, got 0x%lX)\n",
							__LINE__,
							(long) GSC_WAIT_FLAG_TIMEOUT,
							(long) wait_dst.flags);
					errs++;
				}
			}
			else if (wait_src[index].wait.timeout_ms == _15_SEC)
			{
				if (wait_dst.flags != GSC_WAIT_FLAG_DONE)
				{
					printf(	"FAIL <---  (%d. flag: expect 0x%lX, got 0x%lX)\n",
							__LINE__,
							(long) GSC_WAIT_FLAG_DONE,
							(long) wait_dst.flags);
					errs++;
				}
			}
			else if (wait_src[index].wait.timeout_ms == _30_SEC)
			{
				if (wait_dst.flags != GSC_WAIT_FLAG_CANCEL)
				{
					printf(	"FAIL <---  (%d. flag: expect 0x%lX, got 0x%lX)\n",
							__LINE__,
							(long) GSC_WAIT_FLAG_TIMEOUT,
							(long) wait_dst.flags);
					errs++;
				}
			}
			else
			{
				printf("FAIL <---  (%d. INTERNAL ERROR)\n", __LINE__);
				errs++;
			}
		}
	}

	wait.flags			= 0;
	wait.main			= 0xFFFFFFFF;
	wait.gsc			= 0xFFFFFFFF;
	wait.alt			= 0xFFFFFFFF;
	wait.io				= 0xFFFFFFFF;
	wait.timeout_ms		= 0;
	wait.count			= 0;
	ret		= aiss8ao4_ioctl(wait_fd, AISS8AO4_IOCTL_WAIT_CANCEL, &wait);
	errs	+= ret ? 1 : 0;
	errs	+= os_thread_destroy(&thread);

	if ((status >= 0) && (errs == 0))
		printf("PASS\n");

	return(errs);
}



//*****************************************************************************
static int _function_test(int fd)
{
	int	errs	= 0;
	int	i;

	printf("\n");
	gsc_label_level_inc();

	// Create threads awaiting each of the source crtiteria
	// and verify that it resumes when the criteria is met.

	for (i = 0; wait_src[i].wait.flags == 0; i++)
	{
		gsc_label(wait_src[i].name);
		errs	+= aiss8ao4_initialize(fd, -1, 0);

		wait_fd	= fd;
		memset(&wait_dst, 0, sizeof(wait_dst));
		errs	+= _process_thread(i);

		errs	+= aiss8ao4_initialize(fd, -1, 0);
	}

	gsc_label_level_dec();
	return(errs);
}



/******************************************************************************
*
*	Function:	wait_event_tests
*
*	Purpose:
*
*		Perform a test of the IOCTL service AISS8AO4_IOCTL_WAIT_EVENT.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int wait_event_tests(int fd)
{
	int	errs	= 0;

	gsc_label("AISS8AO4_IOCTL_WAIT_EVENT");
	errs	+= _service_test();
	errs	+= _function_test(fd);

	return(errs);
}


