// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/12AISS8AO4/samples/id/main.c $
// $Rev: 51893 $
// $Date: 2022-10-20 09:58:42 -0500 (Thu, 20 Oct 2022) $

// 12AISS8AO4: Sample Application: source file

#include "main.h"



//*****************************************************************************
static int _parse_args(int argc, char** argv, args_t* args)
{
	const gsc_arg_item_t	list[]	=
	{
		//	type				var						arg			values	desc
		{	GSC_ARG_S32_FLAG,	&args->continuous,		"-c",		{ -1 },	"Continue testing until an error occurs."		},
		{	GSC_ARG_S32_FLAG,	&args->continuous,		"-C",		{ +1 },	"Continue testing even if errors occur."		},
		{	GSC_ARG_S32_FLAG,	&args->detail,			"-d",		{ 1 },	"Print out detailed register dump."				},
		{	GSC_ARG_S32_MIN,	&args->minute_limit,	"-m",		{ 1 },	"Continue testing for up to # minutes."			},
		{	GSC_ARG_S32_MIN,	&args->test_limit,		"-n",		{ 1 },	"Continue testing for up to # iterations."		},
		{	GSC_ARG_DEV_INDEX,	&args->index,			"index",	{ 1 },	"The zero based index of the device to access."	}
	};

	const gsc_arg_set_t	set	=
	{
		/* name	*/	"id",
		/* desc	*/	"Identify the specified device.",
		/* qty	*/	SIZEOF_ARRAY(list),
		/* list	*/	list
	};

	int	errs;

	memset(args, 0, sizeof(args_t));

	errs	= gsc_args_parse(argc, argv, &set);

	return(errs);
}



//*****************************************************************************
static void _show_access_index(const args_t* args)
{
	gsc_label("Accessing Device");
	printf("%d\n", args->index);
}



//*****************************************************************************
static void _show_time_stamp(void)
{
	const char*	psz;
	struct tm*	stm;
	time_t		tt;

	time(&tt);
	stm	= localtime(&tt);
	psz	= (char*) asctime(stm);
	gsc_label("Time Stamp");
	printf("%s", psz);
}



/******************************************************************************
*
*	Function:	main
*
*	Purpose:
*
*		Control the overall flow of the application.
*
*	Arguments:
*
*		argc			The number of command line arguments.
*
*		argv			The list of command line arguments.
*
*	Returned:
*
*		EXIT_SUCCESS	No errors were encounterred.
*		EXIT_FAILURE	One or more problems were encounterred.
*
******************************************************************************/

int main(int argc, char** argv)
{
	args_t	args;
	int		errs		= 0;
	time_t	exec		= time(NULL);
	long	failures	= 0;
	long	hours;
	long	mins;
	time_t	now;
	long	passes		= 0;
	int		qty			= 0;
	int		ret;
	long	secs;
	time_t	t_limit;
	time_t	test;
	long	tests		= 0;

	for (;;)
	{
		gsc_label_init(27);
		test	= time(NULL);
		errs	= _parse_args(argc, argv, &args);

		if (errs)
			break;

		// Introduction

		t_limit	= exec + (args.minute_limit * 60);
		os_id_host();
		_show_time_stamp();
		_show_access_index(&args);
		errs	+= aiss8ao4_init_util(1);
		errs	+= os_id_driver(aiss8ao4_open, aiss8ao4_read, aiss8ao4_close);
		errs	+= aiss8ao4_count_boards(1, &qty);

		if ((errs) || (qty <= 0))
			break;

		// Open access to device

		errs	= aiss8ao4_open_util(args.index, 0, -1, 1, &args.fd);

		if (errs == 0)
		{
			// Device identification

			errs	+= aiss8ao4_id_device(args.fd, -1, 1);

			// Perform testing

			errs	+= perform_tests(&args);

			// Close device access

			errs	+= aiss8ao4_close_util(args.fd, -1, 1);
		}

		// End of processing.

		now	= time(NULL);
		tests++;

		if (errs)
		{
			failures++;
			printf(	"\nRESULTS: FAIL <---  (%d error%s)",
					errs,
					(errs == 1) ? "" : "s");
		}
		else
		{
			passes++;
			printf("\nRESULTS: PASS");
		}

		secs	= (long) (now - test);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(" (duration %ld:%ld:%02ld)\n", hours, mins, secs);

		secs	= (long) (now - exec);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(	"SUMMARY: tests %ld, pass %ld, fail %ld"
				" (duration %ld:%ld:%02ld)\n\n",
				tests,
				passes,
				failures,
				hours,
				mins,
				secs);

		if (args.continuous == 0)
			break;

		if ((args.continuous < 0) && (errs))
			break;

		if ((args.minute_limit > 0) && (now >= t_limit))
			break;

		if ((args.test_limit > 0) && (tests >= args.test_limit))
			break;
	}

	ret	= ((qty <= 0) || (errs) || (failures)) ? EXIT_FAILURE : EXIT_SUCCESS;
	return(ret);
}



