// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/12AISS8AO4/driver/read.c $
// $Rev: 51904 $
// $Date: 2022-10-20 10:38:02 -0500 (Thu, 20 Oct 2022) $

// 12AISS8AO4: Device Driver: source file

#include "main.h"



//*****************************************************************************
static void _dev_io_sw_init(dev_data_t* dev, dev_io_t* io)
{
	io->io_mode			= AISS8AO4_IO_MODE_DEFAULT;
	io->overflow_check	= AISS8AO4_IO_OVERFLOW_DEFAULT;
	io->timeout_s		= AISS8AO4_IO_TIMEOUT_DEFAULT;
}



//*****************************************************************************
static void _dev_io_close(dev_data_t* dev, dev_io_t* io)
{
	io->dma_channel	= NULL;
}



//*****************************************************************************
static void _dev_io_open(dev_data_t* dev, dev_io_t* io)
{
	_dev_io_sw_init(dev, io);
}



//*****************************************************************************
static int _dev_io_startup(dev_data_t* dev, dev_io_t* io)
{
	u32		bctlr;
	long	ret		= 0;

	if (io->overflow_check)
	{
		bctlr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);

		if (bctlr & D15)
			ret	= -EIO;
	}

	return(ret);
}



//*****************************************************************************
static long _dev_pio_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	u32	ibsr;
	u32	samples;

	ibsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ibsr_32);
	samples	= ibsr & 0xFFFF;
	bytes	= samples * 4;
	return(bytes);
}



//*****************************************************************************
static long _dev_pio_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	long	qty;

	qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	return(qty);
}



//*****************************************************************************
static long _dev_bmdma_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	u32	ibsr;
	u32	samples;

	ibsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ibsr_32);
	samples	= ibsr & 0xFFFF;
	bytes	= samples * 4;
	return(bytes);
}



//*****************************************************************************
static long _dev_bmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	long			qty;
	long			samples	= bytes / 4;
	gsc_dma_setup_t	setup;

	if (samples < io->pio_threshold)
	{
		qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	}
	else
	{
		memset(&setup, 0, sizeof(gsc_dma_setup_t));
		setup.alt		= dev;
		setup.dev		= dev;
		setup.io		= io;
		setup.mem		= mem;
		setup.st_end	= st_end;
		setup.bytes		= bytes;
		setup.ability	= GSC_DMA_CAP_BMDMA_READ;

		setup.mode		= GSC_DMA_MODE_BLOCK_DMA
						| GSC_DMA_MODE_SIZE_32_BITS
						| GSC_DMA_MODE_INPUT_ENABLE
						| GSC_DMA_MODE_BURSTING_LOCAL
						| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
						| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
						| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		setup.dpr		= GSC_DMA_DPR_BOARD_TO_HOST
						| GSC_DMA_DPR_END_OF_CHAIN
						| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty	= gsc_dma_perform(&setup);
	}

	return(qty);
}



//*****************************************************************************
static long _dev_dmdma_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	u32	ibtr;
	u32	samples;

	ibtr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ibtr_32);
	samples	= (ibtr & 0xFFFF) + 1;
	bytes	= samples * 4;
	return(bytes);
}



//*****************************************************************************
static long _dev_dmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	long			qty;
	long			samples	= bytes / 4;
	gsc_dma_setup_t	setup;

	if (samples < io->pio_threshold)
	{
		qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	}
	else
	{
		memset(&setup, 0, sizeof(gsc_dma_setup_t));
		setup.alt		= dev;
		setup.dev		= dev;
		setup.io		= io;
		setup.mem		= mem;
		setup.st_end	= st_end;
		setup.bytes		= bytes;
		setup.ability	= GSC_DMA_CAP_DMDMA_READ;

		setup.mode		= GSC_DMA_MODE_DM_DMA
						| GSC_DMA_MODE_SIZE_32_BITS
						| GSC_DMA_MODE_INPUT_ENABLE
						| GSC_DMA_MODE_BURSTING_LOCAL
						| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
						| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
						| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		setup.dpr		= GSC_DMA_DPR_BOARD_TO_HOST
						| GSC_DMA_DPR_END_OF_CHAIN
						| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty	= gsc_dma_perform(&setup);
	}

	return(qty);
}






/******************************************************************************
*
*	Function:	dev_read_create
*
*	Purpose:
*
*		Perform a one-tine setup of the Read Analog Input streaming I/O structure.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*		io		The I/O structure for this stream.
*
*	Returned:
*
*		None.
*
******************************************************************************/

int dev_read_create(dev_data_t* dev, dev_io_t* io)
{
	int	ret;

	io->bytes_per_sample	= 4;
	io->pio_threshold		= 32;
	io->io_reg_offset		= GSC_REG_OFFSET(AISS8AO4_GSC_AIBR);
	io->io_reg_vaddr		= dev->vaddr.gsc_aibr_32;

	io->dev_io_sw_init		= _dev_io_sw_init;
	io->dev_io_close		= _dev_io_close;
	io->dev_io_open			= _dev_io_open;
	io->dev_io_startup		= _dev_io_startup;
	io->dev_pio_available	= _dev_pio_available;
	io->dev_bmdma_available	= _dev_bmdma_available;
	io->dev_dmdma_available	= _dev_dmdma_available;
	io->dev_pio_xfer		= _dev_pio_xfer;
	io->dev_bmdma_xfer		= _dev_bmdma_xfer;
	io->dev_dmdma_xfer		= _dev_dmdma_xfer;

	io->wait.abort			= AISS8AO4_WAIT_IO_RX_ABORT;
	io->wait.done			= AISS8AO4_WAIT_IO_RX_DONE;
	io->wait.error			= AISS8AO4_WAIT_IO_RX_ERROR;
	io->wait.timeout		= AISS8AO4_WAIT_IO_RX_TIMEOUT;

	ret	= gsc_io_create(dev, io, dev->cache.fifo_size_rx * 4);

	return(ret);
}



