// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/12AISS8AO4/driver/ioctl.c $
// $Rev: 51904 $
// $Date: 2022-10-20 10:38:02 -0500 (Thu, 20 Oct 2022) $

// 12AISS8AO4: Device Driver: source file

#include "main.h"



// macros *********************************************************************

#define	AUTO_CAL_START			0x10000000L		// BCTLR
#define	AUTO_CAL_PASS			0x20000000L		// BCTLR
#define	INIT_START				0x80000000L		// BCTLR

#define	IRQ_AUTO_CAL_DONE		0x00000001L		// ICSR



//*****************************************************************************
static int _query(dev_data_t* dev, s32* arg)
{
	switch (arg[0])
	{
		default:							arg[0]	= AISS8AO4_IOCTL_QUERY_ERROR;		break;
		case AISS8AO4_QUERY_AUTO_CAL_MS:	arg[0]	= dev->cache.auto_cal_ms;			break;
		case AISS8AO4_QUERY_CHANNEL_AI_MAX:	arg[0]	= dev->cache.ai_chans_max;			break;
		case AISS8AO4_QUERY_CHANNEL_AI_QTY:	arg[0]	= dev->cache.ai_chan_qty;			break;
		case AISS8AO4_QUERY_CHANNEL_AO_MAX:	arg[0]	= dev->cache.ao_chans_max;			break;
		case AISS8AO4_QUERY_CHANNEL_AO_QTY:	arg[0]	= dev->cache.ao_chan_qty;			break;
		case AISS8AO4_QUERY_COUNT:			arg[0]	= AISS8AO4_IOCTL_QUERY_LAST;		break;
		case AISS8AO4_QUERY_DEVICE_TYPE:	arg[0]	= dev->board_type;					break;
		case AISS8AO4_QUERY_FGEN_MAX:		arg[0]	= dev->cache.rate_gen_fgen_max;		break;
		case AISS8AO4_QUERY_FGEN_MIN:		arg[0]	= dev->cache.rate_gen_fgen_min;		break;
		case AISS8AO4_QUERY_FIFO_SIZE_RX:	arg[0]	= dev->cache.fifo_size_rx;			break;
		case AISS8AO4_QUERY_FSAMP_MAX:		arg[0]	= dev->cache.fsamp_max;				break;
		case AISS8AO4_QUERY_FSAMP_MIN:		arg[0]	= dev->cache.fsamp_min;				break;
		case AISS8AO4_QUERY_INIT_MS:		arg[0]	= dev->cache.initialize_ms;			break;
		case AISS8AO4_QUERY_MASTER_CLOCK:	arg[0]	= dev->cache.master_clock;			break;
		case AISS8AO4_QUERY_NDIV_MASK:		arg[0]	= dev->cache.rate_gen_ndiv_mask;	break;
		case AISS8AO4_QUERY_NDIV_MAX:		arg[0]	= dev->cache.rate_gen_ndiv_max;		break;
		case AISS8AO4_QUERY_NDIV_MIN:		arg[0]	= dev->cache.rate_gen_ndiv_min;		break;
		case AISS8AO4_QUERY_RATE_GEN_QTY:	arg[0]	= dev->cache.rate_gen_qty;			break;
	}

	return(0);
}



//*****************************************************************************
int initialize_ioctl(dev_data_t* dev, void* arg)
{
	u32		bctlr;
	int		i;
	long	ms		= dev->cache.initialize_ms + 5000;
	int		ret		= 0;
	int		tmp;

	// Initiate initialization.
	os_reg_mem_tx_u32(dev, dev->vaddr.gsc_bctlr_32, INIT_START);

	// There is no Initialize Done interrupt so we wait manually.
	tmp	= gsc_poll_u32(dev, ms, dev->vaddr.gsc_bctlr_32, INIT_START, 0);

	if (tmp)
	{
		ret	= ret ? ret : tmp;
		printf(	"%s: INITIALIZATION DID NOT COMPLETE WITHIN %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually check for completion as the IRQ wait may have ended early.
	bctlr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);

	if (bctlr & INIT_START)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: INITILIZE STILL ACTIVE AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Initialize the software settings.

	for (i = 0; i < DEV_IO_STREAM_QTY; i++)
	{
		if (dev->io.io_streams[i])
		{
			if (dev->io.io_streams[i]->dev_io_sw_init)
			{
				(dev->io.io_streams[i]->dev_io_sw_init)(dev, dev->io.io_streams[i]);
			}
		}
	}

	return(ret);
}



//*****************************************************************************
static int _auto_cal_start(dev_data_t* dev, void* arg)
{
	// Initiate auto-calibration.
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, AUTO_CAL_START, AUTO_CAL_START);
	return(0);
}



//*****************************************************************************
static int _auto_calibrate(dev_data_t* dev, void* arg)
{
	u32			bctlr;
	u32			mask;
	long		ms		= dev->cache.auto_cal_ms + 5000;
	int			ret;
	os_sem_t	sem;
	int			tmp;
	u32			value;
	gsc_wait_t	wt;

	// Safely enable the Auto-Cal Done interrupt.
	mask	= IRQ_AUTO_CAL_DONE;
	value	= IRQ_AUTO_CAL_DONE;
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_icsr_32, value, mask);

	// Wait for the local interrupt.
	os_sem_create(&sem);	// dummy, required for wait operations.
	memset(&wt, 0, sizeof(wt));
	wt.flags		= GSC_WAIT_FLAG_INTERNAL;
	wt.gsc			= AISS8AO4_WAIT_GSC_AUTO_CAL_DONE;
	wt.timeout_ms	= ms;
	ret				= gsc_wait_event(dev, &wt, _auto_cal_start, NULL, &sem);
	os_sem_destroy(&sem);

	if (wt.flags & GSC_WAIT_FLAG_TIMEOUT)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATE IRQ TIMED OUT AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually wait for completion in case something terminates our wait early.
	tmp	= gsc_poll_u32(dev, ms, dev->vaddr.gsc_bctlr_32, AUTO_CAL_START, 0);

	if (tmp)
	{
		ms	*= 2;
		ret	= ret ? ret : tmp;
		printf(	"%s: AUTOCALIBRATION DID NOT COMPLETE WITHIN %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually check for completion as the IRQ wait may have ended early.
	bctlr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);

	if (bctlr & AUTO_CAL_START)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATION STILL ACTIVE AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Final results.
	bctlr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);

	if ((bctlr & AUTO_CAL_PASS) == 0)
	{
		ret	= ret ? ret : -EIO;
		printf(	"%s: AUTOCALIBRATION FAILED (%ld ms).\n",
				dev->model,
				ms);
	}

	if (ret == 0)
	{
		// Wait for settling.
		os_time_sleep_ms(100);
	}

	return(ret);
}



//*****************************************************************************
static int _auto_cal_sts(dev_data_t* dev, s32* arg)
{
	u32	reg;
	s32	value;

	reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);

	if (reg & AUTO_CAL_START)
		value	= AISS8AO4_AUTO_CAL_STS_ACTIVE;
	else if ((reg & AUTO_CAL_PASS) == 0)
		value	= AISS8AO4_AUTO_CAL_STS_FAIL;
	else
		value	= AISS8AO4_AUTO_CAL_STS_PASS;

	arg[0]	= value;
	return(0);
}



//*****************************************************************************
static int _ai_buf_clear(dev_data_t* dev, void* arg)
{
	#undef	MASK
	#undef	SHIFT
	#define	MASK	(0x1L << SHIFT)
	#define	SHIFT	13

	int	ret;

	// Clear the buffer.
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, MASK, MASK);

	// Clear the overflow flag.
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, 0, 0x8000);

	// The bit should clear quickly.
	ret	= gsc_poll_u32(dev, 1, dev->vaddr.gsc_bctlr_32, MASK, 0);

	return(ret);
}



//*****************************************************************************
static int _ai_buf_level(dev_data_t* dev, s32* arg)
{
	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_range_reg(dev, arg, 0, 0x1FFFF, dev->vaddr.gsc_ibsr_32, 16, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_overflow(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_BUF_OVERFLOW_NO,
		AISS8AO4_BUF_OVERFLOW_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 15, 15);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_thr_lvl(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFF, dev->vaddr.gsc_ibtr_32 , 15, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_thr_sts(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_AI_BUF_THR_STS_CLEAR,
		AISS8AO4_AI_BUF_THR_STS_SET,
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 14, 14);
	return(ret);
}



//*****************************************************************************
static int _ai_burst_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_BURST_ENABLE_NO,
		AISS8AO4_BURST_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 9, 9);
	return(ret);
}



//*****************************************************************************
static int _ai_burst_size(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFF, dev->vaddr.gsc_aicr_32 , 15, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_burst_status(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_BURST_STATUS_NOT_READY,
		AISS8AO4_BURST_STATUS_READY,
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 10, 10);
	return(ret);
}



//*****************************************************************************
static int _ai_chan_sel(dev_data_t* dev, void* arg)
{
	int	ret;

	if (dev->cache.ai_chan_qty == 8)
		ret	= gsc_s32_range_reg(dev, arg, 0, 0xFF, dev->vaddr.gsc_aicr_32, 31, 24);
	else
		ret	= gsc_s32_range_reg(dev, arg, 0, 0x0F, dev->vaddr.gsc_aicr_32, 27, 24);

	return(ret);
}



//*****************************************************************************
static int _ai_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_AIO_ENABLE_NO,
		AISS8AO4_AIO_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 12, 12);
	return(ret);
}



//*****************************************************************************
static int _ai_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_AI_MODE_DIFF,
		AISS8AO4_AI_MODE_SINGLE,
		AISS8AO4_AI_MODE_ZERO,
		AISS8AO4_AI_MODE_VREF,
		AISS8AO4_AI_MODE_AO_0,
		AISS8AO4_AI_MODE_AO_1,
		AISS8AO4_AI_MODE_AO_2,
		AISS8AO4_AI_MODE_AO_3,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 2, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_range_a(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_AI_RANGE_100MV,
		AISS8AO4_AI_RANGE_1V,
		AISS8AO4_AI_RANGE_10V,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 5, 4);
	return(ret);
}



//*****************************************************************************
static int _ai_range_b(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_AI_RANGE_100MV,
		AISS8AO4_AI_RANGE_1V,
		AISS8AO4_AI_RANGE_10V,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 7, 6);
	return(ret);
}



//*****************************************************************************
static int _ai_sw_clock(dev_data_t* dev, void* arg)
{
	#undef	MASK
	#undef	SHIFT
	#define	MASK	(0x1L << SHIFT)
	#define	SHIFT	8

	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, MASK, MASK);
	return(0);
}



//*****************************************************************************
static int _ai_sw_trigger(dev_data_t* dev, void* arg)
{
	#undef	MASK
	#undef	SHIFT
	#define	MASK	(0x1L << SHIFT)
	#define	SHIFT	11

	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, MASK, MASK);
	return(0);
}



//*****************************************************************************
static int _ao_clock_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_AO_CLOCK_MODE_ASYNC,
		AISS8AO4_AO_CLOCK_MODE_SYNC,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 19, 19);
	return(ret);
}



//*****************************************************************************
static int _ao_range(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_AO_RANGE_2_5V,
		AISS8AO4_AO_RANGE_5V,
		AISS8AO4_AO_RANGE_10V,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 17, 16);
	return(ret);
}



//*****************************************************************************
static int _ao_sw_clock(dev_data_t* dev, void* arg)
{
	#undef	MASK
	#undef	SHIFT
	#define	MASK	(0x1L << SHIFT)
	#define	SHIFT	20

	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, MASK, MASK);
	return(0);
}



//*****************************************************************************
static int _ao_timing(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_AO_TIMING_IMMED,
		AISS8AO4_AO_TIMING_SIMUL,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 18, 18);
	return(ret);
}



//*****************************************************************************
static int _data_format(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_DATA_FORMAT_2S_COMP,
		AISS8AO4_DATA_FORMAT_OFF_BIN,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 25, 25);
	return(ret);
}



//*****************************************************************************
static int _dio_dir_out(dev_data_t* dev, s32* arg)
{
	#undef	MASK
	#define	MASK	0x01000100

	int	ret		= 0;
	u32	v;

	if (arg[0] == -1)
	{
		v		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_diopr_32);
		arg[0]	= ((v & 0x01000000) ? 0x2 : 0)
				| ((v & 0x00000100) ? 0x1 : 0);
	}
	else if ((arg[0] >= 0) && (arg[0] <= 3))
	{
		v		= ((arg[0] & 0x2) ? 0x01000000 : 0)
				| ((arg[0] & 0x1) ? 0x00000100 : 0);
		os_reg_mem_mx_u32(dev, dev->vaddr.gsc_diopr_32, v, MASK);
		arg[0]	= -1;
		ret		= _dio_dir_out(dev, arg);
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static int _dio_read(dev_data_t* dev, s32* arg)
{
	u32	v;

	v		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_diopr_32);
	arg[0]	= ((v & 0x00FF0000) >> 8) | (v & 0x000000FF);
	return(0);
}



//*****************************************************************************
static int _dio_write(dev_data_t* dev, s32* arg)
{
	#undef	MASK
	#define	MASK	0x00FF00FF

	int	ret;
	u32	v;

	if (arg[0] == -1)
	{
		ret	= _dio_read(dev, arg);
	}
	else if ((arg[0] >= 0) && (arg[0] <= 0xFFFF))
	{
		v	= (arg[0] << 8) | arg[0];
		os_reg_mem_mx_u32(dev, dev->vaddr.gsc_diopr_32, v, MASK);
		ret	= _dio_read(dev, arg);
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static int _gen_a_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_GEN_ENABLE_NO,
		AISS8AO4_GEN_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 26, 26);
	return(ret);
}



//*****************************************************************************
static int _gen_a_ndiv(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(
			dev,
			arg,
			dev->cache.rate_gen_ndiv_min,
			dev->cache.rate_gen_ndiv_max,
			dev->vaddr.gsc_rgar_32,
			23,
			0);
	return(ret);
}



//*****************************************************************************
static int _gen_b_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_GEN_ENABLE_NO,
		AISS8AO4_GEN_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 27, 27);
	return(ret);
}



//*****************************************************************************
static int _gen_b_ndiv(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(
			dev,
			arg,
			dev->cache.rate_gen_ndiv_min,
			dev->cache.rate_gen_ndiv_max,
			dev->vaddr.gsc_rgbr_32,
			23,
			0);
	return(ret);
}



//*****************************************************************************
static int _irq_enable(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0x1FF, dev->vaddr.gsc_icsr_32, 8, 0);
	return(ret);
}



//*****************************************************************************
static int _oper_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_OPER_MODE_TARG,
		AISS8AO4_OPER_MODE_INIT,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 24, 24);
	return(ret);
}



//*****************************************************************************
static int _rx_io_mode(dev_data_t* dev, s32* arg)
{
	static const s32	list[]	=
	{
		GSC_IO_MODE_PIO,
		GSC_IO_MODE_BMDMA,
		GSC_IO_MODE_DMDMA,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.rx.io_mode);
	return(ret);
}



//*****************************************************************************
static int _rx_io_overflow(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS8AO4_IO_OVERFLOW_IGNORE,
		AISS8AO4_IO_OVERFLOW_CHECK,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, options, &dev->io.rx.overflow_check);
	return(ret);
}



//*****************************************************************************
static int _rx_io_timeout(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_var(
			arg,
			AISS8AO4_IO_TIMEOUT_MIN,
			AISS8AO4_IO_TIMEOUT_INFINITE,
			&dev->io.rx.timeout_s);
	return(ret);
}



//*****************************************************************************
static int _rx_io_abort(dev_data_t* dev, s32* arg)
{
	arg[0]	= gsc_read_abort_active_xfer(dev, &dev->io.rx);
	return(0);
}



// variables ******************************************************************

const gsc_ioctl_t	dev_ioctl_list[]	=
{
	{ AISS8AO4_IOCTL_REG_READ,			(void*) gsc_reg_read_ioctl		},
	{ AISS8AO4_IOCTL_REG_WRITE,			(void*) gsc_reg_write_ioctl		},
	{ AISS8AO4_IOCTL_REG_MOD,			(void*) gsc_reg_mod_ioctl		},
	{ AISS8AO4_IOCTL_QUERY,				(void*) _query					},
	{ AISS8AO4_IOCTL_INITIALIZE,		(void*) initialize_ioctl		},
	{ AISS8AO4_IOCTL_AUTO_CALIBRATE,	(void*) _auto_calibrate			},
	{ AISS8AO4_IOCTL_AUTO_CAL_STS,		(void*) _auto_cal_sts			},
	{ AISS8AO4_IOCTL_AI_BUF_CLEAR,		(void*) _ai_buf_clear			},
	{ AISS8AO4_IOCTL_AI_BUF_LEVEL,		(void*) _ai_buf_level			},
	{ AISS8AO4_IOCTL_AI_BUF_OVERFLOW,	(void*) _ai_buf_overflow		},
	{ AISS8AO4_IOCTL_AI_BUF_THR_LVL,	(void*) _ai_buf_thr_lvl			},
	{ AISS8AO4_IOCTL_AI_BUF_THR_STS,	(void*) _ai_buf_thr_sts			},
	{ AISS8AO4_IOCTL_AI_BURST_ENABLE,	(void*) _ai_burst_enable		},
	{ AISS8AO4_IOCTL_AI_BURST_SIZE,		(void*) _ai_burst_size			},
	{ AISS8AO4_IOCTL_AI_BURST_STATUS,	(void*) _ai_burst_status		},
	{ AISS8AO4_IOCTL_AI_CHAN_SEL,		(void*) _ai_chan_sel			},
	{ AISS8AO4_IOCTL_AI_ENABLE,			(void*) _ai_enable				},
	{ AISS8AO4_IOCTL_AI_MODE,			(void*) _ai_mode				},
	{ AISS8AO4_IOCTL_AI_RANGE_A,		(void*) _ai_range_a				},
	{ AISS8AO4_IOCTL_AI_RANGE_B,		(void*) _ai_range_b				},
	{ AISS8AO4_IOCTL_AI_SW_CLOCK,		(void*) _ai_sw_clock			},
	{ AISS8AO4_IOCTL_AI_SW_TRIGGER,		(void*) _ai_sw_trigger			},
	{ AISS8AO4_IOCTL_AO_CLOCK_MODE,		(void*) _ao_clock_mode			},
	{ AISS8AO4_IOCTL_AO_RANGE,			(void*) _ao_range				},
	{ AISS8AO4_IOCTL_AO_SW_CLOCK,		(void*) _ao_sw_clock			},
	{ AISS8AO4_IOCTL_AO_TIMING,			(void*) _ao_timing				},
	{ AISS8AO4_IOCTL_DATA_FORMAT,		(void*) _data_format			},
	{ AISS8AO4_IOCTL_DIO_DIR_OUT,		(void*) _dio_dir_out			},
	{ AISS8AO4_IOCTL_DIO_READ,			(void*) _dio_read				},
	{ AISS8AO4_IOCTL_DIO_WRITE,			(void*) _dio_write				},
	{ AISS8AO4_IOCTL_GEN_A_ENABLE,		(void*) _gen_a_enable			},
	{ AISS8AO4_IOCTL_GEN_A_NDIV,		(void*) _gen_a_ndiv				},
	{ AISS8AO4_IOCTL_GEN_B_ENABLE,		(void*) _gen_b_enable			},
	{ AISS8AO4_IOCTL_GEN_B_NDIV,		(void*) _gen_b_ndiv				},
	{ AISS8AO4_IOCTL_IRQ_ENABLE,		(void*) _irq_enable				},
	{ AISS8AO4_IOCTL_OPER_MODE,			(void*) _oper_mode				},
	{ AISS8AO4_IOCTL_RX_IO_MODE,		(void*) _rx_io_mode				},
	{ AISS8AO4_IOCTL_RX_IO_OVERFLOW,	(void*) _rx_io_overflow			},
	{ AISS8AO4_IOCTL_RX_IO_TIMEOUT,		(void*) _rx_io_timeout			},
	{ AISS8AO4_IOCTL_RX_IO_ABORT,		(void*) _rx_io_abort			},
	{ AISS8AO4_IOCTL_WAIT_EVENT,		(void*) gsc_wait_event_ioctl	},
	{ AISS8AO4_IOCTL_WAIT_CANCEL,		(void*) gsc_wait_cancel_ioctl	},
	{ AISS8AO4_IOCTL_WAIT_STATUS,		(void*) gsc_wait_status_ioctl	},

	{ -1, NULL }
};


