// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/12AISS8AO4/driver/device.c $
// $Rev: 51904 $
// $Date: 2022-10-20 10:38:02 -0500 (Thu, 20 Oct 2022) $

// 12AISS8AO4: Device Driver: source file

#include "main.h"



// macros *********************************************************************

#define	_16K	(16L * 1024)



// variables ******************************************************************

const gsc_dev_id_t	dev_id_list[]	=
{
	// model		Vendor	Device	SubVen	SubDev	type

	{ "12AISS8AO4",	0x10B5, 0x9080, 0x10B5, 0x2880,	GSC_DEV_TYPE_12AISS8AO4	},
	{ NULL }
};



//*****************************************************************************
static void _channels_compute(dev_data_t* dev)
{
	if (dev->cache.gsc_bcfgr_32 & 0x10000)
		dev->cache.ai_chan_qty	= 4;
	else
		dev->cache.ai_chan_qty	= 8;

	if (dev->cache.gsc_bcfgr_32 & 0x40000)
		dev->cache.ao_chan_qty	= 4;
	else
		dev->cache.ao_chan_qty	= 0;

	dev->cache.ai_chans_max	= 8;
	dev->cache.ao_chans_max	= 4;
}



//*****************************************************************************
static void _fifo_size_compute(dev_data_t* dev)
{
	if (dev->cache.gsc_bcfgr_32 & 0x20000)
		dev->cache.fifo_size_rx	= _16K;
	else
		dev->cache.fifo_size_rx	= _64K;
}



/******************************************************************************
*
*	Function:	dev_device_create
*
*	Purpose:
*
*		Do everything needed to setup and use the given device.
*
*	Arguments:
*
*		dev		The structure to initialize.
*
*	Returned:
*
*		0		All went well.
*		< 0		The error status for the problem encounterred.
*
******************************************************************************/

int dev_device_create(dev_data_t* dev)
{
	static const gsc_bar_maps_t	bar_map	=
	{
		{
			// mem	io	rw
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 0: PLX registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 1: PLX registers, I/O mapped
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RW },	// BAR 2: GSC registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 3: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 4: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO }	// BAR 5: unused
		}
	};

	u32	dma0;
	u32	dma1;
	int	ret;

	for (;;)	// A convenience loop.
	{
		// Verify some macro contents.
		ret	= gsc_macro_test_base_name(AISS8AO4_BASE_NAME);
		if (ret)	break;

		ret	= gsc_macro_test_model();
		if (ret)	break;

		// PCI setup.
		ret	= os_pci_dev_enable(&dev->pci);
		if (ret)	break;

		ret	= os_pci_master_set(&dev->pci);
		if (ret)	break;

		// Control ISR access to the device and data structure.
		ret	= os_spinlock_create(&dev->spinlock);
		if (ret)	break;

		// Control access to the device and data structure.
		ret	= os_sem_create(&dev->sem);
		if (ret)	break;

		// Access the BAR regions.
		ret	= gsc_bar_create(dev, &dev->bar, &bar_map);
		if (ret)	break;

		// Firmware access.
		dev->vaddr.gsc_aibr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS8AO4_GSC_AIBR));
		dev->vaddr.gsc_aicr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS8AO4_GSC_AICR));
		dev->vaddr.gsc_bcfgr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS8AO4_GSC_BCFGR));
		dev->vaddr.gsc_bctlr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS8AO4_GSC_BCTLR));
		dev->vaddr.gsc_diopr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS8AO4_GSC_DIOPR));
		dev->vaddr.gsc_ibsr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS8AO4_GSC_IBSR));
		dev->vaddr.gsc_ibtr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS8AO4_GSC_IBTR));
		dev->vaddr.gsc_icsr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS8AO4_GSC_ICSR));
		dev->vaddr.gsc_rgar_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS8AO4_GSC_RGAR));
		dev->vaddr.gsc_rgbr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS8AO4_GSC_RGBR));

		// Data cache initialization.
		dev->cache.gsc_bcfgr_32			= os_reg_mem_rx_u32(NULL, dev->vaddr.gsc_bcfgr_32);

		dev->cache.auto_cal_ms			= 3000;
		dev->cache.initialize_ms		= 3;
		dev->cache.fsamp_max			= 2000000L;
		dev->cache.fsamp_min			= 1;
		dev->cache.master_clock			= 32000000L;
		dev->cache.rate_gen_fgen_max	= 2000000L;
		dev->cache.rate_gen_fgen_min	= 488;
		dev->cache.rate_gen_ndiv_mask	= 0xFFFF;
		dev->cache.rate_gen_ndiv_max	= 0xFFFF;
		dev->cache.rate_gen_ndiv_min	= 16;
		dev->cache.rate_gen_qty			= 2;

		_channels_compute(dev);
		_fifo_size_compute(dev);

		// Initialize additional resources.
		ret	= dev_irq_create(dev);
		if (ret)	break;

		ret	= dev_io_create(dev);
		if (ret)	break;

		dma0	= GSC_DMA_SEL_STATIC
				| GSC_DMA_CAP_BMDMA_READ
				| GSC_DMA_CAP_DMDMA_READ;
		dma1	= 0;

		ret		= gsc_dma_create(dev, dma0, dma1);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	dev_device_destroy
*
*	Purpose:
*
*		Do everything needed to release the referenced device.
*
*	Arguments:
*
*		dev		The partial data for the device of interest.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void dev_device_destroy(dev_data_t* dev)
{
	if (dev)
	{
		gsc_dma_destroy(dev);
		dev_io_destroy(dev);
		dev_irq_destroy(dev);
		gsc_bar_destroy(&dev->bar);
		os_sem_destroy(&dev->sem);
		os_spinlock_destroy(&dev->spinlock);
		os_pci_master_clear(&dev->pci);
		os_pci_dev_disable(&dev->pci);
	}
}


